/*

    Copyright (C) 2009  Matthew William Coan

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

Description:

A simple C compiler that genrates Intel x86 32-bit NASM, TASM32, MASM and gas
assemboly language files.  This source file is written in the standard C++
programming language.

Author: Matthew William Coan
Date: 8/7/2007

*/

#include <vector>
#include <list>
#include <map>
#include <string>

#include <iostream>
#include <fstream>

#include <cstring>
#include <cctype>
#include <cstdlib>
#include <cstdio>

// dump the std namespace into the global scope.
using namespace std;

// DEBUG Flag
//#define DEBUG_CC

// error messages.
char * file_name;
int lineno;

class FileLine {
public:
   const char * file;
   int lineno;   
   size_t offset;

   FileLine()
   { file = 0; lineno = 0; offset = 0U; }

   FileLine(const FileLine & cp)
   { file = cp.file; lineno = cp.lineno; offset = cp.offset; }

   virtual ~FileLine() { }

   FileLine & operator=(const FileLine & cp)
   { file = cp.file; lineno = cp.lineno; offset = cp.offset; return *this; }
};

typedef vector< FileLine > file_line_vector_type;

void 
_error(const string msg, const FileLine & fn)
{
   cerr << "error: " << fn.file << ": " << fn.lineno 
        << ": " << msg << endl;
   exit(1);
}
void 
_warning(const string msg, const FileLine & fn)
{
   cerr << "warning: " << fn.file << ": " << fn.lineno 
        << ": " << msg << endl;
}

#define error(msg) _error(msg,get_file_line())
#define warning(msg) _warning(msg,get_file_line())

//#define error(msg) c_error(msg,__FILE__,__LINE__)

// constants
#define DATA_WORD_SIZE 4
#define DATA_WORD_SIZE_STR "4"

// token type.
enum token_type { 
   STRING_VALUE, 
   CHAR_VALUE,
   SHORT_VALUE,
   INT_VALUE, 
   LONG_VALUE, 
   FLOAT_VALUE, 
   DOUBLE_VALUE, 
   UCHAR_VALUE,
   USHORT_VALUE,
   UINT_VALUE,
   ULONG_VALUE,
   ID,
   KEY_WORD,
   OPERATOR 
};

// a token
class token {
public:
   string value;
   token_type type; 

   token()
   { 
   }

   token(const token & t) 
   { 
      value = t.value;
      type = t.type;
   }

   ~token() 
   { 
   }

   token & operator=(const token & t) 
   {
      value = t.value;
      type = t.type;

      return *this;
   }
};

// token table (filled by lex()).
typedef vector< token > token_vector_type;

// symbol table.

// an instruction
class instruction {
public:
   string label;
   string op;
   string arg1;
   string arg2;
   string arg3;

   instruction() { }

   instruction(const instruction & c) 
   {
      label = c.label;
      op = c.op;
      arg1 = c.arg1;
      arg2 = c.arg2;
      arg3 = c.arg3;
   }

   instruction & operator=(const instruction & i) 
   {
      label = i.label;
      op = i.op;
      arg1 = i.arg1;
      arg2 = i.arg2;
      arg3 = i.arg3;

      return *this;
   } 
};

// instruction table type (filled by parse()).
typedef vector< instruction > instruction_vector_type;

// param
typedef pair< string, string > param_type;

// param list.
typedef vector< param_type > param_vector_type;

// array dimentions
typedef vector< size_t > int_vector_type;

class c_compiler;

// variable representation.
class variable_rep {
public:
   string value;
   string type;          // variable type.
   string label;         // id string.
   string default_value; // default value of variable.
   int pointer_count;    // pointer count.
   size_t address;
   int_vector_type array;
   size_t size;
   c_compiler * p_c_compiler;
   bool global;

   variable_rep(c_compiler * p_c = 0) { 
      pointer_count = 0;
      address = 0U;
      size = 0U;
      p_c_compiler = p_c;
      global = false;
   }

   variable_rep(const variable_rep & vr) {
      value = vr.value;
      type = vr.type;
      label = vr.label;
      default_value = vr.default_value;
      pointer_count = vr.pointer_count;
      address = vr.address;
      size = vr.size;
      array = vr.array;
      p_c_compiler = vr.p_c_compiler;
      global = vr.global;
   }

   virtual ~variable_rep() {

   }

   size_t get_struct_size(const string & struct_name);

   size_t get_size() {
      size_t sz = 0U;
      if(type.find("*") != string::npos || pointer_count > 0) {
         sz += DATA_WORD_SIZE;
      }
      else if(type.find("long") != string::npos
         || type.find("double") != string::npos) {
         if(array.size()) {
            for(size_t i = 0; i < array.size(); i++) {
               sz += 8 * array[i];
            }
         }
         else {
            sz += 8;
         }
      }
      else if(type.find("int") != string::npos 
              || type.find("float") != string::npos 
              || type.find("*") != string::npos) {
         if(array.size()) {
            for(size_t i = 0; i < array.size(); i++) {
               sz += 4 * array[i];
            }
         }
         else {
            sz += 4;
         }
      }
      else if(type.find("short") != string::npos) {
         sz += 2;
         //sz += 4;
      }
      else if(type.find("char") != string::npos
              && type.find("*") != string::npos) {
         sz += 4;
      }
      else if(type.find("char") != string::npos) {
         sz += 1;
      }
      else if(type.find("struct ") != string::npos) {
//***DEBUG
         sz += get_struct_size(type);
         //sz += 4;
//***DEBUG
      }
      else {
         sz += 4;
      }

      //for(size_t i = 0; i < array.size(); i++) {
      //   sz *= array[i];
      //}

      size = sz;

      return sz;
   }

   string get_address() {
      const size_t MAX_BUFFER = 1024;
      char buffer[MAX_BUFFER];
      memset(buffer, 0, MAX_BUFFER);
      sprintf(buffer, "%d", address + get_size());
      string ret = string(buffer);
      return ret;
   }

   string get_pointer_count() {
      const size_t MAX_BUFFER = 1024;
      char buffer[MAX_BUFFER];
      memset(buffer, 0, MAX_BUFFER);
      sprintf(buffer, "%d", pointer_count);
      string str_pc = string(buffer);
      return str_pc;
   }

   variable_rep & operator=(const variable_rep & vr) {
      label = vr.label;

      default_value = vr.default_value;

      type = vr.type;

      pointer_count = vr.pointer_count;

      address = vr.address;

      array = vr.array;

      size = vr.size;

      p_c_compiler = vr.p_c_compiler;

      global = vr.global;

      value = vr.value;

      return *this;
   }
};

// variable array.
typedef vector< variable_rep > variable_vector_type;

// function representation.
class function_rep {
public:
   string name;                                // function name.
   instruction_vector_type instruction_vector; // the instructions for this f.
   param_vector_type param_vector;             // the function param vector.
   string return_type;                         // function return type.
   variable_vector_type variable_vector;       // vector of local variables.

   function_rep() { }

   function_rep(const function_rep & arg) {
      name = arg.name;
   
      instruction_vector = arg.instruction_vector;

      param_vector = arg.param_vector;

      return_type = arg.return_type;

      variable_vector = arg.variable_vector;
   }

   function_rep & operator=(const function_rep & arg) {
      name = arg.name;

      instruction_vector = arg.instruction_vector;

      param_vector = arg.param_vector;

      return_type = arg.return_type;

      variable_vector = arg.variable_vector;

      return *this;
   }

   bool has_param(const string & param) {
      bool ret = false;

      for(size_t i = 0; i < param_vector.size(); i++) {
cout << "(\"" << param_vector[i].first << "\",\"" << param << "\")" << endl << flush;
         if(param.find(param_vector[i].first) != string::npos) {
            ret = true;
            break;
         }
      }

cout << "ret == " << ret << endl << flush;
//cin.get();cin.get();

      return ret;
   }
};

// struct representation.
class struct_rep {
public:
   string name;
   variable_vector_type var_vec;
   c_compiler * p_c_compiler;

   struct_rep(c_compiler * p_c) { p_c_compiler = p_c; }
   
   struct_rep(const struct_rep & arg) {
      name = arg.name;
      var_vec = arg.var_vec;
      p_c_compiler = arg.p_c_compiler;
   }

   struct_rep & operator=(const struct_rep & arg) {
      name = arg.name;
      var_vec = arg.var_vec; 
      p_c_compiler = arg.p_c_compiler;
      return *this;
   }

   size_t get_size() { 
      size_t sz = 0;

      for(size_t i = 0; i < var_vec.size(); i++) {
         sz += var_vec[i].get_size();
      }

      return sz;
   }

   string get_type(const string & id) {
      string ret;

      for(size_t i = 0; i < var_vec.size(); i++) {
         if(var_vec[i].label == id) {
            ret = var_vec[i].type;
            break;
         }
      }

      return ret;
   }

   variable_rep * get_variable(const string & id) {
      variable_rep * ret = 0;
      for(size_t i = 0; i < var_vec.size(); i++) {
         if(var_vec[i].label == id) {
            ret = (&(var_vec[i]));
            break;
         }
      }
      return ret;
   }
};

typedef vector< pair< string, int > > e_vector_type;

// enum representation.
class enum_rep {
public:
   string name;
   e_vector_type vec;

   int find(const string & enum_name) {
      int ret = -1;

      for(size_t i = 0; i < vec.size(); i++) {
         if(vec[i].first == enum_name) {
            ret = i;
            break;
         }
      }

      return ret;
   }

   void add_member(const string & type, const int value) {
      vec.push_back(pair< string, int >(name, value));
   }
};

typedef vector< pair< string, string > > u_vector_type;

class union_rep {
public:
   string name;
   u_vector_type vec;

   int find(const string & u_name) {
      int ret = -1;
      for(size_t i = 0; i < vec.size(); i++) {
         if(vec[i].second == u_name) {
            ret = i;
            break; 
         }
      }
      return ret;
   }

   void add_member(const string & type, const string & name) {
      vec.push_back(pair< string, string >(type, name));
   }
};

// typedef representation.
class typedef_rep {
public:
   string name; // the new type name.
   string type; // it's type.

   typedef_rep() { }

   typedef_rep(const string & t, const string & n) {
      name = n;
      type = t;
   }

   typedef_rep(const typedef_rep & arg) {
      name = arg.name;

      type = arg.type;
   }

   typedef_rep & operator=(const typedef_rep & arg) {
      name = arg.name;
  
      type = arg.type;

      return *this;
   }
};

// function array.
typedef vector< function_rep > function_vector_type;

// struct array.
typedef vector< struct_rep > struct_vector_type;

// typedef array.
typedef vector< typedef_rep > typedef_vector_type;

// union array
typedef vector< union_rep > union_vector_type;

// eunum array
typedef vector< enum_rep > enum_vector_type;

// symbol table type.
typedef map< string , variable_rep >  symbol_table_type;

// string map
typedef map< string, string > string_map_type;

//
class Expression {
public:
   instruction_vector_type code;
   string expr;
   string type;

   Expression() 
   { 
   }

   Expression(const Expression & ex) 
   { 
      code = ex.code;
      expr = ex.expr;
      type = ex.type;
   }

   Expression(instruction_vector_type & c, string & ex, string & t)
   { 
      code = c; 
      expr = ex;
      type = t;
   }

   Expression & operator=(const Expression & ex)
   {
      code = ex.code;
      expr = ex.expr;
      type = ex.type;
      return *this;
   }
};

typedef vector< Expression > expr_vector_type;

typedef vector< pair< string, size_t > > temp_vector_type;

string to_string(int i);

// C compiler class
class c_compiler {
   enum_vector_type _enum_vector;
   union_vector_type _union_vector;
   variable_vector_type _variable_vector;
   function_vector_type _function_vector;       // vector of functions.
   struct_vector_type _struct_vector;           // vector of structs.
   typedef_vector_type _typedef_vector;         // vector of typedefs.
   ifstream _in;                                // input file.
   ofstream _out;                               // output file.
   token_vector_type _token_vector;             // the file in tokens.
   size_t _offset;                              // the offset in the token. 
   function_rep _current_function;              // the current function.
                                                // vector.
   instruction_vector_type _instruction_vector; // the generated code.
   int _current_temp;                           // current temp.
   int _current_label;                          // current label.
   int _current_const;                          // current constant.
   symbol_table_type _symbol_table;             // the symbol table.
   size_t _count_peren;
   string _loop_condition;
   string _break_location;
   string_map_type _type_map;
   bool _first_compound_statement;
   bool _no_assign;
   string _struct_name;
   struct_rep _current_struct;
   bool _got_return;
   bool _is_global;
   size_t _size;
   file_line_vector_type _file_line_vector;
   bool _in_call_list;
   instruction_vector_type _init_code;
   bool _address_of;
   expr_vector_type _expr_call_list;
   temp_vector_type _temp_vec;
   temp_vector_type _reuse_temp_vec;

protected:
   bool is_array(const string & name);

   string to_asm_string(const string & str);

   string to_hex2(const int i);

   function_rep * get_function(const string & name);

   string to_param(const string_map_type & param_map, const string & param);

   param_vector_type get_param();

   bool is_function(const string & f);

   bool is_param(const string & param);

   bool is_global(const string & id);

   bool is_struct(const string & t, const string & id);

   bool is_struct(const string & t);

   string to_struct(const string & st);

   bool is_assign(const string & op) {
      bool ret = false;
      if(op == "=" || op == "+=" || op == "-=" || op == "/=" || op == "*=" || op == "%="
         || op == "&=" || op == "|=" || op == "^=" || op == "<<=" || op == ">>=") {
         ret = true;
      }
      return ret;
   }

   string to_memory_type(const string & m) {
      string ret;

      if(is_type(m, "int") || is_type(m, "float") || is_type(m, "*")) {
         ret = "dword";
      }
      else if(is_type(m, "long") || is_type(m, "double")) {
         ret = "qword";
      }
      else if(is_type(m, "short")) {
         ret = "word";
         //ret = "dword";
      }
      else if(is_type(m, "char")) {
         ret = "byte";
      }
      else {
         ret = "dword";
      }

      return ret;
   }

   void assign_code(const string & op, 
                    const string & type, 
                    const string & arg1, 
                    const string & arg2,
                    const bool lval = false) {
      instruction code;
      struct_rep * p_struct;

cout << "assign_code(\"" << op << "\",\"" << type << "\",\"" << arg1 << "\",\"" << arg2 << "\"," << lval << ")" << endl << flush;
//cin.get(),cin.get();

      if(op == "=") {
         if(lval) {
            code = instruction();
            code.op = "mov_dword";
            code.arg1 = "eax";
            code.arg2 = arg2;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "mov3";
            code.arg1 = arg1;
            code.arg2 = "eax";
            _instruction_vector.push_back(code);
         }
         else if((p_struct = get_struct(to_struct(type))) != 0 
                 && _type_map[arg1].find("*") == string::npos 
                 && _type_map[arg2].find("*") != string::npos) {
            code = instruction();
            code.op = "lea";
            code.arg1 = "edx";
            code.arg2 = arg1;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "lea";
            code.arg1 = "ebx";
            code.arg2 = arg2;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "mov_struct3";
            code.arg1 = "edx";
            code.arg2 = "ebx";
            code.arg3 = to_string(p_struct->get_size());
            _instruction_vector.push_back(code);
         }
         else if((p_struct = get_struct(to_struct(type))) != 0 && type.find("*") == string::npos) {
            code = instruction();
            code.op = "lea";
            code.arg1 = "edx";
            code.arg2 = arg1;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "lea";
            code.arg1 = "ebx";
            code.arg2 = arg2;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "mov_struct3";
            code.arg1 = "edx";
            code.arg2 = "ebx";
            code.arg3 = to_string(p_struct->get_size());
            _instruction_vector.push_back(code);
         }
         else {
            if(is_const(arg2) || is_register(arg2)) {
               code = instruction();
               code.op = "mov_" + to_memory_type(type);
               code.arg1 = arg1;
               code.arg2 = arg2;
               _instruction_vector.push_back(code);
            }
            else if(is_register(arg1)) {
/*
               if(is_param(arg2)) {
                  code = instruction();
                  code.op = "lea";
                  code.arg1 = "eax";
                  code.arg2 = arg2;
                  _instruction_vector.push_back(code);

                  code = instruction();
                  code.op = "add";
                  code.arg1 = "eax";
                  code.arg2 = DATA_WORD_SIZE_STR;
                  _instruction_vector.push_back(code);

                  code = instruction();

                  code = instruction();
                  code.op = "mov2";
                  code.arg1 = "ebx";
                  code.arg2 = "eax";
                  _instruction_vector.push_back(code);

                  code = instruction();
                  code.op = "mov_" + to_memory_type(type);
                  code.arg1 = arg1;
                  code.arg2 = "ebx";
                  _instruction_vector.push_back(code);
               }
               else {
*/
                  code = instruction();
                  code.op = "mov_" + to_memory_type(type);
                  code.arg1 = arg1;
                  code.arg2 = arg2;
                  _instruction_vector.push_back(code);
/*
               }
*/
            }
            else {
               if(to_memory_type(type) == "qword") {
                  code = instruction();
                  code.op = "mov_" + to_memory_type(type);
                  code.arg1 = arg1;
                  code.arg2 = arg2;
                  _instruction_vector.push_back(code);
               }
               else {
                  code = instruction();
                  code.op = "mov_" + to_memory_type(type);
                  code.arg1 = "edx";
                  code.arg2 = arg2;
                  _instruction_vector.push_back(code);

                  code = instruction();
                  code.op = "mov_" + to_memory_type(type);
                  code.arg1 = arg1;
                  code.arg2 = "edx";
                  _instruction_vector.push_back(code);
               }
            }
         }
      }
      else if(op == "+=") {
         code.op = type + "_add";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         code.op = "mov";
         code.arg1 = arg1;
         code.arg2 = "eax";
         _instruction_vector.push_back(code);
      }
      else if(op == "-=") {
         code.op = type + "_sub";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         code.op = "mov";
         code.arg1 = arg1;
         code.arg2 = "eax";
         _instruction_vector.push_back(code);
      }
      else if(op == "/=") {
         code.op = type + "_div";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         code.op = "mov";
         code.arg1 = arg1;
         code.arg2 = "eax";
         _instruction_vector.push_back(code);
      }
      else if(op == "*=") {
         code.op = type + "_mul";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         code.op = "mov";
         code.arg1 = arg1;
         code.arg2 = "eax";
         _instruction_vector.push_back(code);
      }
      else if(op == "%=") {
         code.op = type + "_mod";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         code.op = "mov";
         code.arg1 = arg1;
         code.arg2 = "eax";
         _instruction_vector.push_back(code);
      }
      else if(op == "<<=") {
         code.op = type + "_lsh";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         code.op = "mov";
         code.arg1 = arg1;
         code.arg2 = "eax";
         _instruction_vector.push_back(code);
      }
      else if(op == ">>=") {
         code.op = type + "_rsh";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         code.op = "mov";
         code.arg1 = arg1;
         code.arg2 = "eax";
         _instruction_vector.push_back(code);
      }
      else if(op == "&=") {
         code.op = type + "_and";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         code.op = "mov";
         code.arg1 = arg1;
         code.arg2 = "eax";
         _instruction_vector.push_back(code);
      }
      else if(op == "|=") {
         code.op = type + "_or";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         code.op = "mov";
         code.arg1 = arg1;
         code.arg2 = "eax";
         _instruction_vector.push_back(code);
      }
      else if(op == "^=") {
         code.op = type + "_xor";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         code.op = "mov";
         code.arg1 = arg1;
         code.arg2 = "eax";
         _instruction_vector.push_back(code);
      }
   }

   bool is_register(const string & r);

   size_t get_size(const string & str);

   string to_nasm(const string & str);

public:
   struct_rep * get_struct(const string & name);

   FileLine get_file_line() 
   {
      FileLine ret;

      for(size_t i = 0; i < _file_line_vector.size(); i++) {
         if((i+1) < _file_line_vector.size()) {
            if(_file_line_vector[i].offset >= _offset 
               && _offset <= _file_line_vector[i+1].offset) {
               ret = _file_line_vector[i]; 
               break;
            }
         }
      }
 
      return ret;
   }

protected:

   string to_hex(const float f);

   bool is_double(const string & str);

   bool is_float(const string & str);

   bool is_int(const string & str);

   bool is_char(const string & str);

   bool is_short(const string & str);

   bool is_long(const string & str);

   // tools:
   bool is_type(const string & type_string, 
                const string & type);

   bool is_typename(const string & name) {
      if(is_type(name, "int")
         || is_type(name, "float")
         || is_type(name, "void")
         || is_type(name, "char")
         || is_type(name, "short")
         || is_type(name, "long")
         || is_type(name, "double")
         || is_type(name, "static")
         || is_type(name, "unsigned")
         || is_type(name, "signed")
         || is_type(name, "register")
         || is_type(name, "const")) {
         return true;
      }
      else {
         for(size_t i = 0; i < _struct_vector.size(); i++) {
            if(_struct_vector[i].name == name) {
               return true;
            }
         }

         for(size_t i = 0; i < _typedef_vector.size(); i++) {
            if(_typedef_vector[i].name == name) {
               return true;
            }
         }

         for(size_t i = 0; i < _enum_vector.size(); i++) {
            if(_enum_vector[i].name == name) {
               return true;
            }
         }

         for(size_t i = 0; i < _union_vector.size(); i++) {
            if(_union_vector[i].name == name) {
               return true;
            }
         }
      }

      return false;
   }

   bool is_function(const string & name, int & index);

   bool match(const string val,
              token_type type);

   void reuse_temp() {
      for(size_t i = 0; i < _temp_vec.size(); i++)
         _reuse_temp_vec.push_back(_temp_vec[i]);

      _temp_vec.clear();
   }

   string get_next_temp(const string & struct_name, size_t size) {
      string temp;

      if(_reuse_temp_vec.size()) {
         bool found = false;

         temp_vector_type::iterator ptr;

         for(ptr = _reuse_temp_vec.begin(); ptr != _reuse_temp_vec.end(); ptr++) {
            if(ptr->second == size) {
               temp = ptr->first;
               found = true;
               break;
            }
         }

         if(found) {
            _reuse_temp_vec.erase(ptr);
            return temp;
         }
      }

      const size_t MAX = 64;

      char buffer[MAX];

      memset(buffer, 0, MAX);

      sprintf(buffer, "T%d", _current_temp);

      _current_temp++; 

      variable_rep var(this);

      var.label = string(buffer);

      var.type = struct_name;

      var.size = size;

      _temp_vec.push_back(pair< string, size_t >(buffer, size));

      _symbol_table[var.label] = var;

      _type_map[var.label] = var.type;

      //_current_function.variable_vector.push_back(var);

      _variable_vector.push_back(var);

      temp = string(buffer);

      return temp;
   }

   string get_next_temp(int size = 4) { 
      const size_t MAX = 64;

      char buffer[MAX];

      memset(buffer, 0, MAX);

      sprintf(buffer, "T%d", _current_temp);

      _current_temp++;

      variable_rep var(this);

      var.label = string(buffer);

      if(size == 8) {
         var.type = "double";
      }
      else if(size == 4) {
         var.type = "int";
      }

      _symbol_table[var.label] = var;

      //_current_function.variable_vector.push_back(var);

      _variable_vector.push_back(var);

      return string(buffer);
   }

   string get_next_label() {
      const size_t MAX = 64;

      char buffer[MAX];

      memset(buffer, 0, MAX);

      sprintf(buffer, ".L%d", _current_label);

      _current_label++;

      return string(buffer);
   }

   string get_next_const() {
      const size_t MAX = 64;

      char buffer[MAX];

      memset(buffer, 0, MAX);

      sprintf(buffer, "C%d", _current_const);

      _current_const++;

      return string(buffer);
   }

   bool is_const(const string & str);

   bool is_variable(const string & str) {
      bool ret = false;

      for(size_t i = 0; i < _variable_vector.size(); i++) {
         if(_variable_vector[i].label == str) {
            ret = true;
            break;
         }
      }

      return ret;
   }

   string to_code(const string & str);
   string to_code2(const string & str);

   // statements:
   bool global_statement();
   bool var_decl_statement(size_t & the_size);
   bool function_decl_statement();
   bool function_impl_statement();
   bool typedef_statement();
   bool struct_statement();
   bool union_statement();
 
   bool statement();
   bool return_statement();
   bool continue_statement();
   bool break_statement();
   bool goto_statement();
   bool label_statement();
   bool if_statement();
   bool switch_statement();
   bool while_statement();
   bool for_statement();
   bool do_while_statement();
   bool compound_statement();
   bool assignment_statement();
   bool expression_statement();

   // expression:
   int expression_call_list();
   string const_init();
   string expression();
   string boolean();
   string relation();
   string term();
   string factor();
   string bit_math();
   string primary();

public:
   c_compiler(const char * ifile_name,
              const char * ofile_name)
   :_in(ifile_name),
    _out(ofile_name),
    _current_struct(this)
   {
      _offset = 0;

      _current_temp = 0;

      _current_label = 0;

      _current_const = 0;

      _count_peren = 0U;

      _no_assign = false;

      _first_compound_statement = false;

      _got_return = false;

      _is_global = true;

      _size = 0U;

      _in_call_list = false;

      _address_of = false;
   }

   ~c_compiler() {
      _out.close();
      _in.close();
   }

   operator void*() {
      void * ret;

      if(!_in)
        ret = (void*)0;
      else if(!_out)
        ret = (void*)0;
      else
        ret = (void*)1;

      return ret;
   }

   bool lex();

   bool parse();

   bool optimize();

   bool strong_optimize() { return false; }

   bool emit_nasm();

   bool emit_gas();

   bool emit() { return emit_nasm(); }

   string type_string(token_type t) {
      string str;

      switch(t) {
         case STRING_VALUE:
            str = "STRING_VALUE";
            break;

         case CHAR_VALUE:
            str = "CHAR_VALUE";
            break;

         case SHORT_VALUE:
            str = "SHORT_VALUE";
            break;

         case INT_VALUE:
            str = "INT_VALUE";
            break;

         case LONG_VALUE:
            str = "LONG_VALUE";
            break;

         case FLOAT_VALUE:
            str = "FLOAT_VALUE";
            break;

         case DOUBLE_VALUE:
            str = "DOUBLE_VALUE";
            break;

         case UCHAR_VALUE:
            str = "UCHAR_VALUE";
            break;

         case USHORT_VALUE:
            str = "USHORT_VALUE";
            break;

         case UINT_VALUE:
            str = "UINT_VALUE";
            break;

         case ULONG_VALUE: 
            str = "ULONG_VALUE";
            break;

         case ID:
            str = "ID";
            break;

         case KEY_WORD:
            str = "KEY_WORD";
            break;

         case OPERATOR:
            str = "OPERATOR";
            break;
      }

      return str;
   }

   void print_token_list() {
      for(size_t i = 0; i < _token_vector.size(); i++) {
         cout << "TOKEN(" 
              << i << ",\""
              << (_token_vector[i].value) 
              << "\"," << type_string(_token_vector[i].type) 
              << ")" << endl << flush;
      }
   }

   bool is_key_word(const char * word);
};

typedef vector< string > string_vector_type;

string
c_compiler::to_hex2(int i)
{
   char buffer[1024];
   memset(buffer, 0, 1024);
   sprintf(buffer, "%x", i);
   return string(buffer);
}

string
c_compiler::to_asm_string(const string & str)
{
   string ret;
   int i = atoi(str.c_str());

   ret = to_hex2(i);

   return ret;
}

string
c_compiler::const_init()
{
   string ret, arg;

   if(match("{", OPERATOR)) {
      _offset++;

      string temp = const_init();

      while(temp.size()) {
         arg += to_asm_string(temp);

         if(match(",", OPERATOR)) {
            arg += ",";

            _offset++;
         }
         else {
            break;
         }

         temp = const_init();
      }

      if(match("}", OPERATOR)) {
         _offset++;

         ret = get_next_const();

         variable_rep var(this);

         var.label = ret;

         var.default_value = arg;
          
         var.type = "int *";

         var.pointer_count = 1;

         _variable_vector.push_back(var);

         _symbol_table[ret] = var;

         _type_map[ret] = "int *";
      }
   }
   else {
      _in_call_list = true;
      size_t off = _offset;
      arg = expression();
      _in_call_list = false;

      if(is_const(arg))
         ret = arg;
      else
         _offset = off;
   }

   return ret;
}

param_vector_type c_compiler::get_param()
{
   param_vector_type param_vector;
   param_type param;

   while(true) {
      param.second = "";

      if(match("const", KEY_WORD)) {
         param.second += "const ";
         _offset++;
      }

      if(match("unsigned", KEY_WORD)) {
         param.second += "unsigned ";
         _offset++;

         if(match("short", KEY_WORD)) {
            param.second += "short";
            _offset++;
         }
         else if(match("int", KEY_WORD)) {
            param.second += "int";
            _offset++;
         }
         else if(match("char", KEY_WORD)) {
            param.second += "char";
            _offset++;
         }
      }
      else {
         if(match("int", KEY_WORD)) {
            param.second += "int";
            _offset++;
         }
         else if(match("float", KEY_WORD)) {
            param.second += "float";
            _offset++;
         }
         else if(match("double", KEY_WORD)) {
            param.second += "double";
            _offset++;
         }
         else if(match("long", KEY_WORD)) {
            param.second += "long";
            _offset++;
         }
         else if(match("short", KEY_WORD)) {
            param.second += "short";
            _offset++;
         }
         else if(match("char", KEY_WORD)) {
            param.second += "char";
            _offset++;
         }
         else if(match("...", OPERATOR)) {
            param.second += "...";
            _offset++;
         }
         else if(match("struct", KEY_WORD)) {
            param.second += "struct";
            _offset++;

            if(_token_vector[_offset].type == ID) {
               param.second += " " + _token_vector[_offset].value;
               _offset++;
            }
         }
         else if(match("void", KEY_WORD)) {
            param.second += "void";
            _offset++;
         }

         while(_token_vector[_offset].value == "*") {
            param.second += "*";
            _offset++;
         }

         if(_token_vector[_offset].type == ID) {
            param.first = _token_vector[_offset].value;
            _offset++;

            variable_rep var(this);

            var.type = param.second;

            var.label = param.first;

            _symbol_table[param.first] = var;

            _type_map[param.first] = param.second;
         }
         else {
            param.first = "";
         }

         param_vector.push_back(param);

         if(match(",", OPERATOR)) {
            _offset++;
         }
         else {
            break;
         }
      }
   }

   return param_vector;
}

size_t 
variable_rep::get_struct_size(const string & struct_name)
{
   size_t ret = 0;

   if(p_c_compiler) {
      struct_rep * p_struct_rep = p_c_compiler->get_struct(struct_name);
      ret += p_struct_rep->get_size();
   }

   return ret;
}

bool
c_compiler::compound_statement()
{
   bool ret = false;

   if(match("{", OPERATOR)) {
      _offset++;

      if(_first_compound_statement) {
         instruction code;
         code = instruction();
         code.op = "push";
         code.arg1 = "ebp";
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "mov";
         code.arg1 = "ebp";
         code.arg2 = "esp";
         _instruction_vector.push_back(code);
      }

      size_t off = _offset;

      size_t the_size = 0U;
      size_t the_temp = 0;

      bool got_var_decl = false;

      while(var_decl_statement(the_size)) {
         the_size += the_temp;
cout << "got_var_decl..." << endl << flush;
         off = _offset;
         got_var_decl = true;
      }

      instruction code;
      code.op = "sub";
      code.arg1 = "esp";
      code.arg2 = to_string(the_size);
      _instruction_vector.push_back(code);

      for(size_t i = 0; i < _init_code.size(); i++)
         _instruction_vector.push_back(_init_code[i]);

      _init_code.clear();

      _offset = off;

      _size = the_size;

      bool temp = _first_compound_statement;

      _first_compound_statement = false;

      while(statement()) {
cout << "statement..." << endl << flush;
      }

      _first_compound_statement = temp;

      size_t index = _instruction_vector.size()-1;

      index--;

      if(_instruction_vector[index].op != "leave" && _instruction_vector[index+1].op != "ret") {
         if(got_var_decl) {
            char buffer[1024];
            memset(buffer, 0, 1024);
            sprintf(buffer, "%d", the_size);

            instruction code;
            code = instruction();
            code.op = "add";
            code.arg1 = "esp";
            code.arg2 = buffer;
            _instruction_vector.push_back(code);
         }
      }

      if(match("}", OPERATOR)) {
         _offset++;
         ret = true;
 
         size_t index = _instruction_vector.size()-1;

         index--;
    
         if(_instruction_vector[index].op != "leave" && _instruction_vector[index+1].op != "ret") {
            if(_first_compound_statement) {
               code = instruction();
               code.op = "leave";
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "ret";
               _instruction_vector.push_back(code);
            }
         }
      }
   }

   return ret;
}

bool
c_compiler::if_statement()
{
   string label2;
   bool ret = false;

   if(match("if", KEY_WORD)) {
      _offset++;
      ret = true;

      label2 = get_next_label();

      if(match("(", OPERATOR)) {
         _offset++;

         string str = expression();

         if(match(")", OPERATOR)) {
            _offset++;

            instruction code;

            string label = get_next_label();

            code = instruction();
            code.op = "mov_dword";
            code.arg1 = "eax";
            code.arg2 = str;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "cmp";
            code.arg1 = "eax";
            code.arg2 = "0";
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "je";
            code.arg1 = label;
            _instruction_vector.push_back(code);

            if(!statement()) {
               error("expected: <statement>");
            }

            code = instruction();
            code.op = "jmp";
            code.arg1 = label2;
            _instruction_vector.push_back(code);

            code = instruction();
            code.label = label;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "nop";
            _instruction_vector.push_back(code);

            while(match("else", KEY_WORD)) {
               _offset++;

               if(match("if", KEY_WORD)) {
                  _offset++;

                  if(match("(", OPERATOR)) {
                     _offset++;

                     string expr = expression();

                     if(match(")", OPERATOR)) {
                        _offset++;

                        code = instruction();
                        code.op = "mov_dword";
                        code.arg1 = "eax";
                        code.arg2 = expr;
                        _instruction_vector.push_back(code);

                        code = instruction();
                        code.op = "cmp";
                        code.arg1 = "eax";
                        code.arg2 = "0";
                        _instruction_vector.push_back(code);

                        label = get_next_label();
                        string label3 = get_next_label();

                        code = instruction();
                        code.op = "je";
                        code.arg1 = label;
                        _instruction_vector.push_back(code);

                        if(!statement()) {
                           error("expected: <statement>");
                        }

                        code = instruction();
                        code.op = "jmp";
                        code.arg1 = label2;
                        _instruction_vector.push_back(code);

                        code = instruction();
                        code.label = label;
                        _instruction_vector.push_back(code);

                        code = instruction();
                        code.op = "nop";
                        _instruction_vector.push_back(code);
                     }
                  }
               }
               else {
                  _offset--;
                  break;
               }
            }

            if(match("else", KEY_WORD)) {
               _offset++;
               if(!statement()) {
                  error("expected: <statement>");
               }
            }

            code = instruction();
            code.label = label2;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "nop";
            _instruction_vector.push_back(code);
         }
      }
   }

   return ret;
}

bool
c_compiler::for_statement()
{
   bool ret = false;
   string temp = _loop_condition;

   if(match("for", KEY_WORD)) {
      _offset++;
      instruction code;

      if(match("(", OPERATOR)) {
         _offset++;
         string label1 = get_next_label();
         string label2 = get_next_label();

         string start = expression();

         if(match(";", OPERATOR)) {
            _offset++;
         }

         code = instruction();
         code.label = label1;
         _instruction_vector.push_back(code);

         _loop_condition = label1;

         string condition = expression();

         code = instruction();
         code.op = "mov_dword";
         code.arg1 = "eax";
         code.arg2 = condition;
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "cmp";
         code.arg1 = "eax";
         code.arg2 = "0"; 
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "je";
         code.arg1 = label2;
         _instruction_vector.push_back(code);

         if(match(";", OPERATOR)) {
            _offset++;
         }

         instruction_vector_type temp = _instruction_vector;

         string increment = expression();

         instruction_vector_type inc_code;

         for(size_t i = temp.size(); i < _instruction_vector.size(); i++) {
            inc_code.push_back(_instruction_vector[i]);
         }

         _instruction_vector = temp;

         if(match(")", OPERATOR)) {
            _offset++;
         }

         if(statement()) {
            ret = true;

            for(size_t i = 0; i < inc_code.size(); i++) {
               _instruction_vector.push_back(inc_code[i]);
            }

            code = instruction();
            code.op = "jmp";
            code.arg1 = label1;
            _instruction_vector.push_back(code);

            code = instruction();
            code.label = label2; 
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "nop"; 
            _instruction_vector.push_back(code);
         }
      }
   }

   _loop_condition = temp;

   return ret;
}

bool
c_compiler::while_statement()
{
   bool ret = false;
   string temp = _loop_condition;
   if(match("while", KEY_WORD)) {
      _offset++;

      if(match("(", OPERATOR)) {
         _offset++;

         string label = get_next_label();
         string label2 = get_next_label();

         instruction code;

         code = instruction();
         code.label = label;
         _loop_condition = label;
         _instruction_vector.push_back(code);

         string str = expression();

         if(str.size() == 0) {
            error("expression expected");
         }

         code = instruction();
         code.op = "mov_dword";
         code.arg1 = "eax";
         code.arg2 = str;
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "cmp";
         code.arg1 = "eax";
         code.arg2 = "0";
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "je";
         code.arg1 = label2;
         _instruction_vector.push_back(code);

         if(match(")", OPERATOR)) {
            _offset++;

            if(statement()) {
               ret = true;
            }
         }

         code = instruction();
         code.op = "jmp";
         code.arg1 = label;
         _instruction_vector.push_back(code);

         code = instruction();
         code.label = label2;
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "nop";
         _instruction_vector.push_back(code);
      }
   }
   _loop_condition = temp;
   return ret;
}

bool
c_compiler::do_while_statement()
{
   bool ret = false;

   if(match("do", KEY_WORD)) {
      _offset++;

      string label = get_next_label();

      string temp = _loop_condition;

      _loop_condition = label;

      instruction code;
      code.label = label;
      _instruction_vector.push_back(code);

      if(compound_statement()) {

         if(match("while", KEY_WORD)) {
            _offset++;

            if(match("(", OPERATOR)) {
               _offset++;

               string str = expression(); 

               if(match(")", OPERATOR)) {
                  _offset++;

                  if(match(";", OPERATOR)) {
                     _offset++;
                     ret = true;

                     code = instruction();
                     code.op = "mov_dword";
                     code.arg1 = "eax";
                     code.arg2 = str;
                     _instruction_vector.push_back(code);

                     code = instruction();
                     code.op = "cmp";
                     code.arg1 = "eax";
                     code.arg2 = "0";
                     _instruction_vector.push_back(code);

                     code = instruction();
                     code.op = "jne";
                     code.arg1 = label;
                     _instruction_vector.push_back(code);
                  }
               }
            }
         }
      }

      _loop_condition = temp;
   }
   return ret;
}

bool
c_compiler::switch_statement()
{
   bool ret = false;

   if(match("switch", KEY_WORD)) {
      _offset++;

      if(match("(", OPERATOR)) {
         _offset++;

         string expr = expression();

         if(match(")", OPERATOR)) {
            _offset++;

            if(match("{", OPERATOR)) {
               _offset++;

               string label2 = get_next_label();
               instruction code;

               while(match("case", KEY_WORD)) {
                  _offset++;

                  if(_token_vector[_offset].type == INT_VALUE || _token_vector[_offset].type == CHAR_VALUE) {
                     string value = _token_vector[_offset].value;

                     _offset++;

                     if(match(":", OPERATOR)) {
                        _offset++;
                     }

                     code = instruction();
                     code.op = "mov_dword";
                     code.arg1 = "eax";
                     code.arg2 = value;
                     _instruction_vector.push_back(code);

                     code = instruction();
                     code.op = "cmp";
                     code.arg1 = "eax";
                     code.arg2 = "0";
                     _instruction_vector.push_back(code);

                     string label = get_next_label();

                     code = instruction();
                     code.op = "je";
                     code.arg1 = label;
                     _instruction_vector.push_back(code);

                     while(statement()) {

                     }

                     code = instruction();
                     code.label = label;
                     _instruction_vector.push_back(code);

                     code = instruction();
                     code.op = "nop";
                     _instruction_vector.push_back(code);

                     if(match("break", KEY_WORD)) {
                        _offset++;

                        if(match(";", OPERATOR)) {
                           _offset++;

                           code = instruction();
                           code.op = "jmp";
                           code.arg1 = label2;
                           _instruction_vector.push_back(code);
                        }
                     }
                  }
               }

               if(match("default", KEY_WORD)) {
                  _offset++;

                  if(match(":", OPERATOR)) {
                     _offset++;

                     while(statement()) {

                     }
                  }
               }

               if(match("}", OPERATOR)) {
                  _offset++;

                  code = instruction();
                  code.label = label2;
                  _instruction_vector.push_back(code);

                  code = instruction();
                  code.op = "nop";
                  _instruction_vector.push_back(code);

                  ret = true;
               }
            }
         }
      }
   }

   return ret;
}

bool
c_compiler::return_statement()
{
   bool ret = false;

   if(match("return", KEY_WORD)) {
      _offset++;
      _got_return = true;

      if(match(";", OPERATOR)) {
         _offset++;

         ret = true;

         instruction code;

         code = instruction();
         code.op = "mov_dword";
         code.arg1 = "eax";
         code.arg2 = "0";
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "leave";
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "ret";
         _instruction_vector.push_back(code);
      }
      else {
         string str = expression();

         if(match(";", OPERATOR)) {
            _offset++;

            ret = true;

            struct_rep * p_struct;

            instruction code;

            if((p_struct = get_struct(to_struct(_type_map[str]))) != 0) {
               code = instruction();
               code.op = "mov_dword";
               code.arg1 = "edx";
               //code.arg2 = "retval+4";
               code.arg2 = "retval";
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "lea";
               code.arg1 = "ebx";
               code.arg2 = str;
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "mov_struct3";
               code.arg1 = "edx";
               code.arg2 = "ebx";
               code.arg3 = to_string(p_struct->get_size());
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "leave";
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "ret";
               _instruction_vector.push_back(code);

               _type_map["eax"] = _type_map[str] + "*";

cout << "return_statement: \"" << _type_map["eax"] << "\"" << endl << flush;
//cin.get(),cin.get();
            }
            else {
               code = instruction();
               code.op = "mov_dword";
               code.arg1 = "eax";
               code.arg2 = str;
               _instruction_vector.push_back(code); 

               code = instruction();
               code.op = "leave";
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "ret";
               _instruction_vector.push_back(code);
            }
         }
      }
   }

   return ret;
}

bool
c_compiler::goto_statement()
{
   bool ret = false;
 
   if(match("goto", KEY_WORD)) {
      _offset++;

      if(_token_vector[_offset].type == ID) {
         _offset++;

         instruction code;
         code.op = "jmp";
         code.arg1 = _token_vector[_offset].value;
         _instruction_vector.push_back(code);
      }
   }

   return ret;
}

bool
c_compiler::label_statement()
{
   bool ret = false;
   if(_token_vector[_offset].type == ID) {
      string label = _token_vector[_offset].value;
      _offset++;

      if(!match(":", OPERATOR)) {
         return false;
      }

      instruction code;
      code.label = "." + label;
      _instruction_vector.push_back(code);

      ret = true;
   }
   return ret;
}

bool
c_compiler::break_statement()
{
   bool ret = false;

   if(match("break", KEY_WORD)) {
      _offset++;

      if(match(";", OPERATOR)) {
         _offset++;

         ret = true;

         instruction code;
         code.op = "jmp";
         code.arg1 = _break_location;
         _instruction_vector.push_back(code);
      }
   }

   return ret;
}

bool
c_compiler::continue_statement()
{
   bool ret = false;

   if(match("continue", KEY_WORD)) {
      _offset++;

      if(match(";", OPERATOR)) {
         _offset++;

         ret = true;

         instruction code;
         code.op = "jmp";
         code.arg1 = _loop_condition;
         _instruction_vector.push_back(code);
      }
   }

   return ret;
}

bool
c_compiler::var_decl_statement(size_t & var_size) 
{
   string_vector_type key_word_vector;

   if(_offset >= _token_vector.size())
      return false;

   if(!is_key_word(_token_vector[_offset].value.c_str()))
      return false;

   string key_word_string;
   string key_word;
   string id;
   bool is_struct = false;
   param_vector_type param_vector;
   instruction_vector_type temp_code;
   //size_t old_offset = _offset;

   if(match("struct", KEY_WORD)) {
      is_struct = true;
      _offset++;
      int pointer_count;
      string struct_name;

      pointer_count = 0;

      vector< size_t > sz_vec;
 
      string star_str;
      string type;

      if(_token_vector[_offset].type == ID) {
         type = _token_vector[_offset].value;
         _offset++;
      }

      if(get_struct(type) == 0) {
         error(string(string("error: unknown struct \"") + type + string("\"")).c_str());
      }

      while(_token_vector[_offset].type == ID || match("*", OPERATOR)) {
         star_str.clear();
         pointer_count = 0;

         while(match("*", OPERATOR)) {
            _offset++;
            pointer_count++;
            star_str += "*";
         }

         if(_token_vector[_offset].type == ID) {
            id = _token_vector[_offset].value; 
            _offset++;
         }

         while(match("[", OPERATOR)) {
            _offset++;

            if(_token_vector[_offset].type == INT_VALUE) {
               string sz = _token_vector[_offset].value;

               sz_vec.push_back(atoi(sz.c_str()));

               _offset++;

               if(match("]", OPERATOR)) {
                  _offset++;
               }
               else {
                  break;
               }
            }
            else {
               break;
            }
         }

         _type_map[id] = string("struct ") + type + star_str;

         string dvalue;
         if(match("=", OPERATOR)) {
            _offset++;

            dvalue = const_init();
         }

cout << "id: \"" << id << "\" type: \"" << _type_map[id] << "\"" << endl << flush;

         if(match(",", OPERATOR)) {
            _offset++;

            pointer_count = 0;

            //print_struct(id, struct_name, var_size, pointer_count);

            variable_rep var(this);

            var.type = string("struct ") + type + star_str;

            var.label = id;

            var.pointer_count = pointer_count;

            var.global = _is_global;

            var.array = sz_vec;

            var.default_value = dvalue;

            if(_struct_name == "") {
cout << "NOT IN STRUCT: " << id << endl << flush;
cout << "VAR: " << var.label << endl << flush;
cout << "TYPE: " << var.type << endl << flush;
//cin.get();cin.get();

               _symbol_table[id] = var;

               if(_current_function.name == "") {
                  _variable_vector.push_back(var);
               }
               else {
                  //_current_function.variable_vector.push_back(var);

                  _variable_vector.push_back(var);
               }

               _type_map[var.label] = var.type;

               //print_struct(id, struct_name, var_size, pointer_count);
            }
            else {
cout << "IN STRUCT: " << _struct_name << endl << flush;
cout << "VAR: " << var.label << endl << flush;
cout << "TYPE: " << var.type << endl << flush;
//cin.get();cin.get();
               _current_struct.var_vec.push_back(var);

               //_variable_vector.push_back(var);

               _type_map[var.label] = var.type;
            }

            var_size += var.get_size();
         }
         else {
            break;
         }
      }

      if(match(";", OPERATOR)) {
         _offset++;

         variable_rep var(this);

         var.type = string("struct ") + type +  star_str;

         // DEBUG STAR: star
         //var.type = "struct " + type +  " * " + star_str;

         var.label = id;

         var.pointer_count = pointer_count;

         var.global = _is_global;

         var.array = sz_vec;

         if(_struct_name == "") {
cout << "POSITION(2a)" << endl << flush; 
cout << "NOT IN STRUCT: " << id << endl << flush;
cout << "VAR: " << id << endl << flush;
cout << "TYPE: " << var.type << endl << flush;
//cin.get();cin.get();

            _symbol_table[id] = var;

            if(_current_function.name == "") {
               _variable_vector.push_back(var);
            }
            else {
               //_current_function.variable_vector.push_back(var);

               _variable_vector.push_back(var);
            }

            _type_map[var.label] = var.type;

            //print_struct(id, struct_name, var_size, pointer_count);
         }
         else {
cout << "POSITION(2b)" << endl << flush; 
cout << "IN STRUCT: " << _struct_name << endl << flush;
cout << "VAR: " << var.label << endl << flush;
cout << "TYPE: " << var.type << endl << flush;
//cin.get();cin.get();
            _current_struct.var_vec.push_back(var);

            //_variable_vector.push_back(var);

            _type_map[var.label] = var.type;
         }

         var_size += var.get_size();

         return true;
      }

      return false;
   }

   while(is_key_word(_token_vector[_offset].value.c_str())) {
      key_word = _token_vector[_offset].value;

      key_word_vector.push_back(key_word);

      if(key_word_string.size()) {
         key_word_string += " ";
      }
 
      key_word_string += key_word;

      _offset++;
   }

cout << "var_decl_statement: \"" << key_word << "\"" << endl << flush;

   size_t address = 0U;

   while(true) {
      int pointer_count = 0;
  
      while(match("*", OPERATOR)) {
         key_word_string += "*";
         pointer_count++;

         _offset++;
      }

/*
      if(is_type(key_word_string, "void") 
         && key_word_string.find("*") == string::npos) {
         _offset = old_offset;
         return false;
      }
*/

      if(_token_vector[_offset].value == "(") {
         _offset++;

         if(_token_vector[_offset].value == "*") {
            _offset++;

            if(_token_vector[_offset].type == ID) {
               string the_id = _token_vector[_offset].value;

               _offset++;

               if(_token_vector[_offset].value == ")") {
                  _offset++;

                  if(match("(", OPERATOR)) {
                     _offset++;

                     param_vector = get_param();

                     if(match(")", OPERATOR)) {
                        _offset++;

                        variable_rep var(this);

                        if(param_vector.size()) {
                           var.type = key_word_string + " (*)()";
                        }
                        else {
                           var.type = key_word_string + " (*)(...)";
                        }

                        var.label = the_id;

                        var.pointer_count = 1;

                        var.address = DATA_WORD_SIZE;

                        //var.param_vector = param_vector;

                        address += var.get_size();

                        var_size += var.get_size();

                        if(_struct_name == "") {
                           _symbol_table[the_id] = var;

                           if(_current_function.name == "") {
                              _variable_vector.push_back(var);
                           }
                           else {
                              //_current_function.variable_vector.push_back(var);

                              _variable_vector.push_back(var);
                           }
                        }
                        else {
                           _current_struct.var_vec.push_back(var);

                           //_variable_vector.push_back(var);
                        }
                     }
                     else {
                        error(string("expected: \")\" got: \"" +_token_vector[_offset].value + "\"").c_str());
                     }
                  }
               }
            }
         }
      }
      else if(_token_vector[_offset].type == ID) {
         id = _token_vector[_offset].value;
 
cout << "id: " << id << endl << flush;
cout << "type: \"" << key_word << "\"" << endl << flush;
         
         variable_rep var(this);

         _offset++;

         while(match("[", OPERATOR)) {
            _offset++;

            if(is_const(_token_vector[_offset].value)
               && is_int(_token_vector[_offset].value)) {

               var.array.push_back(atoi(_token_vector[_offset].value.c_str()));

               _offset++;

               if(match("]", OPERATOR)) {
                  _offset++;
               }
               else {
                  error("expected: \"]\"");
                  break;
               }
            }
         }

         if(match("(", OPERATOR)) {
            return false;
         }

         if(match("=", OPERATOR)) {
            _offset++;

            var.default_value = const_init();

            if(var.default_value.size() == 0) {
               error("bad constant initializer");
            }
         }

         var.type = key_word_string;

         var.label = id;

         _type_map[id] = var.type;
        
         var.pointer_count = pointer_count;
 
         var.address = address;

         address += var.get_size();

         var_size += var.get_size();

         if(_struct_name == "") {
            _symbol_table[id] = var;

            if(_current_function.name != "") 
               //_current_function.variable_vector.push_back(var);
               _variable_vector.push_back(var);
            else
               _variable_vector.push_back(var);
         }
         else {
            _current_struct.var_vec.push_back(var);
            //_variable_vector.push_back(var);
         }
         
         if(var.default_value.size()) {
            if(var.array.size() == 0) {
               instruction code;
               code.op = "mov";
               code.arg1 = var.label;
               code.arg2 = var.default_value;
               _init_code.push_back(code);
            }
            else {
               instruction code;
               code = instruction();
               code.op = "push";
               code.arg1 = to_string(var.get_size());
               _init_code.push_back(code);

               code = instruction();
               code.op = "push";
               code.arg1 = var.default_value; 
               _init_code.push_back(code);

               code = instruction();
               code.op = "lea";
               code.arg1 = "edx";
               code.arg2 = var.label;
               _init_code.push_back(code);

               code = instruction();
               code.op = "push";
               code.arg1 = "edx";
               _init_code.push_back(code);

               code = instruction();
               code.op = "call";
               code.arg1 = "memcpy";
               _init_code.push_back(code);

               code = instruction();
               code.op = "add";
               code.arg1 = "esp";
               code.arg2 = "12";
               _init_code.push_back(code);
            }
         }
      }

      if(match(",", OPERATOR)) {
         _offset++;
      }
      else if(match(";", OPERATOR)) {
         break;
      }
      else {
         break;
      }
   }

   if(match(";", OPERATOR)) {
      _offset++;
      char buffer[1024];
      memset(buffer, 0, 1024);
      var_size = address;
      sprintf(buffer, "%d", address);

      if(!_first_compound_statement) {
         if(_struct_name == "") {
            instruction code;
            code.op = "sub";
            code.arg1 = "esp";
            code.arg2 = buffer;
            _instruction_vector.push_back(code);
         }
      }
   }
   else {
      return false;
   }

   return true;
}

bool
c_compiler::match(const string val, 
                  token_type type)
{
   if(_offset >= _token_vector.size()) {
      return false;
   }

   bool ret;

   token tok = _token_vector[_offset];

   if(tok.type == type) {
      if(tok.value == val) {
         ret = true;
      }
      else {
         ret = false;
      }
   }
   else {
      ret = false;
   }

   return ret;
}

bool
c_compiler::is_key_word(const char * arg)
{
   bool ret;

   if(strcmp(arg, "for") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "extern") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "while") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "do") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "if") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "else") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "switch") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "break") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "return") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "goto") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "struct") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "typedef") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "sizeof") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "short") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "int") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "long") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "float") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "double") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "char") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "void") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "unsigned") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "register") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "static") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "extern") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "volatile") == 0) {
      ret = true;
   }
   else if(strcmp(arg, "const") == 0) {
      ret = true;
   }
   else {
      ret = false;
   }

   return ret;
}

bool
is_digit(char ch)
{
   bool ret = false;

   switch(ch) {
   case '0':
   case '1':
   case '2':
   case '3':
   case '4':
   case '5':
   case '6':
   case '7':
   case '8':
   case '9':
      ret = true;
   break;
   }

   return ret;
}

bool
c_compiler::lex()
{
   char ch, next;
   string str;
   token tok;
   FileLine fn;
   size_t offset = 0U;

   _in.clear();

   _in.seekg(0L, ios::beg);

   while(_in) {

      ch = _in.get();

      if(!_in) {
         break;
      }

      next = _in.peek();

      if(ch == '#') {
         fn.offset =  offset;

         while((ch = _in.get()) != '\n') {

         }

         fn.file = file_name;

         fn.lineno = lineno;

         _file_line_vector.push_back(fn);

         lineno++;
      }
      else if(ch == '\'') {
         ch = _in.get();

         if(_in.peek() != '\'') {
            error("bad char token");
         }

         _in.get();

         char buffer[1024];
         memset(buffer, 0, 1024);
         sprintf(buffer, "\'%c\'", ch);

         tok.value = string(buffer);
 
         tok.type = CHAR_VALUE;

         _token_vector.push_back(tok); 
      }
      else if(ch == '\"') {
         string buffer;

         buffer = "\"";

         ch = _in.get();

         while(_in) {
            if(ch == '\\' && _in.peek() == '\"') {
               ch = _in.get(); 
               buffer += "\\\"";
            }
            else if(ch == '\"') {
               buffer += "\"";
               break;
            }
            else {
               buffer += ch;
            }

            ch = _in.get();
         }

         buffer += ",0";

         tok.value = buffer;
     
         tok.type = STRING_VALUE;

         _token_vector.push_back(tok);
      }
      else if(ch == ';') {
         tok.value = ";";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == ',') {
         tok.value = ",";
    
         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '=' && next == '=') {
         _in.get();

         tok.value = "==";

         tok.type = OPERATOR;
 
         _token_vector.push_back(tok);
      }
      else if(ch == '!' && next == '=') {
         _in.get();
         
         tok.value = "!=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else  if(ch == '=') {
         tok.value = "=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '<' && next == '=') {
         _in.get();

         tok.value = "<=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '<' && next == '<') {
         next = _in.get();
         next = _in.peek();

         if(next == '=') {
            tok.value = "<<=";

            tok.type = OPERATOR;
 
            _token_vector.push_back(tok);
         }
         else {
            tok.value = "<<";
 
            tok.type = OPERATOR;

            _token_vector.push_back(tok);
         }
      }
      else if(ch == '<') {
         tok.value = "<";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '>' && next == '>') {
         _in.get();

         next = _in.get();

         if(next == '=') {
            tok.value = ">>=";

            tok.type = OPERATOR;

            _token_vector.push_back(tok);
         }
         else {
            next = _in.get();

            tok.value = ">>";

            tok.type = OPERATOR;

           _token_vector.push_back(tok);
         }   
      }
      else if(ch == '>' && next == '=') {
         _in.get();

         tok.value = ">=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '>') {
         tok.value = ">";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '+' && next == '+') {
         _in.get();
 
         tok.value = "++";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '+' && next == '=') {
         _in.get();

         tok.value = "+=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '+') {
         tok.value = "+";
        
         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '-' && next == '-') {
         _in.get();

         tok.value = "--";
 
         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '-' && next == '=') {
         _in.get();

         tok.value = "-=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '-' && next != '>') {
         tok.value = "-";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '*' && next == '=') {
         _in.get();

         tok.value = "*=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '*') {
         tok.value = "*";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '%' && next == '=') {
         _in.get();

         tok.value = "%=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '%') {
         tok.value = "%";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '/' && next == '=') {
         _in.get();

         tok.value = "/=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '/') {
         if(_in.peek() == '*') {
            ch = _in.get();
            ch = _in.get();

            while(ch != '*' && _in.peek() != '/') {
               ch = _in.get();
            }
         }

         tok.value = "/";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if((ch == '&') && (next == '&')) {
         _in.get();

         tok.value = "&&";
 
         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if((ch == '|') && (next == '|')) {
         _in.get();

         tok.value = "||";
          
         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '.') {
         if(next == '.') {
            ch = _in.get();

            if(_in.peek() == '.') {
               ch = _in.get();

               tok.value = "...";

               tok.type = OPERATOR;

               _token_vector.push_back(tok);
            }
            else {
               error("bad lex token: \"..\"");
            }
         }
         else {
            tok.value = ".";

            tok.type = OPERATOR;

            _token_vector.push_back(tok);
         }
      }
      else if(ch == '-' && next == '>') {
         _in.get();

         tok.value = "->";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '&' && next == '=') {
         _in.get();

         tok.value = "&=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '&') {
         tok.value = "&";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '|' && next == '=') {
         _in.get();

         tok.value = "|=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '|') {
         tok.value = "|";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '!') {
         tok.value = "!";
 
         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '~' && next == '=') {
         _in.get();

         tok.value = "~=";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '~') {
         tok.value = "~";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '^' && next == '=') {
         _in.get();

         tok.value = "^=";
      
         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '^') {
         tok.value= "^";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '(') {
         tok.value = "(";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == ')') {
        tok.value = ")";

        tok.type = OPERATOR;

        _token_vector.push_back(tok);
      }
      else if(ch == '{') {
         tok.value = "{";
      
         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '}') {
         tok.value = "}";
         
         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == '[') {
         tok.value = "[";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(ch == ']') {
         tok.value = "]";

         tok.type = OPERATOR;

         _token_vector.push_back(tok);
      }
      else if(isalpha(ch) || ch == '_') {
         tok.value = "";

         tok.value += ch;

         while((ch = _in.peek())) {
            if(!isalpha(ch) && !is_digit(ch) && ch != '_') {
               break;
            }
 
            ch = _in.get();

            tok.value += ch;
         }

         if(is_key_word(tok.value.c_str())) {
            tok.type = KEY_WORD;
         }
         else {
            tok.type = ID;
         }

         _token_vector.push_back(tok);
      }
      else if(isdigit(ch)) {
         tok.value = "";

         tok.value += ch;

         while(isdigit(ch = _in.peek())) {
            tok.value += ch;

            _in.get();
         }

         if(ch == '.') {
            tok.value += ".";
            _in.get();

            while(isdigit(_in.peek())) {
               tok.value += _in.get();
            }

            if(_in.peek() == 'F') {
               ch = _in.get();
               tok.type = FLOAT_VALUE;
            }
            else {
               tok.type = DOUBLE_VALUE;
            }
         }
         else {
            if(_in.peek() == 'L') {
               ch = _in.get();
               tok.type = LONG_VALUE;
            }
            else if(_in.peek() == 'U') {
               ch = _in.get();
               tok.type = UINT_VALUE;
            }
            else {
               tok.type = INT_VALUE;
            }
         }

         _token_vector.push_back(tok);
      }
      else if(ch == '\n') {
         fn.lineno = lineno;
         fn.file = file_name;
         fn.offset = offset;
         _file_line_vector.push_back(fn);
         lineno++;
      }
      else if(isspace(ch)) {
         ;
      }
      else if(ch == ':') {
         ch = _in.get();

         next = _in.peek();

         tok.type = OPERATOR;

         tok.value = ":";

         _token_vector.push_back(tok);
      }
      else if(ch == '\?') {
         ch = _in.get();

         next = _in.peek();

         tok.type = OPERATOR;

         tok.value = "\?";

         _token_vector.push_back(tok);
      }
      else {
         error((string("bad lex token: got: \"") + ch + string("\"")).c_str()); 
      }

      offset++;
   }

   return true;
}

string
to_string(int count)
{
   const size_t MAX = 1024;
   char buffer[MAX];
   memset(buffer, 0, MAX);
   sprintf(buffer,"%d", count);
   return string(buffer);
}

string
to_code_string(string str)
{
   string ret;
   char ch;

   for(size_t i = 0; i < str.size(); i++) {
      switch(str[i]) {
      case '\\':
         if(str[i+1] == 'n') {
            ch = 10;
            ret += "\"," + to_string((int)ch) + ",\"";
            i++;
         }
         else if(str[i+1] == 'r') {
            ch = 10;
            ret += "\"," + to_string((int)ch) + ",\"";
            i++;
         }
         else if(str[i+1] == '\'') {
            ch = (int)('\'');
            ret += "\"," + to_string((int)ch) + ",\"";
            i++;
         }
         else if(str[i+1] == '\"') {
            ch = (int)('\"');
            ret += "\"," + to_string((int)ch) + ",\"";
            i++;
         }
         break;

      default:
         ret += str[i];
      }
   }

   return ret;
}

int
c_compiler::expression_call_list()
{
   int count = 0;

   _expr_call_list.clear();

   if(match(")", OPERATOR)) 
      return 0;

   string expr;
   instruction code;

   instruction_vector_type old_code = _instruction_vector;

   while(true) {
      _instruction_vector.clear();

      expr = expression();

      if(!expr.size()) 
         break;

      _expr_call_list.insert(_expr_call_list.begin(), Expression(_instruction_vector, expr, _type_map[expr]));

      if(match(",", OPERATOR)) 
         _offset++;
      else 
         break;
   }

   _instruction_vector = old_code;

   for(size_t i = 0; i < _expr_call_list.size(); i++) {
      for(size_t j = 0; j < _expr_call_list[i].code.size(); j++)
         _instruction_vector.push_back(_expr_call_list[i].code[j]);

      expr = _expr_call_list[i].expr;

      _type_map[expr] = _expr_call_list[i].type;

      if(is_float(expr)) {
         if(_current_function.name == "printf") {
            code = instruction();
            code.op = "fld_dword";
            code.arg1 = expr;
            _instruction_vector.push_back(code);

            string t =  get_next_temp(8);

            code = instruction();
            code.op = "fstp";
            code.arg1 = t;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "push_qword";
            code.arg1 = t;
            _instruction_vector.push_back(code);

            count += 8;
         }
         else {
            code = instruction();
            code.op = "push";
            code.arg1 = expr;
            _instruction_vector.push_back(code);

            count += 4;
         }
      }
      else if(is_double(expr)) {
         count += 4;

//cout << "is_double(\"" << expr << "\")" << endl << flush;

         code = instruction();
         code.op = "push_qword";
         code.arg1 = expr;
         _instruction_vector.push_back(code);

         count += 4;
      }
      else if(_type_map[expr].find("*") != string::npos) {
         code = instruction();
         code.op = "push";
         code.arg1 = expr;
         _instruction_vector.push_back(code);

         count += 4;
      }
      else if(is_char(expr)) {
         code = instruction();
         code.op = "mov_byte";
         code.arg1 = "dx";
         code.arg2 = expr;
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "push_byte";
         code.arg1 = "dx";
         _instruction_vector.push_back(code);

         count++;
      }
      else if(is_short(expr)) {
         if(_current_function.name == "printf") {
            code = instruction();
            code.op = "mov_word";
            code.arg1 = "dx";
            code.arg2 = expr;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "push_word";
            code.arg1 = "0";
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "push_word";
            code.arg1 = "dx";
            _instruction_vector.push_back(code);

            count += 4;
         }
         else {
            code = instruction();
            code.op = "mov_word";
            code.arg1 = "dx";
            code.arg2 = expr;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "push_word";
            code.arg1 = "dx";
            _instruction_vector.push_back(code);

            count += 2;
         }
      }
      else if(is_int(expr)) {
         code = instruction();
         code.op = "push";
         code.arg1 = expr;
         _instruction_vector.push_back(code);

         count += 4;
      }
      else if(is_long(expr)) {
         code = instruction();
         code.op = "push_qword";
         code.arg1 = expr;
         _instruction_vector.push_back(code);

         count += 8;
      }
      else if(is_type(_type_map[expr], "struct")) {
         struct_rep * p_struct = get_struct(to_struct(_type_map[expr]));

         if(p_struct) {
            code = instruction();
            code.op = "mov";
            code.arg1 = "ebx";
            code.arg2 = "esp";
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "sub";
            code.arg1 = "esp";
            code.arg2 = to_string(p_struct->get_size());
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "lea";
            code.arg1 = "edx";
            code.arg2 = expr;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "mov_struct3";
            code.arg1 = "ebx";
            code.arg2 = "edx";
            code.arg3 = to_string(p_struct->get_size());
            _instruction_vector.push_back(code);

            count += p_struct->get_size();
         }
         else {
            error(string(string("bad struct name: got: \"") + expr + string("\"")).c_str());
         }
      }
      else {
         code = instruction();
         code.op = "push";
         code.arg1 = expr;
         _instruction_vector.push_back(code);

         count += 4;
      }
   }

   return count;
}

/****
int
c_compiler::expression_call_list() 
{
   int count = 0;

   if(match(")", OPERATOR)) {
      return 0;
   }

   string expr;

   expr = expression();

   if(match(",", OPERATOR)) {
      _offset++;
   }

   if(expr.size()) {
      //count = 4;

      count += expression_call_list();

      instruction code;

      if(is_float(expr)) {
         if(_current_function.name == "printf") {
            code = instruction();
            code.op = "fld_dword";
            code.arg1 = expr;
            _instruction_vector.push_back(code);

            string t =  get_next_temp(8);

            code = instruction();
            code.op = "fstp";
            code.arg1 = t;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "push_qword";
            code.arg1 = t;
            _instruction_vector.push_back(code);

            count += 8;
         }
         else {
            code = instruction();
            code.op = "push";
            code.arg1 = expr;
            _instruction_vector.push_back(code);

            count += 4;
         }
      }
      else if(is_double(expr)) {
         count += 4;

//cout << "is_double(\"" << expr << "\")" << endl << flush;

         code = instruction();
         code.op = "push_qword";
         code.arg1 = expr;
         _instruction_vector.push_back(code);

         count += 4;
      }
      else if(_type_map[expr].find("*") != string::npos) {
         code = instruction();
         code.op = "push";
         code.arg1 = expr;
         _instruction_vector.push_back(code);

         count += 4;
      }
      else if(is_char(expr)) {
         code = instruction();
         code.op = "mov_byte";
         code.arg1 = "dx";
         code.arg2 = expr;
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "push_byte";
         code.arg1 = "dx";
         _instruction_vector.push_back(code);

         count++;
      }
      else if(is_short(expr)) {
         if(_current_function.name == "printf") {
            code = instruction();
            code.op = "mov_word";
            code.arg1 = "dx";
            code.arg2 = expr;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "push_word";
            code.arg1 = "0";
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "push_word";
            code.arg1 = "dx";
            _instruction_vector.push_back(code);

            count += 4;
         }
         else {
            code = instruction();
            code.op = "mov_word";
            code.arg1 = "dx";
            code.arg2 = expr;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "push_word";
            code.arg1 = "dx";
            _instruction_vector.push_back(code);

            count += 2;
         }
      }
      else if(is_int(expr)) {
         code = instruction();
         code.op = "push";
         code.arg1 = expr;
         _instruction_vector.push_back(code);
 
         count += 4;
      }
      else if(is_long(expr)) {
         code = instruction();
         code.op = "push_qword";
         code.arg1 = expr;
         _instruction_vector.push_back(code);

         count += 8;
      }
      else if(is_type(_type_map[expr], "struct")) {
         struct_rep * p_struct = get_struct(to_struct(_type_map[expr]));

         if(p_struct) {
            code = instruction();
            code.op = "mov";
            code.arg1 = "ebx";
            code.arg2 = "esp";
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "sub";
            code.arg1 = "esp";
            code.arg2 = to_string(p_struct->get_size());
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "lea";
            code.arg1 = "edx";
            code.arg2 = expr;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "mov_struct3";
            code.arg1 = "ebx";
            code.arg2 = "edx";
            code.arg3 = to_string(p_struct->get_size());
            _instruction_vector.push_back(code);

            count += p_struct->get_size();
         }
         else {
            error(string(string("bad struct name: got: \"") + expr + string("\"")).c_str());
         }
      }
      else {
         code = instruction();
         code.op = "push";
         code.arg1 = expr;
         _instruction_vector.push_back(code);

         count += 4;
      }
   }

   return count;
}

******/

struct_rep * 
c_compiler::get_struct(const string & name)
{
   struct_rep * p = 0;

   for(size_t i = 0; i < _struct_vector.size(); i++) {
      p = &(_struct_vector[i]);
      if(("struct " + p->name) == name || p->name == name) {
         break; 
      } 
      p = 0;
   }

   return p;
}

bool
c_compiler::is_struct(const string & t)
{
   bool ret = false;

   for(size_t i = 0; i < _struct_vector.size(); i++) {
      if(_struct_vector[i].name == to_struct(t)) { 
         ret = true;
         break;
      }
   }

   return ret;
}

bool
c_compiler::is_struct(const string & t, const string & id)
{
   bool ret = false;

   for(size_t i = 0; i < _struct_vector.size(); i++) {
      if(_struct_vector[i].name == t) {
         for(size_t j = 0; j < _struct_vector[i].var_vec.size(); j++) {
            if(_struct_vector[i].var_vec[j].label == id) {
               ret = true;
               break;
            }
         }
      }
      if(ret) {
         break;
      }
   }

   return ret;
}

function_rep * 
c_compiler::get_function(const string & name)
{
   function_rep * ret = 0;

   for(size_t i = 0; i < _function_vector.size(); i++) {
      if(_function_vector[i].name == name) {
         ret = &_function_vector[i];
         break;
      }
   }

   return ret;
}

bool
c_compiler::is_array(const string & name)
{
   bool ret = false;

   variable_rep var = _symbol_table[name];

   if(var.array.size())
      ret = true;

   return ret;
}

string
c_compiler::primary()
{
   instruction code;
   string pointer;
   bool again = true;
   string ret;

   while(again) {
   again = false;

   if(_offset >= _token_vector.size())
      return "";

   ret = "";

   if(match("(", OPERATOR) && pointer.size() == 0) {
       _offset++;
       
       if((ret = expression()) == "") {
          error("expected: <expression>");
       }

       if(match(")", OPERATOR)) {
          _offset++;
       }
   }
   else if(match("sizeof", KEY_WORD)) {
      _offset++;
 
      if(match("(", OPERATOR)) {
         _offset++;

         string the_type = _token_vector[_offset].value;
         _offset++;

         if(match(")", OPERATOR)) {
            _offset++;

            if(the_type == "char") {
               ret = "1";
            }
            else if(the_type == "short") {
               ret = "2";
               //ret = "4";
            }
            else if(the_type == "int") {
               ret = "4";
            }
            else if(the_type == "float") {
               ret = "4";
            }
            else if(the_type == "long") {
               ret = "8";
            }
            else if(the_type == "double") {
               ret = "8";
            }
            else if(is_struct(the_type)) {
               struct_rep * p = get_struct(the_type);
               if(p != 0) {
                  ret = to_string(p->get_size());
               }
            }
            else {
               ret = "4";
            }

            _type_map[ret] = "int";
         }
         else {
            error("expected: \"(\"");
         }
      }
   }
   else if(match("++", OPERATOR)) {
      _offset++;

      int code1_index = _instruction_vector.size();

//DEBUG ++
      instruction code1;
      code1.op = "mov_dword";
      code1.arg1 = "edx";
      _instruction_vector.push_back(code1);

      instruction code2;
      code2.op = "inc";
      code2.arg1 = "edx";
      _instruction_vector.push_back(code2);

      int code3_index = _instruction_vector.size();

      instruction code3;
      code3.op = "mov_dword";
      code3.arg2 = "edx";
      _instruction_vector.push_back(code3);

      string str = expression();

      _instruction_vector[code1_index].arg2 = str;
      _instruction_vector[code3_index].arg1 = str;

      ret = str;
   }
   else if(match("--", OPERATOR)) {
      _offset++;
   
      int code1_index = _instruction_vector.size();

//DEBUG --
      instruction code1;
      code1.op = "mov_dword";
      code1.arg1 = "edx";
      _instruction_vector.push_back(code1);

      instruction code2;
      code2.op = "dec";
      code2.arg1 = "edx";
      _instruction_vector.push_back(code2);

      int code3_index = _instruction_vector.size();

      instruction code3;
      code3.op = "mov_dword";
      code3.arg2 = "edx";
      _instruction_vector.push_back(code3);

      string str = expression();

      _instruction_vector[code1_index].arg2 = str;
      _instruction_vector[code3_index].arg1 = str;
      
      ret = str;
   }
   else if(_token_vector[_offset].type == ID || pointer.size()) {
      string id = _token_vector[_offset].value;

      if(pointer.size()) {
         id = pointer;
      }

      ret = id;

      int count = 0;

      string size;

      string type_string = _type_map[id];

      size = type_string;

      variable_rep var(this);

      symbol_table_type::iterator p;

      if((p = _symbol_table.find(id)) != _symbol_table.end()) {
         var = p->second;
      }

      string the_type;

      if(is_type(type_string, "int")) {
         size = "4";
         if(is_type(type_string, "unsigned")) {
            the_type = "uint";
         } 
         else {
            the_type = "int";
         }
      }
      else if(is_type(type_string, "float")) {
         size = "4";
         the_type = "float";
      }
      else if(is_type(type_string, "long")) {
         size = "8";
         if(is_type(type_string, "unsigned")) {
            the_type = "ulong";
         }
         else {
            the_type = "long";
         }
      }
      else if(is_type(type_string, "double")) {
         size = "8";
         the_type = "double";
      }
      else if(is_type(type_string, "short")) {
         size = "2";
         //size = "4";
         if(is_type(type_string, "unsigned")) {
            the_type = "ushort";
         }
         else {
            the_type = "short";
         }
      }
      else if(is_type(type_string, "char")) {
         size = "1";
         if(is_type(type_string, "unsigned")) {
            the_type = "uchar";
         }
         else {
            the_type = "char";
         }
      }
      else if(is_type(type_string, "struct") && type_string.find("*") == string::npos) {
         struct_rep * p_struct = get_struct(type_string);
         size_t sz = p_struct->get_size();
         char buffer[1024];
         memset(buffer, 0, 1024);
         sprintf(buffer, "%u", sz);
         size = string(buffer);
         the_type = "struct " + to_struct(type_string);
      }
      else {
         size = DATA_WORD_SIZE_STR;
         the_type = type_string;
      }

      if(pointer.size() == 0) {
         _offset++;
      }

      if(match("[", OPERATOR)) {
         _offset++;

         if(p == _symbol_table.end()) {
            error(("unknown variable \"" + id + "\"").c_str());
            return "";
         }

         string array_index = expression();
         string array_index2;

         if(match("]", OPERATOR)) {
            _offset++;
            instruction_vector_type temp_vec;
            size_t index = 0;

            while(match("[", OPERATOR)) {
               _offset++;

               if(index > var.array.size()) {
                  error("bad array index");
                  break;
               }

               array_index2 = expression();

               if(match("]", OPERATOR)) {
                  _offset++;

                  if(array_index2 != "") {
                     code = instruction();
                     code.op = "imul";
                     code.arg1 = "edx";
                     code.arg2 = to_string(var.array[index]);
                     temp_vec.push_back(code);

                     index++;
 
                     code = instruction();
                     code.op = "add";
                     code.arg1 = "edx";
                     code.arg2 = array_index2;
                     temp_vec.push_back(code);
                  } 
               }
               else {
                  error("expected: \"]\"");
                  break;
               }
            }

            string id_str = "0";
            string the_size = size;

/*
            if(match(".", OPERATOR)) {
               _offset++;
               if(_token_vector[_offset].type == ID) {
                  string name = _token_vector[_offset].value;

                  string t = _type_map[id];

                  struct_rep * p_struct = get_struct(to_struct(t));

                  string tt;

                  variable_rep var(this);

                  for(size_t x = 0; x < p_struct->var_vec.size(); x++) {
                     if(p_struct->var_vec[x].label == name) {
                        tt = p_struct->var_vec[x].type;
                        var = p_struct->var_vec[x];
                        break;
                     }
                  }

                  id_str = to_struct(t) + "-" + to_struct(t) + "_" + name;

                  if(is_type(tt, "int") && is_type(tt, "float")) {
                     the_size = "4";
                  }
                  else if(is_type(tt, "long") && is_type(tt, "double")) {
                     the_size = "8";
                  }
                  else if(is_type(tt, "short")) {
                     the_size = "2";
                  }
                  else if(is_type(tt, "char")) {
                     the_size = "1";
                  }
                  else {
                     the_size = "4";
                  }

                  _offset++;
               }
            }
*/

            if(is_assign(_token_vector[_offset].value)) {
               string op = _token_vector[_offset].value;

               _offset++;

               string t = expression();

               string tt;

               code = instruction();
               code.op = "push";
               code.arg1 = the_size;
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "push2";
               code.arg1 = id_str;
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "push";
               code.arg1 = size;
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "mov_dword";
               code.arg1 = "edx";
               code.arg2 = t;
               _instruction_vector.push_back(code);

               tt = get_next_temp();

               code = instruction();
               code.op = "mov_dword";
               code.arg1 = tt;
               code.arg2 = "edx";
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "lea";
               code.arg1 = "edx";
               code.arg2 = tt;
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "push_dword";
               code.arg1 = "edx";
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "mov_dword";
               code.arg1 = "edx";
               code.arg2 = array_index;
               _instruction_vector.push_back(code);

               for(size_t ii = 0; ii < temp_vec.size(); ii++) {
                  _instruction_vector.push_back(temp_vec[ii]);
               }

               code = instruction();
               code.op = "push";
               code.arg1 = "edx";
               _instruction_vector.push_back(code);

               if(_type_map[id].find("*") != string::npos && is_type(_type_map[id], "struct")) {
                  code = instruction();
                  code.op = "lea";
                  code.arg1 = "edx";
                  code.arg2 = id;
                  _instruction_vector.push_back(code);
               }
               else if(_type_map[id].find("*") == string::npos) {
                  code = instruction();
                  code.op = "lea";
                  code.arg1 = "edx"; 
                  code.arg2 = id; 
                  _instruction_vector.push_back(code);
               }
               else {
                  code = instruction();
                  code.op = "mov_dword";
                  code.arg1 = "edx"; 
                  code.arg2 = id; 
                  _instruction_vector.push_back(code);
               }

               code = instruction();
               code.op = "push_dword";
               code.arg1 = "edx"; 
               _instruction_vector.push_back(code);

               if(op == "=") {
                  code = instruction();
                  code.op =  "array_put";
                  _instruction_vector.push_back(code);
               }
               else if(op == "+=") {
                  code = instruction();
                  code.op = "add_array_put_" + the_type;
                  _instruction_vector.push_back(code);
               }
               else if(op == "-=") {
                  code = instruction();
                  code.op = "sub_array_put_" + the_type;
                  _instruction_vector.push_back(code);
               }
               else if(op == "*=") {
                  code = instruction();
                  code.op = "mul_array_put_" + the_type;
                  _instruction_vector.push_back(code);
               }
               else if(op == "/=") {
                  code = instruction();
                  code.op = "div_array_put_" + the_type;
                  _instruction_vector.push_back(code);
               }
               else if(op == "%=") {
                  code = instruction();
                  code.op = "mod_array_put_" + the_type;
                  _instruction_vector.push_back(code);
               }
               else if(op == "|=") {
                  code = instruction();
                  code.op = "or_array_put_" + the_type;
                  _instruction_vector.push_back(code);
               }
               else if(op == "&=") {
                  code = instruction();
                  code.op = "and_array_put_" + the_type;
                  _instruction_vector.push_back(code);
               }
               else if(op == "<<=") {
                  code = instruction();
                  code.op = "lsh_array_put_" + the_type;
                  _instruction_vector.push_back(code);
               }
               else if(op == ">>=") {
                  code = instruction();
                  code.op = "rsh_array_put_" + the_type;
                  _instruction_vector.push_back(code);
               }
               else {
                  code = instruction();
                  code.op = "array_put";
                  _instruction_vector.push_back(code);
               }

               code = instruction();
               code.op = "add";
               code.arg1 = "esp";
               code.arg2 = "24";
               _instruction_vector.push_back(code);
  
               ret = "eax";
              
               _type_map[ret] = "int";
            }
            else {
               code = instruction();
               code.op = "push";
               code.arg1 = the_size;
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "push2";
               code.arg1 = id_str;
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "push";
               code.arg1 = size;
               _instruction_vector.push_back(code);

               string t = get_next_temp();

               code = instruction();
               code.op = "lea";
               code.arg1 = "edx";
               code.arg2 = t;
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "push";
               code.arg1 = "edx";
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "mov_dword";
               code.arg1 = "edx";
               code.arg2 = array_index; 
               _instruction_vector.push_back(code);

               for(size_t ii = 0; ii < temp_vec.size(); ii++) {
                  _instruction_vector.push_back(temp_vec[ii]);
               }

               code = instruction();
               code.op = "push_dword";
               code.arg1 = "edx";
               _instruction_vector.push_back(code);

               if(_type_map[id].find("*") == string::npos || is_array(id)) {
                  code = instruction();
                  code.op = "lea";
                  code.arg1 = "eax";
                  code.arg2 = id;
                  _instruction_vector.push_back(code);
               }
               else {
                  code = instruction();
                  code.op = "mov_dword";
                  code.arg1 = "eax";
                  code.arg2 = id;
                  _instruction_vector.push_back(code);
               }

               code = instruction();
               code.op = "push_dword";
               code.arg1 = "eax";
               _instruction_vector.push_back(code);

               if(_address_of || match(".", OPERATOR)) {
                  code = instruction();
                  code.op = "address_get";
                  _instruction_vector.push_back(code);
               }
               else {
                  code = instruction();
                  code.op = "array_get";
                  _instruction_vector.push_back(code);
               }

               code = instruction();
               code.op = "add";
               code.arg1 = "esp";
               code.arg2 = "24";
               _instruction_vector.push_back(code);

               ret = t;
              
               _type_map[ret] = "int";

               if(match(".", OPERATOR)) {
                  pointer = ret;

                  _token_vector[_offset].value = "->";

                  _type_map[pointer] = _type_map[id] + " *";

                  again = true;

                  continue;
               }

               if(match("->", OPERATOR)) {
                  pointer = ret;
                  //pointer = ret = "edx";

                  _type_map[pointer] = _type_map[id];

                  code = instruction();
                  code.op = "mov_dword";
                  code.arg1 = "eax";
                  code.arg2 = pointer;
                  _instruction_vector.push_back(code);

                  code = instruction();
                  code.op = "mov2";
                  code.arg1 = "edx";
                  code.arg2 = "eax";
                  _instruction_vector.push_back(code);


                  code = instruction();
                  code.op = "mov_dword";
                  code.arg1 = pointer;
                  code.arg2 = "edx";
                  _instruction_vector.push_back(code);

cout << "pointer: \"" << pointer << "\"" << endl
     << "type: \"" << _type_map[pointer] << "\"" << endl << endl << flush;
//cin.get();cin.get();

                  again = true;

                  continue;
               }

               if(match("(", OPERATOR)) {
                  code = instruction();
                  code.op = "mov_dword";
                  code.arg1 = "eax";
                  code.arg2 = ret;
                  _instruction_vector.push_back(code);
                  
                  pointer = get_next_temp();
      
                  code = instruction();
                  code.op = "mov_dword";
                  code.arg1 = pointer;
                  code.arg2 = "eax";
                  _instruction_vector.push_back(code);

                  _type_map[pointer] = "int";

                  again = true;

                  continue;
               }

               if(match("[", OPERATOR)) {
                  pointer = id_str;

                  _type_map[pointer] = "int";

                  variable_rep var(this);

                  var.label = pointer;

                  var.type = "int";

                  _symbol_table[pointer] = var;

                  again = true;

                  continue;
               }

               pointer = "";
            }
         }
         else {
            error("expected \"]\"");
         }
      }
      else if(match(".", OPERATOR)) {
         _offset++;

         string t,t2;
         string id2;

         t = _type_map[id];
         ret = id;
         string the_id;
         string last;

         while(_token_vector[_offset].type == ID) {
            id2 = _token_vector[_offset].value;
            the_id = id2;
            struct_rep * p_struct = get_struct(t);

            if(p_struct) {
               t2 = p_struct->get_type(id2);
            }
            else {
               t2 = _type_map[id2];
            }

            _offset++;

            ret += "+" + to_struct(t) + "-" + to_struct(t) + "_" + id2;
            last = to_struct(t);

            if(get_struct(t2)) {
               //ret += "-" + to_struct(t2);
//DEBUG 11
               //ret += "+" + to_struct(t2) + "-" + to_struct(t2) + "_" + id2;
//DEBUG 11
            }

            t = t2;

            if(match(".", OPERATOR)) {
               _offset++;
            }
            else if(match("(", OPERATOR)) {
               break;
            }
            else if(match("[", OPERATOR)) {
               break;
            }
            else {
               break;
            }
         }

         _type_map[ret] = t;

         if(match("->", OPERATOR)) {
            pointer = get_next_temp();

            _type_map[pointer] = _type_map[ret];

            again = true;

            continue;
         }

         if(match("(", OPERATOR)) {
            code = instruction();
            code.op = "mov_dword";
            code.arg1 = "eax";
            code.arg2 = ret;
            _instruction_vector.push_back(code);

            pointer = get_next_temp();

            code = instruction();
            code.op = "mov_dword";
            code.arg1 = pointer;
            code.arg2 = "eax";
            _instruction_vector.push_back(code);

            _type_map[pointer] = _type_map[ret];

            again = true;

            continue;
         }
         
         if(match("[", OPERATOR)) {
            pointer = ret;// + "+" + last;

            variable_rep var(this);

            var.label = pointer;

            var.type = t;

            _type_map[pointer] = t;

            _symbol_table[pointer] = var;

            again = true;

            continue;
         }

         pointer = "";

         if(is_assign(_token_vector[_offset].value)) {
            string op = _token_vector[_offset].value;

            _offset++;

            string tt = expression();

            string type = _type_map[tt];

            if(is_type(type, "int") && is_type(type, "unsigned")) {
               assign_code(op, "int", ret, tt);
            }
            else if(is_type(type, "int")) {
               assign_code(op, "int", ret, tt);
            }
            else if(is_type(type, "float")) {
               assign_code(op, "float", ret, tt);
            }
            else if(is_type(type, "short") && is_type(type, "unsigned")) {
               assign_code(op, "ushort", ret, tt);
            }
            else if(is_type(type, "short")) {
               assign_code(op, "short", ret, tt);
            }
            else if(is_type(type, "char") && is_type(type, "unsigned")) {
               assign_code(op, "uchar", ret, tt);
            }
            else if(is_type(type, "char")) {
               assign_code(op, "char", ret, tt);
            }
            else if(is_type(type, "long") && is_type(type, "unsigned")) {
               assign_code(op, "ulong", ret, tt);
            }
            else if(is_type(type, "long")) {
               assign_code(op, "long", ret, tt);
            }
            else if(is_type(type, "double")) {
               assign_code(op, "double", ret, tt);
            }
            else if(is_type(type, "struct")) {
cout << "type: \"" << type << "\"" << endl << flush;
//cin.get(),cin.get();
               assign_code(op, type, ret, tt);
            }
            else {
cout << "type: \"" << type << "\"" << endl << flush;
//cin.get(),cin.get();
               assign_code(op, "int", ret, tt);
            }
         }
      }
      else if(match("->", OPERATOR)) {
         string id2,temp_id = id,the_id;
         struct_rep * p_struct;
         variable_rep * p_variable;
         string t, t2;

         while(match("->", OPERATOR)) {
            _offset++;

            if(_token_vector[_offset].type == ID) {
               id2 = _token_vector[_offset].value;
               _offset++;

               p_struct = get_struct(_type_map[id]);

               if(p_struct) {
                  p_variable = p_struct->get_variable(id2);
               }
               else {
                  p_variable = 0;
               }

               if(p_variable != 0) {
                  if(get_struct(p_variable->type)) {
                     the_id = ret = to_struct(p_variable->type) + "_" + id2;
                  }
                  else {
                     the_id = ret = to_struct(_type_map[id]) + "_" + id2;
                  }
               }
               else {
                  the_id = ret = to_struct(_type_map[id]) + "_" + id2;
               }

               t = to_struct(_type_map[id]);
               //t = _type_map[id];

               if(p_variable) {
                  t2 = p_variable->type;
               }
               else {
                  t2 = ""; 
               }

               if(match(".", OPERATOR)) {
                  break;
               }

               if(match("[", OPERATOR)) {
                  break;
               }

               //if(match("->", OPERATOR)) {
               //   break;
               //}

               if(temp_id == "edx") {
                  code = instruction();
                  code.op = "mov_dword";
                  code.arg1 = "eax";
                  code.arg2 = temp_id;// + "+" + to_struct(t2) + "-" + to_struct(t2) + "_" + id2; 
                  _instruction_vector.push_back(code);

                  code = instruction();
                  code.op = "mov2";
                  code.arg1 = "edx";
                  code.arg2 = "eax";
                  _instruction_vector.push_back(code);
               }
               else {
/*
                  if(is_param(temp_id)) { 
                     code = instruction();
                     code.op = "lea";
                     code.arg1 = "eax";
                     code.arg2 = temp_id;// + "+" + to_struct(t) + "-" + to_struct(t) + "_" + id2; 
                     _instruction_vector.push_back(code);

                     code = instruction();
                     code.op = "add";
                     code.arg1 = "eax";
                     code.arg2 = DATA_WORD_SIZE_STR; 
                     _instruction_vector.push_back(code);

                     code = instruction();
                     code.op = "mov2";
                     code.arg1 = "edx";
                     code.arg2 = "eax"; 
                     _instruction_vector.push_back(code);
                  }
                  else {
*/
                     code = instruction();
                     code.op = "mov_dword";
                     code.arg1 = "edx";
                     code.arg2 = temp_id;// + "+" + to_struct(t) + "-" + to_struct(t) + "_" + id2; 
                     _instruction_vector.push_back(code);
/*
                  }
*/
               }

/*
               code = instruction();
               code.op = "mov2";
               code.arg1 = "edx";
               code.arg2 = "eax";
               _instruction_vector.push_back(code);
*/

               if(t != "") {
                  code = instruction();
                  code.op = "add2";
                  code.arg1 = "edx";
                  //code.arg2 = "-" + to_struct(t) + "+" + to_struct(t) + "-" + to_struct(t) + "_" + id2;
                  code.arg2 = to_struct(t) + "-" + to_struct(t) + "_" + id2;
                  _instruction_vector.push_back(code);
               }
               else if(t2 != "") {
                  code = instruction();
                  code.op = "add2";
                  code.arg1 = "edx";
                  //code.arg2 = "-" + to_struct(t2) + "+" + to_struct(t2) + "-" + to_struct(t2) + "_" + id2;
                  code.arg2 = to_struct(t2) + "-" + to_struct(t2) + "_" + id2;
                  _instruction_vector.push_back(code);
               }

               if(!is_assign(_token_vector[_offset].value)) {
                  code = instruction();
                  code.op = "mov2";
                  code.arg1 = "eax";
                  code.arg2 = "edx";
                  _instruction_vector.push_back(code);

                  ret = "eax";

                  _type_map[ret] = t2;
               }

               temp_id = "edx";

/*
               if(match("->", OPERATOR)) {
                  struct_rep * p_struct = get_struct(t);
                  if(p_struct) {
                     variable_rep * p_variable = p_struct->get_variable(id2);
                     if(p_variable) {
                        t2 = p_variable->type;
                     }
                  }

                  temp_id = id = get_next_temp();
 
                  if(ret == "eax") {
                     code = instruction();
                     code.op = "mov_dword";
                     code.arg1 = id;
                     code.arg2 = "eax";
                     _instruction_vector.push_back(code);

                     if(t2 != "")
                        _type_map[id] = t2;
                     else
                        _type_map[id] = t;
                  }
                  else {
                     code = instruction();
                     code.op = "mov_dword";
                     code.arg1 = id;
                     code.arg2 = "edx";
                     _instruction_vector.push_back(code);
         
                     if(t2 != "") 
                        _type_map[id] = t2;
                     else 
                        _type_map[id] = t;
                  }
               }
*/
            }
            else {
               break;
            }
         }

         if(match("->", OPERATOR)) {
            pointer = get_next_temp();

            code = instruction(); 
            code.op = "mov_dword";
            code.arg1 = "eax";
            code.arg2 = id;
            _instruction_vector.push_back(code);

            code = instruction(); 
            code.op = "add2";
            code.arg1 = "eax";
            code.arg2 = to_struct(t) + "-" + to_struct(t) + "_" + id2;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "mov_dword";
            code.arg1 = pointer;
            code.arg2 = "eax";
            _instruction_vector.push_back(code);

            struct_rep * p_struct = get_struct(t);
            if(p_struct) {
               variable_rep * p_variable = p_struct->get_variable(id2);
               if(p_variable) {
                  t2 = p_variable->type;
               }
            }

            if(t2 != "")
               _type_map[pointer] = t2 + " *";
            else if(t != "")
               _type_map[pointer] = t + " *";

            again = true;

            continue;
         }

         if(match(".", OPERATOR)) {
            pointer = get_next_temp();

            code = instruction();
            code.op = "mov_dword";
            code.arg1 = "eax";
            code.arg2 = id;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "add2";
            code.arg1 = "eax";
            code.arg2 = to_struct(t) + "-" + to_struct(t) + "_" + id2;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "mov_dword";
            code.arg1 = pointer;
            code.arg2 = "eax";
            _instruction_vector.push_back(code);

            _token_vector[_offset].value = "->";

cout << "pointer == " << pointer << endl << flush;
cout << "type == " << t << endl << flush;
cout << "type2 == " << t2 << endl << flush;
//cin.get();
//cin.get();

            struct_rep * p_struct = get_struct(t);
            if(p_struct) {
               variable_rep * p_variable = p_struct->get_variable(id2);
               if(p_variable) {
                  t2 = p_variable->type;
               }
            }

            if(t2 != "") 
               _type_map[pointer] = t2 + " *";
            else if(t != "") 
               _type_map[pointer] = t + " *";

            again = true;

            continue;
         }

         if(match("(", OPERATOR)) {
            if(temp_id == "edx") {
               code = instruction();
               code.op = "mov2";
               code.arg1 = "eax";
               code.arg2 = temp_id;
               _instruction_vector.push_back(code);
            }
            else {
               code = instruction();
               code.op = "mov2";
               code.arg1 = "eax";
               code.arg2 = ret;
               _instruction_vector.push_back(code);
            }

            pointer = get_next_temp();

            code = instruction();
            code.op = "mov_dword";
            code.arg1 = pointer;
            code.arg2 = "eax";
            _instruction_vector.push_back(code);
 
            _type_map[pointer] = "int";

            again = true;

            continue;
         }

         if(match("[", OPERATOR)) {
            // mov eax,[ebp+p]
            // add eax,Node_array+Node
            // mov [ebp+pointer],eax
            //
            //pointer = id + "+" + the_id + "+" + to_struct(_type_map[id]);
       
/************/
            code = instruction();
            code.op = "mov_dword";
            code.arg1 = "eax";
            code.arg2 = id;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "add2";
            code.arg1 = "eax";
            code.arg2 = to_struct(_type_map[id]) + "-" + the_id;// + "+" + to_struct(_type_map[id]);
            _instruction_vector.push_back(code);

            pointer = get_next_temp();

            code = instruction();
            code.op = "mov_dword";
            code.arg1 = pointer;
            code.arg2 = "eax";
            _instruction_vector.push_back(code);
/************/

            variable_rep var(this);

            var.label = pointer;
 
            var.type = "int*";

            var.pointer_count = 1;

            _symbol_table[var.label] = var;

            _type_map[pointer] = "int*";

            again = true;

            continue;
         }

         if(is_assign(_token_vector[_offset].value)) {
            string op = _token_vector[_offset].value;

            _offset++;

            ret = "edx";

            _type_map[ret] = _type_map[id2];

            string tt = expression();

            string type = _type_map[tt];

            if(is_type(type, "int") && is_type(type, "unsigned")) {
               assign_code(op, "int", ret, tt, true);
            }
            else if(is_type(type, "int")) {
               assign_code(op, "int", ret, tt, true);
            }
            else if(is_type(type, "float")) {
               assign_code(op, "float", ret, tt, true);
            }
            else if(is_type(type, "short") && is_type(type, "unsigned")) {
               assign_code(op, "ushort", ret, tt, true);
            }
            else if(is_type(type, "short")) {
               assign_code(op, "short", ret, tt, true);
            }
            else if(is_type(type, "char") && is_type(type, "unsigned")) {
               assign_code(op, "uchar", ret, tt, true);
            }
            else if(is_type(type, "char")) {
               assign_code(op, "char", ret, tt, true);
            }
            else if(is_type(type, "long") && is_type(type, "unsigned")) {
               assign_code(op, "ulong", ret, tt, true);
            }
            else if(is_type(type, "long")) {
               assign_code(op, "long", ret, tt, true);
            }
            else if(is_type(type, "double")) {
               assign_code(op, "double", ret, tt, true);
            }
            else if(is_type(type, "struct")) {
               assign_code(op, type, ret, tt, true);
            }
            else {
               assign_code(op, "int", ret, tt, true);
            }
         }
      }
      else if(match("(", OPERATOR)) {
         _offset++;
         size_t param_index = _instruction_vector.size();

         if(match(")", OPERATOR)) {
            _offset++;

            count = 0;
         }
         else {
             _current_function.name = id;

             function_rep * p_function = get_function(id);

             if(p_function) 
                _current_function = (*p_function);
             else {
                if(_type_map[id].find("(*)") == string::npos) {
                   error((string("unknown function: \"") + id + string("\" found type: \"") + _type_map[id] + string("\"")).c_str());
                }
             }

             _in_call_list = true;             

             param_index = _instruction_vector.size();

             count = expression_call_list();

             _in_call_list = false;

             if(!match(")", OPERATOR)) {
                error("exptected: \")\"...");
             }
             else {
                _offset++;
             }
         }

         if(match(".", OPERATOR)) {
            pointer = get_next_temp();

            code = instruction();
            code.op = "mov_dword"; 
            code.arg1 = pointer;
            code.arg2 = "eax";
            _instruction_vector.push_back(code);

            _type_map[pointer] = "int";

            again = true;

            continue;
         }

         if(match("->", OPERATOR)) {
            pointer = get_next_temp();

            code = instruction();
            code.op = "mov_dword"; 
            code.arg1 = pointer;
            code.arg2 = "eax";
            _instruction_vector.push_back(code);

            _type_map[pointer] = "int";

            again = true;

            continue;
         }

         if(match("[", OPERATOR)) {
            pointer = get_next_temp();

            code = instruction();
            code.op = "mov_dword"; 
            code.arg1 = pointer;
            code.arg2 = "eax";
            _instruction_vector.push_back(code);

            _type_map[pointer] = "int";

            again = true;

            continue;
         }

         if(match("(", OPERATOR)) {
            pointer = get_next_temp();

            code = instruction();
            code.op = "mov_dword"; 
            code.arg1 = pointer;
            code.arg2 = "eax";
            _instruction_vector.push_back(code);

            _type_map[pointer] = "int";

            again = true;

            continue;
         }
 
         ret =  "eax";
         
        if(pointer.size()) {
           if(is_type(_current_function.return_type, "struct")
              && _current_function.return_type.find("*") == string::npos) {
              struct_rep * p_struct = get_struct(_current_function.return_type);
              if(p_struct) {
                 string temp = get_next_temp(_current_function.return_type, p_struct->get_size());

                 code = instruction();
                 code.op = "lea";
                 code.arg1 = "edx";
                 code.arg2 = temp;

                 if(param_index == _instruction_vector.size())
                    _instruction_vector.push_back(code);
                 else {
                    instruction_vector_type::iterator p = _instruction_vector.begin();

                    for(size_t i = 0; i < param_index; i++)
                       p++;

                    _instruction_vector.insert(p, code);
                 }
                 code = instruction();
                 code.op = "push";
                 code.arg1 = "edx";
 
                 if(param_index == _instruction_vector.size())
                    _instruction_vector.push_back(code);
                 else {
                    instruction_vector_type::iterator p = _instruction_vector.begin();

                    for(size_t i = 0; i <= param_index; i++)
                       p++;

                    _instruction_vector.insert(p, code);
                 }

                 ret = temp;

                 _type_map[ret] = _current_function.return_type + "*";

                 count += 4;
              }
           }


           code = instruction();
           code.op = "call2";
           code.arg1 = id;
           _instruction_vector.push_back(code); 
        }
        else {
           if(is_type(_current_function.return_type, "struct")
              && _current_function.return_type.find("*") == string::npos) {
              struct_rep * p_struct = get_struct(_current_function.return_type);
              if(p_struct) {
                 string temp = get_next_temp(_current_function.return_type, p_struct->get_size());

                 code = instruction();
                 code.op = "lea";
                 code.arg1 = "edx";
                 code.arg2 = temp;

                 if(param_index == _instruction_vector.size())
                    _instruction_vector.push_back(code);
                 else {
                    instruction_vector_type::iterator p = _instruction_vector.begin();

                    for(size_t i = 0; i < param_index; i++)
                       p++;

                    _instruction_vector.insert(p, code);
                 }

                 code = instruction();
                 code.op = "push";
                 code.arg1 = "edx";

                 if(param_index == _instruction_vector.size())
                    _instruction_vector.push_back(code);
                 else {
                    instruction_vector_type::iterator p = _instruction_vector.begin();

                    for(size_t i = 0; i <= param_index; i++)
                       p++;

                    _instruction_vector.insert(p, code);
                 }

                 ret = temp;

                 _type_map[ret] = _current_function.return_type + "*";

                 count += 4;
              }
           }

           code = instruction();
           code.op = "call";
           code.arg1 = id;
           _instruction_vector.push_back(code); 
        }

        code = instruction();
        code.op = "add";
        code.arg1 = "esp";
        code.arg2 = to_string(count);
        _instruction_vector.push_back(code);

         //ret =  "eax";

         int index = -1;

         if(is_function(id, index)) {
            if(is_type(_function_vector[index].return_type, "struct")) {
               _type_map[ret] = _function_vector[index].return_type + "*";
            }
            else {
               _type_map[ret] = _function_vector[index].return_type;
            }
         }
      }
      else if(_no_assign) {
         ret = id;
         return ret;
      }
      else if(is_assign(_token_vector[_offset].value)) {
         string op = _token_vector[_offset].value;

         _offset++;

         string arg = expression();

         ret = id;

cout << "\"" << ret << "\" == \"" << arg << "\"" << endl << flush;
cout << "is_char(\"" << ret << "\"): \"" << is_char(ret) << "\"" << endl << flush;
cout << "is_const(\"" << arg << "\"): \"" << is_const(arg) << "\"" << endl << flush;

         if(is_const(arg)) {
            if(is_char(ret) && _type_map[ret].find("*") != string::npos) {
               assign_code(op, "const char *", ret, arg);
            }
            else if(is_char(ret)) {
cout << "is_const and is_char" << endl << flush;
               code = instruction();
               code.op = "mov_byte";
               code.arg1 = "al";
               code.arg2 = arg;
               _instruction_vector.push_back(code);

               assign_code(op, "char", ret, "al");
            }
            else if(is_float(ret)) {
               code = instruction();
               code.op = "fld";
               code.arg1 = arg;
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "fstp_dword";
               code.arg1 = ret;
               _instruction_vector.push_back(code);
            }
            else if(is_double(ret)) {
               code = instruction();
               code.op = "fld";
               code.arg1 = arg;
               _instruction_vector.push_back(code);

               code = instruction();
               code.op = "fstp";
               code.arg1 = ret;
               _instruction_vector.push_back(code);
            }
            else if(is_short(ret)) {
               assign_code(op, "short", ret, arg);
            }
            else if(is_int(ret)) {
               assign_code(op, "int", ret, arg);
            }
            else if(is_long(ret)) {
               code = instruction();
               code.op = "mov_qword";
               code.arg1 = ret;
               code.arg2 = arg;
               _instruction_vector.push_back(code);
            }
            else {
               assign_code(op, "int", ret, arg);
            }
         } 
         else if(is_double(arg) && is_double(ret)) {
            code = instruction();
            code.op = "fld";
            code.arg1 = arg;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "fstp";
            code.arg1 = ret;
            _instruction_vector.push_back(code);
         }
         else if(is_char(arg) && is_char(ret)) {
            code = instruction();
            code.op = "mov_byte";
            code.arg1 = "dx";
            code.arg2 = arg;
            _instruction_vector.push_back(code);

            assign_code(op, "char", ret, "dx");
         }
         else if(is_short(arg) && is_short(ret)) {
            code = instruction();
            code.op = "mov_word";
            code.arg1 = "dx";
            code.arg2 = arg;
            _instruction_vector.push_back(code);

            assign_code(op, "short", ret, "dx");
         }
         else if(is_int(arg) && is_short(ret)) {
            code = instruction();
            code.op = "mov_word";
            code.arg1 = "dx";
            code.arg2 = arg;
            _instruction_vector.push_back(code);

            code = instruction();
            code.op = "mov_word";
            code.arg1 = "dx";
            code.arg2 = "0";
            _instruction_vector.push_back(code);

            assign_code(op, "int", ret, "al");
         }
         else if(is_long(arg) && is_long(ret)) {
            assign_code(op, "long", ret, arg);
         }
         else if(is_int(arg) && is_int(ret)) {
            assign_code(op, "int", ret, arg);
         }
         else if(is_type(_type_map[arg], "struct") 
                 && is_type(_type_map[ret], "struct")
                 && to_struct(_type_map[arg]) == to_struct(_type_map[ret])) {
cout << "to_struct(\"" << _type_map[ret] << "\",\"" << _type_map[arg] << "\")" << endl << flush;
//cin.get(),cin.get();
            assign_code(op, _type_map[arg], ret, arg);
         }
         else if(is_int(arg) && is_long(ret)) {
            assign_code(op, "int", ret, arg);

            code = instruction();
            code.op = "mov";
            code.arg1 = ret+"+4";
            code.arg2 = "0";
            _instruction_vector.push_back(code);
         }
         else {
            assign_code(op, "int", ret, arg);
         }
      }
      else if(match("++", OPERATOR)) {
         _offset++;

         code = instruction();
         code.op = "push_dword";
         code.arg1 = id;
         code = instruction();
         _instruction_vector.push_back(code);

         code.op = "mov_dword";
         code.arg1 = "edx";
         code.arg2 = id;
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "inc";
         code.arg1 = "edx";
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "mov_dword";
         code.arg1 = id;
         code.arg2 = "edx";
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "pop";
         code.arg1 = "edx";
         code = instruction();
         _instruction_vector.push_back(code);

         ret = "edx";

         _type_map[ret] = _type_map[id];
      }
      else if(match("--", OPERATOR)) {
         _offset++;

         code = instruction();
         code.op = "push";
         code.arg1 = id;
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "mov_dword";
         code.arg1 = "edx";
         code.arg2 = id;
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "inc";
         code.arg1 = "edx";
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "mov_dword";
         code.arg1 = id;
         code.arg2 = "edx";
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "pop";
         code.arg1 = "edx";
         _instruction_vector.push_back(code);

         ret = "edx";

         _type_map[ret] = _type_map[id];
      }
   }
   else if(match("~", OPERATOR)) {
      _offset++;

      ret = expression();

      instruction code;

      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "edx";
      code.arg2 = ret;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "notl";
      code.arg1 = "edx";
      _instruction_vector.push_back(code);

      ret = "edx";

      _type_map[ret] = "int";
   }
   else if(_token_vector[_offset].type == LONG_VALUE) {
      ret = _token_vector[_offset].value;

      _offset++;

      variable_rep var(this);
   
      var.label = get_next_const();
   
      var.type = "long";
     
      var.default_value = ret;

      if(_current_function.name != "") 
         //_current_function.variable_vector.push_back(var);
         _variable_vector.push_back(var);
      else 
         _variable_vector.push_back(var);
  
      _symbol_table[var.label] = var;

      _type_map[var.label] = "long";

      ret = var.label;
   }
   else if(_token_vector[_offset].type == INT_VALUE) {
      ret = _token_vector[_offset].value;

      _type_map[ret] = "int";

      _offset++;
   }
   else if(_token_vector[_offset].type == CHAR_VALUE) {
      ret = _token_vector[_offset].value;

cout << "CHAR[\"" << ret << "\"]" << endl << flush;

      _type_map[ret] = "char";

      _offset++;
   }
   else if(_token_vector[_offset].type == DOUBLE_VALUE) {
      ret = _token_vector[_offset].value;

      _type_map[ret] = "double";

      _offset++;

      variable_rep var(this);

      var.label = get_next_const();
    
      var.type = "double";
      
      var.default_value = ret;

      if(_current_function.name != "") 
         //_current_function.variable_vector.push_back(var);
         _variable_vector.push_back(var);
      else
         _variable_vector.push_back(var);
   
      _symbol_table[var.label] = var;

      ret = var.label;
   }
   else if(_token_vector[_offset].type == FLOAT_VALUE) {
      ret = to_hex(atof(_token_vector[_offset].value.c_str()));

      _type_map[ret] = "float";

      _offset++;
   }
   else if(_token_vector[_offset].type == STRING_VALUE) {
      variable_rep var(this);

      ret = var.label = get_next_const();

      var.type = "const char *";
 
      var.default_value = to_code_string(_token_vector[_offset].value);

      if(_current_function.name != "") 
         //_current_function.variable_vector.push_back(var);
         _variable_vector.push_back(var);
      else
         _variable_vector.push_back(var);

      _symbol_table[var.label] = var;

      _type_map[ret] = "const char *";

      _offset++;
   }
   else if(match("!", OPERATOR)) {
      _offset++;
      string str = expression();

      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "edx";
      code.arg2 = str;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "not";
      code.arg1 = "edx";
      _instruction_vector.push_back(code);

      ret = "edx";

      _type_map[ret] = "int";
   }
   else if(match("*", OPERATOR)) {
      _offset++;

      _no_assign = true;
      string id = primary();
      _no_assign = false;

      if(match("=", OPERATOR)) {
         _offset++;

         string arg = expression();

         code = instruction();
         code.op = "l_val";
         code.arg1 = id;
         code.arg2 = arg;
         _instruction_vector.push_back(code);

         ret = id;
      }
      else {
         code = instruction();
         code.op = "r_val";
         code.arg1 = "edx";
         code.arg2 = id;
         _instruction_vector.push_back(code);

         ret = "edx";
      }

      _type_map[ret] = "int";
   }
   else if(match("&", OPERATOR)) {
      _offset++;
      string arg;
      bool old_address_of = _address_of;
      _address_of = true;
      arg = primary();
      _address_of = old_address_of;

      string t = _type_map[arg];

cout << "address_of: " << arg << endl
     << "type: " << t << endl << endl << flush;
//cin.get();
//cin.get();

      code = instruction();
      code.op = "lea";
      code.arg1 = "edx";
      code.arg2 = arg;
      _instruction_vector.push_back(code);

      ret = "edx";

      _type_map[ret] = _type_map[arg] + "*";
   }
   else if(match("-", OPERATOR)) {

cout << endl << flush;
cout << "MATCH NEGATIVE" << endl << flush;
cout << endl << flush;

      _offset++;

      string arg;

      arg = expression();

cout << "str \"" << arg << "\"" << endl << flush;

      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "edx";
      code.arg2 = arg;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "neg";
      code.arg1 = "edx";
      _instruction_vector.push_back(code);

      ret = "edx";

/**/
      if(is_float(arg)) {
         _type_map[ret] = "float";
      }
      else if(is_double(arg)) {
         _type_map[ret] = "double";
      }
      else if(is_long(arg)) {
         _type_map[ret] = "long";
      }
      else if(is_char(arg)) {
         _type_map[ret] = "char";
      }
      else if(is_int(arg)) {
         _type_map[ret] = "int";
      }
/**/
   }
   else {
cout << "_offset == " << _offset << endl << flush;
      //_offset++;
      ret = "";
   }

   } // end while(again)

   return ret;
}

bool
c_compiler::is_const(const string & str) 
{
   if(str.size()) {
      if(str[0] == 'C') {
//cout << "is_const(\"" << str << "\"): true" << endl << flush;
         return true;
      }
      else if(str[0] == '\'') {
//cout << "is_const(\"" << str << "\"): true" << endl << flush;
         return true;
      }
      else if(str[0] == '\"') {
//cout << "is_const(\"" << str << "\"): true" << endl << flush;
         return true;
      }
      else if(is_digit(str[0])) {
//cout << "is_const(\"" << str << "\"): true" << endl << flush;
         return true;
      }
      else {
//cout << "is_const(\"" << str << "\"): false" << endl << flush;
         return false;
      }
   }

cout << "is_const(\"" << str << "\"): false" << endl << flush;
  
   return false;
}

string
c_compiler::bit_math()
{
   string arg1, arg2, op, temp;
   instruction code;

   arg1 = primary();

   string ret = arg1;

   if(match("<<", OPERATOR)) {
      op = "shl";
      _offset++;
   }
   else if(match(">>", OPERATOR)) {
      op = "shr";
      _offset++;
   }
   else if(match("&", OPERATOR)) {
      op = "and";
      _offset++;
   }
   else if(match("|", OPERATOR)) {
      op = "or";
      _offset++;
   }
   else if(match("^", OPERATOR)) {
      op = "xor";
      _offset++;
   }
   else {
      return arg1;
   }

   arg2 = bit_math();

   if(arg2.size()) {
      instruction code;

      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "edx";
      code.arg2 = arg1;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = op;
      code.arg1 = "edx";
      code.arg2 = arg2;
      _instruction_vector.push_back(code);

      ret = "edx";
   }

   _type_map[ret] = "int";

   return ret;
}

string
c_compiler::factor()
{
   string arg1, arg2, op, temp;
   instruction code;

   arg1 = bit_math();
  
   if(arg1 == "")
      return "";

   if(match("*", OPERATOR)) {
      if(is_long(arg1)) {
         op = "long_mul";
      }
      else if(is_double(arg1)) {
         op = "double_mul";
      }
      else if(is_float(arg1)) {
         op = "float_mul";
      }
      else if(is_int(arg1)) {
         op = "int_mul";
      }
      else if(is_short(arg1)) {
         op = "short_mul";
      }
      else if(is_char(arg1)) {
         op = "char_mul";
      }
      else {
         return arg1;
      }
   }
   else if(match("/", OPERATOR)) {
      if(is_long(arg1)) {
         op = "long_div";
      }
      else if(is_double(arg1)) {
         op = "double_div";
      }
      else if(is_float(arg1)) {
         op = "float_div";
      }
      else if(is_int(arg1)) {
         op = "int_div";
      }
      else if(is_short(arg1)) {
         op = "short_div";
      }
      else if(is_char(arg1)) {
         op = "char_div";
      }
      else {
         return arg1;
      }
   }
   else {
      return arg1;
   }

   _offset++;

   if((arg2 = factor()) != "") {
      if(op == "float_div") {
         code = instruction();
         code.op = "float_div";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         arg1 = "eax";
         _type_map[arg1] = "float";
      }
      else if(op == "double_div") {
         code = instruction();
         code.op = "double_div";
         code.arg1 = arg1;
         code.arg2 = arg2;
         code.arg3 = get_next_temp(8);
         _instruction_vector.push_back(code);
         arg1 = code.arg3;
         _type_map[arg1] = "double";
      }
      else if(op == "int_div") {
         code = instruction();
         code.op = "int_div";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         arg1 = "eax";
         _type_map[arg1] = "int";
      }
      else if(op == "long_div") {
         code = instruction();
         code.op = "long_div";
         code.arg1 = arg1;
         code.arg2 = arg2;
         code.arg3 = get_next_temp(8);
         _instruction_vector.push_back(code);
         arg1 = code.arg3;
         _type_map[arg1] = "long";
      }
      else if(op == "long_mul") {
         code = instruction();
         code.op = "long_mul";  
         code.arg1 = arg1;
         code.arg2 = arg2;
         code.arg3 = get_next_temp(8);
         _instruction_vector.push_back(code);
         arg1 = code.arg3;
         _type_map[arg1] = "long";
      }
      else if(op == "float_mul") {
         code = instruction();
         code.op = "float_mul";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         arg1 = "eax";
         _type_map[arg1] = "float";
      }
      else if(op == "double_mul") {
         code = instruction();
         code.op = "double_mul";
         code.arg1 = arg1;
         code.arg2 = arg2;
         code.arg3 = get_next_temp(8);
         _instruction_vector.push_back(code);
         arg1 = code.arg3;
         _type_map[arg1] = "double";
      }
      else if(op == "int_mul") {
         code = instruction();
         code.op = "int_mul";
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);
         arg1 = "eax";
         _type_map[arg1] = "int";
      }
   }

   return arg1;
}

bool
c_compiler::is_long(const string & ch)
{
   bool ret = false;
   string_map_type::iterator ptr = _type_map.find(ch);
   symbol_table_type::iterator ptr2 = _symbol_table.find(ch);

   if(ptr2 != _symbol_table.end()) {
cout << "found1" << endl << flush;
cout << "value: \"" << (ptr2->second.type) << "\"" << endl << flush;
      if(is_type(ptr2->second.type, "long")
         && !is_type(ptr2->second.type, "*")) {
         ret = true;
      }     
   }
   if(ptr != _type_map.end()) {
cout << "found2" << endl << flush;
      if(is_type(ptr->second, "long")
         && !is_type(ptr->second, "*")) {
         ret = true;
      }     
   }
   for(size_t i = 0; i < _variable_vector.size(); i++) {
      if(_variable_vector[i].label == ch) {
cout << "found3" << endl << flush;
         if(is_type(_variable_vector[i].type, "long")
            && !is_type(_variable_vector[i].type, "*")) {
            ret = true;
            break;
         }
      }
   }
  
   return ret;
}

bool
c_compiler::is_short(const string & ch)
{
   bool ret = false;
   string_map_type::iterator ptr = _type_map.find(ch);
   symbol_table_type::iterator ptr2 = _symbol_table.find(ch);

   if(ptr2 != _symbol_table.end()) {
cout << "found1" << endl << flush;
cout << "value: \"" << (ptr2->second.type) << "\"" << endl << flush;
      if(is_type(ptr2->second.type, "short")
         && !is_type(ptr2->second.type, "*")) {
         ret = true;
      }
   }     
   else if(ptr != _type_map.end()) {
cout << "found2" << endl << flush;
      if(is_type(ptr->second, "short")
         && !is_type(ptr->second, "*")) {
         ret = true;
      }
   }
   else {
      for(size_t i = 0; i < _variable_vector.size(); i++) {
         if(_variable_vector[i].label == ch) {
cout << "found3" << endl << flush;
            if(is_type(_variable_vector[i].type, "short")
               && !is_type(_variable_vector[i].type, "*")) {
               ret = true;
               break;
            }
         }
      }
   }

   return ret;
}


bool
c_compiler::is_char(const string & ch)
{
   bool ret = false;
   string_map_type::iterator ptr = _type_map.find(ch);
   symbol_table_type::iterator ptr2 = _symbol_table.find(ch);

   if(ch.size()) {
      if(ch[0] == '\'') {
cout << "found2" << endl << flush;
         ret = true;
      }
   }
 
   if(ptr2 != _symbol_table.end()) {
cout << "found1" << endl << flush;
cout << "value: \"" << (ptr2->second.type) << "\"" << endl << flush;
      if(is_type(ptr2->second.type, "char")
         && !is_type(ptr2->second.type, "*")) {
         ret = true;
      }
   }
   else if(ptr != _type_map.end()) {
cout << "found2" << endl << flush;
      if(is_type(ptr->second, "char")
         && !is_type(ptr->second, "*")) {
         ret = true;
      }
   }
   else {
      for(size_t i = 0; i < _variable_vector.size(); i++) {
         if(_variable_vector[i].label == ch) {
cout << "found3" << endl << flush;
            if(is_type(_variable_vector[i].type, "char")
               && !is_type(_variable_vector[i].type, "*")) {
               ret = true;
               break;
            }
         }
      }
   }

   return ret;
}

bool
c_compiler::is_double(const string & str)
{
   bool ret = false;

   symbol_table_type::iterator p;
   if((p = _symbol_table.find(str)) != _symbol_table.end()) {
      if(is_type(p->second.type, "double")) {
         ret = true;
      }
   }

   string str2 = _type_map[str];

   if(is_type(str2, "double")) {
      ret = true;
   }

   return ret;
}

bool
c_compiler::is_float(const string & str)
{
   bool ret = false;
   symbol_table_type::iterator p;

   if((p = _symbol_table.find(str)) != _symbol_table.end()) {
      if(is_type(p->second.type, "float")) {
         ret = true;
      }
   }

   if(is_const(str) && isdigit(str[0])) {
      if(str.find('.') != string::npos) {
         ret = true;
      }
   }

   string str2 = _type_map[str];

   if(is_type(str2, "float")) {
      ret = true;
   }

/*
if(ret) {
cout << "is_float(\"" << str << "\")" << endl << flush;
}
*/

   return ret;
}

bool
c_compiler::is_int(const string & str)
{
   bool ret = false;

   if(is_const(str) && isdigit(str[0]) && str.find(".") == string::npos) {
      ret = true;
   }

   symbol_table_type::iterator p;
   if((p = _symbol_table.find(str)) != _symbol_table.end()) {
      if(is_type(p->second.type, "int")) {
         ret = true;
      }
   }

   string_map_type::iterator ptr;

   if((ptr = _type_map.find(str)) != _type_map.end()) {
      if(is_type(ptr->second, "int")) {
         ret = true;
      }
   }

   return ret;
}

string
c_compiler::term()
{
   string arg1, arg2, op, temp;
   instruction code;

   arg1 = factor();

   if(arg1 == "")
      return arg1;

   if(match("+", OPERATOR)) {
      if(is_double(arg1)) {
         op = "double_add";
      }
      else if(is_long(arg1)) {
         if(is_type(_type_map[arg1], "unsigned")) {
            op = "ulong_add";
         }
         else {
            op = "long_add";
         }
      }
      else if(is_short(arg1)) {
         if(is_type(_type_map[arg1], "unsigned")) {
            op = "ushort_add";
         }
         else {
            op = "short_add";
         }
      }
      else if(is_float(arg1)) {
         op = "float_add";
      }
      else if(is_int(arg1)) {
         if(is_type(_type_map[arg1], "unsigned")) {
            op = "uint_add";
         }
         else {
            op = "int_add";
         }
      }
      else if(is_char(arg1)) {
         if(is_type(_type_map[arg1], "unsigned")) {
            op = "uchar_add";
         }
         else {
            op = "char_add";
         }
      }
      else {
         op = "int_add";
      }
   }
   else if(match("-", OPERATOR))  {
      if(is_double(arg1)) {
         op = "double_sub";
      }
      else if(is_long(arg1)) {
         if(is_type(arg1, "unsigned")) {
            op = "ulong_sub";
         }
         else {
            op = "long_sub";
         }
      }
      else if(is_short(arg1)) {
         if(is_type(arg1, "unsigned")) {
            op = "ushort_sub";
         }
         else {
            op = "short_sub";
         }
      }
      else if(is_float(arg1)) {
         op = "float_sub";
      }
      else {
         op = "int_sub";
      }
   }
   else {
      return arg1;
   }

   _offset++;

   while((arg2 = term()) != "") {

      if(op == "float_add" || op == "float_sub") {
         code = instruction();
         if(op == "float_add") {
            code.op = "float_add";
            code.arg1 = arg1;
            code.arg2 = arg2;
         }
         else if(op == "float_sub") {
            code.op = "float_sub";
            code.arg1 = arg1;
            code.arg2 = arg2;
         }
         else {
            code.op = op;
         }
         _instruction_vector.push_back(code);

         arg1 = "eax";

         _type_map[arg1] = "float";
      }
      else if(op == "double_add" || op == "double_sub") {
         code = instruction();
         if(op == "double_add") {
            code.op = "double_add";
            code.arg1 = arg1;
            code.arg2 = arg2;
            code.arg3 = get_next_temp(8);
         }
         else if(op == "double_sub") {
            code.op = "double_sub";
            code.arg1 = arg1;
            code.arg2 = arg2;
            code.arg3 = get_next_temp(8);
         }
         else {
            code.op = op;
         }
         _instruction_vector.push_back(code);

         arg1 = code.arg3;

         _type_map[arg1] = "double";
      }
      else if(op == "long_add" || op == "long_sub") {
         code = instruction();
         code.op = op;
         code.arg1 = arg1;
         code.arg2 = arg2;
         code.arg3 = get_next_temp(8);
         _instruction_vector.push_back(code);

         arg1 = code.arg3;

         _type_map[arg1] = "long";
      }
      else {
         code = instruction();
         if(op == "int_add") {
            code.op = "int_add";
         }
         else if(op == "int_sub") {
            code.op = "int_sub";
         }
         else {
            code.op = op;
         }
         code.arg1 = arg1;
         code.arg2 = arg2;
         _instruction_vector.push_back(code);

         arg1 = "eax";

         _type_map[arg1] = "int";
      }
   }

   return arg1;
}

string
c_compiler::relation()
{
   string arg1, arg2, op, temp;
   instruction code;

   arg1 = term();

   if(arg1 == "")
      return "";

   if(match("<", OPERATOR)) {
      op = "jl";
   }
   else if(match(">", OPERATOR)) {
      op = "jg";
   }
   else if(match("<=", OPERATOR)) {
      op = "jle";
   }
   else if(match(">=", OPERATOR)) {
      op = "jge";
   }
   else if(match("==", OPERATOR)) {
      op = "je";
   }
   else if(match("!=", OPERATOR)) {
      op = "jne";
   }
   else {
      return arg1;
   }

   _offset++;

   if((arg2 = relation()) != "") {
      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "eax";
      code.arg2 = arg1;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "edx";
      code.arg2 = arg2;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "cmp";
      code.arg1 = "eax";
      code.arg2 = "edx";
      _instruction_vector.push_back(code);

      code = instruction();

      string L1 = get_next_label();

      string L2 = get_next_label();

      code.op = op;
      code.arg1 = L1;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "edx";
      code.arg2 = "0";
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "jmp";
      code.arg1 = L2;
      _instruction_vector.push_back(code);

      code = instruction();
      code.label = L1;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "edx";
      code.arg2 = "1";
      _instruction_vector.push_back(code);

      code = instruction();
      code.label = L2;
      _instruction_vector.push_back(code);

      arg1 = "edx";

      code = instruction();
      code.op = "nop";
      _instruction_vector.push_back(code);

      _type_map[arg1] = "int";
   }

   return arg1;
}

string
c_compiler::boolean()
{
   string arg1, arg2, op, temp;
   instruction code;

   arg1 = relation();

   if(arg1 == "") {
      return arg1;
   }

   if(match("&&", OPERATOR)) {
      _offset++;

      arg2 = boolean();

      // AND instructions
      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "eax";
      code.arg2 = arg2;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "cmp";
      code.arg1 = "eax";
      code.arg2 = "0";
      _instruction_vector.push_back(code);

      string label = get_next_label();

      code = instruction();
      code.op = "je";
      code.arg1 = label;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "eax";
      code.arg2 = arg1;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "cmp";
      code.arg1 = "eax";
      code.arg2 = "0";
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "je";
      code.arg1 = label;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "edx";
      code.arg2 = "1";
      _instruction_vector.push_back(code);
 
      string label2 = get_next_label();

      code = instruction();
      code.op = "jmp";
      code.arg1 = label2;
      _instruction_vector.push_back(code);

      code = instruction();
      code.label = label;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "edx";
      code.arg2 = "0";
      _instruction_vector.push_back(code);

      code = instruction();
      code.label = label2;
      _instruction_vector.push_back(code);

      arg1 = "edx";
      _type_map[arg1] = "int";
   }
   else if(match("||", OPERATOR)) {
      _offset++;

      arg2 = boolean();

      // OR instructions
      code = instruction();
      code.op = "mov_dword";
      code.arg1 = "edx";
      code.arg2 = arg1;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "or";
      code.arg1 = "edx";
      code.arg2 = arg2;
      _instruction_vector.push_back(code);

      arg1 = "edx";
      _type_map[arg1] = "int";
   }

   return arg1;
}

string
c_compiler::expression()
{
   string ret;
   instruction code;

   while(true) {
      ret = boolean();

      if(match("?", OPERATOR) && ret.size()) {
         _offset++;

         string L1 = get_next_label();
         string L2 = get_next_label();
         string L3 = get_next_label();

         code = instruction();
         code.op = "mov";
         code.arg1 = "eax";
         code.arg2 = ret;
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "cmp";
         code.arg1 = "eax";
         code.arg2 = "0";
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "jne";
         code.arg1 = L1;
         _instruction_vector.push_back(code);

         code = instruction();
         code.op = "jmp";
         code.arg1 = L2;
         _instruction_vector.push_back(code);

         code = instruction();
         code.label = L1;
         _instruction_vector.push_back(code);

         string expr1 = expression();

         code = instruction();
         code.op = "mov";
         code.arg1 = "edx";
         code.arg2 = expr1;
         _instruction_vector.push_back(code);

         _type_map["edx"] = _type_map[expr1]; 

         code = instruction();
         code.op = "jmp";
         code.arg1 = L3;
         _instruction_vector.push_back(code);

         if(match(":", OPERATOR))
            _offset++;
         else 
            error("expected: \":\"");
  
         code = instruction();
         code.label = L2;
         _instruction_vector.push_back(code);

         string expr2 = expression();

         code = instruction();
         code.op = "mov";
         code.arg1 = "edx";
         code.arg2 = expr2;
         _instruction_vector.push_back(code);

         _type_map["edx"] = _type_map[expr2];

         code = instruction();
         code.label = L3;
         _instruction_vector.push_back(code);

         ret = "edx";
      }

      if(match(",", OPERATOR) && _in_call_list == false) {
         _offset++;
      }
      else {
         break;
      }
   }

   return ret;
}

bool
c_compiler::expression_statement()
{
   bool ret = false;
   string expr;

   if((expr = expression()) != "") {
      if(match(";", OPERATOR)) {
         _offset++;

         ret = true;
      }
   }
   else if(match(";", OPERATOR)) {
      _offset++;
 
      ret = true;
   }

   return ret;
}

bool 
c_compiler::function_decl_statement()
{
   bool ret = false;
   string key_word, key_word_string;
   string key_word2, key_word_string2;
   string function_name, param_name;
   param_type param;
   param_vector_type param_vector;

   while(is_key_word(_token_vector[_offset].value.c_str()) || get_struct(_token_vector[_offset].value) != 0) {
      key_word = _token_vector[_offset].value;

      key_word_string += _token_vector[_offset].value + " ";

      _offset++;

      if(_offset >= _token_vector.size()) {
         break;
      }

      if(_token_vector[_offset].type == ID) {
         break;
      }
   }

   while(match("*", OPERATOR)) {
      key_word_string += "*";
      _offset++;
   }

   if(_token_vector[_offset].type == ID) {
      function_name = _token_vector[_offset].value; 

      _offset++;

      if(match("(", OPERATOR)) {
         _offset++;

         param_vector = get_param();

         if(match(")", OPERATOR)) {
            _offset++;

            if(match(";", OPERATOR)) {
               _offset++;

               function_rep fun;
             

               fun.name = function_name;
               fun.param_vector = param_vector;
               fun.return_type = key_word_string;

               _function_vector.push_back(fun);


               ret = true;
            }
         }
      }
   }

/*

      if(match("(", OPERATOR)) {
         _offset++;

         if(match(")", OPERATOR)) {

         }
         else {
            while(true) {
               if(_offset >= _token_vector.size())
                  break;

               if(match(")", OPERATOR)) {
                  break;
               }

               if(match("...", OPERATOR)) {
                  param.first = "...";

                  param.second = "...";

                  param_vector.push_back(param);

                  param = param_type();

                  _offset++;
               }
               else {
                  key_word2 = "";

                  key_word_string2 = "";

                  while(is_key_word(_token_vector[_offset].value.c_str())) {

                     key_word2 = _token_vector[_offset].value;

                     key_word_string2 += key_word2 + " ";

                     _offset++;

                     if(_offset >= _token_vector.size())
                        break;
                  }

                  string star = "";

                  while(match("*", OPERATOR)) {
                     star += "*";
 
                     _offset++;
                  }

                  key_word_string2 += star;

                  if(_token_vector[_offset].type == ID) {
                     param_name = _token_vector[_offset].value;

                     _offset++;

                     param.first = param_name;

                     param.second = key_word_string2;

                     param_vector.push_back(param);

                     variable_rep var(this);

                     var.label = param_name;
 
                     var.type = key_word_string2;
                 
                     _variable_vector.push_back(var);

                     param = param_type();

                     if(match(",", OPERATOR)) {
                        _offset++;
                     }
                     else if(match(")", OPERATOR)) {
                        break;
                     }
                  }
               }
            }
         }

         if(_offset >= _token_vector.size()) {
            return false;
         }

         if(match(")", OPERATOR)) {
            _offset++;

            if(match(";", OPERATOR)) {
               _offset++;

               function_rep f;
 
               f.return_type = key_word_string;
     
               f.name = function_name;

               f.param_vector = param_vector;
   
               _function_vector.push_back(f);

               return true;
            }
            else {
               return false;
            }
         }
         else {
            error("expecting: \")\"...");
         }
      }
   }
*/

   return ret;
}

bool
c_compiler::function_impl_statement()
{
cout << "<function_impl_statement>" << endl << flush;

   bool ret = false;
   string key_word, key_word_string;
   string key_word2, key_word_string2;
   string function_name, param_name;
   param_type param;
   param_vector_type param_vector;
 
   while(is_key_word(_token_vector[_offset].value.c_str()) || get_struct(_token_vector[_offset].value) != 0) {
      key_word = _token_vector[_offset].value;

      if(key_word_string.size()) 
         key_word_string += " ";

      key_word_string += _token_vector[_offset].value;

      _offset++;

      if(_offset >= _token_vector.size()) {
         break;
      }
   }

   while(match("*", OPERATOR)) {
      key_word_string += "*";
      _offset++;
   }

   variable_vector_type var_temp = _variable_vector;

   if(_token_vector[_offset].type == ID) {
      function_name = _token_vector[_offset].value; 

      _current_function.name = function_name;

      _offset++;

      if(match("(", OPERATOR)) {
         _offset++;

         if(match(")", OPERATOR)) {

         }
         else {
            param_vector = get_param();
         }

         if(match(")", OPERATOR)) {
            _offset++;

            _first_compound_statement = true;

            if(compound_statement()) {

               function_rep f = _current_function;

cout << "_instruction_vector.size() == " << (_instruction_vector.size()) << endl << flush;

               f.instruction_vector = _instruction_vector;

               f.name = function_name;

               _instruction_vector.clear();

               f.return_type = key_word_string;
 
               f.param_vector = param_vector;

               f.variable_vector = _variable_vector;

               _variable_vector = var_temp;

               param_vector = param_vector_type();
 
               _function_vector.push_back(f);

               _current_function = f;

               variable_rep var(this);

               var.type = key_word_string2;

               var.label = param_name;

               _type_map[param_name] = key_word_string2;

               _variable_vector.push_back(var);

               ret = true;
            }
            else {
               ret = false;
            }
         }
      }
   }

   _current_function.name = "";

   _temp_vec.clear();

   _reuse_temp_vec.clear();

   return ret;
}

bool
c_compiler::statement()
{
cout << "<statement>" << endl << flush;

   bool ret = false;
   size_t off = _offset;

cout << "expression statement next" << endl << flush;

   if(expression_statement()) {
cout << "<expression_statement>" << endl << flush;
      reuse_temp();
      return true;
   }

cout << "back from expression_statement" << endl << flush;

   _offset = off;

   string label = get_next_label();

   if(while_statement()) {
cout << "<while_statement>" << endl << flush;
      instruction code;
      code.label = label;
      _instruction_vector.push_back(code);

      code = instruction();
      code.op = "nop";
      _instruction_vector.push_back(code);
      reuse_temp();
      return true;
   }

   _offset = off;

   if(if_statement()) {
cout << "<if_statement>" << endl << flush;
      reuse_temp();
      return true;
   }

   _offset = off;

   if(do_while_statement()) {
cout << "<do_while_statement>" << endl << flush;
      reuse_temp();
      return true;
   }

   _offset = off;

   if(for_statement()) {
cout << "<for_statement>" << endl << flush;
      reuse_temp();
      return true;
   }

   _offset = off;

   if(switch_statement()) {
cout << "<switch_statement>" << endl << flush;
      reuse_temp();
      return true;
   }

   _offset = off;

   if(continue_statement()) {
cout << "<continue_statement>" << endl << flush;
      reuse_temp();
      return true;
   }

   _offset = off;

   if(break_statement()) {
cout << "<break_statement>" << endl << flush;
      reuse_temp();
      return true;
   }

   _offset = off;

   if(return_statement()) {
cout << "<return_statement>" << endl << flush;
      reuse_temp();
      return true;
   }

   _offset = off;

   if(goto_statement()) {
cout << "<goto_statement>" << endl << flush;
      reuse_temp();
      return true;
   }

   _offset = off;

   if(compound_statement()) {
cout << "<compund_statement>" << endl << flush;
      reuse_temp();
      return true;
   }

   _offset = off;

cout << "current token = \"" << _token_vector[_offset].value << "\"" << endl << flush;

cout << "no statement..." << endl << flush;

   return ret;
}

bool 
c_compiler::union_statement()
{
   bool ret = false;
   
   return ret;
}

bool 
c_compiler::typedef_statement()
{
   bool ret = false;

   if(match("typedef", KEY_WORD)) {
      _offset++;

      string type;

      while(is_typename(_token_vector[_offset].value)) {
         if(type.size()) {
            type += " ";
         }
         type += _token_vector[_offset].value;
         _offset++;
      }

      string name;

      if(_token_vector[_offset].type == ID) {
         name = _token_vector[_offset].value;
         _offset++;
      }

      if(match(";", OPERATOR)) {
         _offset++;

cout << "type: \"" << type << "\"" << endl << flush;
cout << "name: \"" << name << "\"" << endl << flush;
cout << endl << flush;
//cin.get(),cin.get();

         typedef_rep td(type, name);

         _typedef_vector.push_back(td);

         ret = true;
      }
   }

   return ret;
}

bool 
c_compiler::struct_statement()
{
cout << "struct_statement" << endl << flush;
   bool ret = false;

   if(match("struct", KEY_WORD)) {
cout << "SEE STRUCT" << endl << flush;
      _offset++;

      string id,type;
      _current_struct = struct_rep(this);

      if(_token_vector[_offset].type == ID) {
         id = _token_vector[_offset].value;
cout << "STRUCT: " << id << endl << flush;
         _current_struct.name = id;
         _offset++;
      }

      bool found_struct = false;
      size_t index = 0;
      for(size_t i = 0; i < _struct_vector.size(); i++) {
         if(id == _struct_vector[i].name) {
            found_struct = true;
            index = i;
            break;
         }
      }

      if(!found_struct) {
         index = _struct_vector.size();
         _struct_vector.push_back(_current_struct);
      }

      if(match("{", OPERATOR)) {
         _offset++;

         size_t the_size = 0;
         size_t off = _offset;

         string temp = _struct_name;

         _struct_name = id;

         while(var_decl_statement(the_size)) {
            off = _offset;
         }

         string struct_name = _struct_name;
         string label;

         _struct_name = temp;
 
         _offset = off;

         if(match("}", OPERATOR)) {
            _offset++;
 
            while(_token_vector[_offset].type == ID) {
               label = _token_vector[_offset].value;

               _offset++;

               variable_rep var(this);

               var.type = "struct " + struct_name;

               var.label = label;

               var.global = true;

               _variable_vector.push_back(var);

               _symbol_table[var.label] = var;

               _type_map[var.label] = var.type;

               if(match(",", OPERATOR)) {
                  _offset++;
               }
               else {
                  break;
               }
            }

            if(match(";", OPERATOR)) {
               _offset++;

               //_struct_vector.push_back(_current_struct);
               _struct_vector[index] = _current_struct;

               _current_struct = struct_rep(this);

               ret = true;
            }
            else {
               error(string("expected: \";\" got: \"" + (_token_vector[_offset].value) + "\"").c_str());
            }
         }
         else {
            error(string("expected: \"}\" got: \"" + _token_vector[_offset].value + "\"").c_str());
         }
      }
   }

cout << "return: " << ret << endl << flush;

   return ret;
}

bool
c_compiler::global_statement()
{
   _is_global = true;

   size_t old_offset = _offset;

   if(_offset >= _token_vector.size()) {
      return false;
   }

   size_t the_size = 0;

   _is_global = true;
   
   _offset = old_offset;

   if(typedef_statement()) {
cout << "<typedef_statement>" << endl << flush;
      return true;  
   }

   _offset = old_offset;

   if(struct_statement()) {
cout << "<struct_statement>" << endl << flush;
      return true;
   }
   
   _offset = old_offset;

   if(var_decl_statement(the_size)) {
cout << "<var_decl_statement>" << endl << flush;
      return true;
   }

   _offset = old_offset;

   if(function_decl_statement()) {
cout << "<function_decl_statement>" << endl << flush;
      return true;
   }

   _offset = old_offset;

   _is_global = false;

   if(function_impl_statement()) {
cout << "<function_impl_statement>" << endl << flush;
      _is_global = true;
      return true;
   }

   _offset = old_offset;

   if(union_statement()) {
cout << "<union_statement>" << endl << flush;
      return true;
   }

   _offset = old_offset;

cout << "no global statement..." << endl << flush;

   return false;
}

bool
c_compiler::parse()
{
   _offset = 0;

   while(global_statement())
      ;

   return true;
}


bool
c_compiler::is_register(const string & r)
{
   bool ret = false;

   if(r == "eax"
      || r == "ecx"
      || r == "edx"
      || r == "ebx"
      || r == "esp"
      || r == "ebp"
      || r == "esi"
      || r == "edi"
      || r == "rax"
      || r == "rcx"
      || r == "rdx"
      || r == "rbx"
      || r == "rsp"
      || r == "rbp"
      || r == "rsi"
      || r == "rdi"
      || r == "r8b"
      || r == "r9b"
      || r == "r10b"
      || r == "r11b"
      || r == "r12b"
      || r == "r13b"
      || r == "r14b"
      || r == "r15b"
      || r == "r8w"
      || r == "r9w"
      || r == "r10w"
      || r == "r11w"
      || r == "r12w"
      || r == "r13w"
      || r == "r14w"
      || r == "r15w"
      || r == "r8d"
      || r == "r9d"
      || r == "r10d"
      || r == "r11d"
      || r == "r12d"
      || r == "r13d"
      || r == "r14d"
      || r == "r15d"
      || r == "r8"
      || r == "r9"
      || r == "r10"
      || r == "r11"
      || r == "r12"
      || r == "r13"
      || r == "r14"
      || r == "r15"
      || r == "al" 
      || r == "cl" 
      || r == "ch" 
      || r == "dl" 
      || r == "dh" 
      || r == "spl" 
      || r == "bpl" 
      || r == "bpl" 
      || r == "sil" 
      || r == "dil" 
      || r == "ax" 
      || r == "cx" 
      || r == "dx" 
      || r == "bx" 
      || r == "bp" 
      || r == "si" 
      || r == "di" 
      || r == "sp" 
      || r == "ah") {
      ret = true;
   }

//cout << "is_register(\"" << r << "\"): " << ret << endl << flush;

   return ret;
}

bool 
c_compiler::optimize()
{
   bool ret = true;
   bool reduce = true;
   instruction code, last;

   for(size_t j = 0; j < _function_vector.size(); j++) {
      function_rep & f = _function_vector[j];

      instruction_vector_type temp_vec;

      reduce = true;

      while(reduce) {
         reduce = false;

         for(size_t i = 0; i < f.instruction_vector.size(); i++) {
            last = code;

            code = f.instruction_vector[i];

            if(code.label.size()) {
               temp_vec.push_back(code);
            }
            else if(code.op == "mov" && code.arg1 == code.arg2) {
               ret = reduce = true;
            }
            else if(code.op == "add" && is_int(code.arg2) && is_const(code.arg2) && atoi(code.arg2.c_str()) == 0) {
               ret = reduce = true;
            }
            else if(code.op == "sub" && is_int(code.arg2) && is_const(code.arg2) && atoi(code.arg2.c_str()) == 0) {
               ret = reduce = true;
            }
            else {
               if((i+1) < f.instruction_vector.size()) {
                  instruction code2 = f.instruction_vector[i+1];
                  if(code.op == "add" && code2.op == "add"
                     && is_register(code.arg1) && is_register(code2.arg1)
                     && is_int(code.arg2) && is_int(code2.arg2)
                     && code.arg1 == code.arg1) {
                     i++;
                     string arg1 = code.arg2;
                     string arg2 = code2.arg2;
                     int i1 = atoi(arg1.c_str());
                     int i2 = atoi(arg2.c_str());
                     code = f.instruction_vector[i];
                     code.arg2 = to_string(i1 + i2);
                     temp_vec.push_back(code);
                     reduce = ret = true;
                  }
                  else if(code.op == "sub" && code2.op == "sub"
                     && is_register(code.arg1) && is_register(code2.arg1)
                     && is_int(code.arg2) && is_int(code2.arg2)
                     && code.arg1 == code.arg1) {
                     i++;
                     string arg1 = code.arg2;
                     string arg2 = code2.arg2;
                     int i1 = atoi(arg1.c_str());
                     int i2 = atoi(arg2.c_str());
                     code = f.instruction_vector[i];
                     code.arg2 = to_string(i1 + i2);
                     temp_vec.push_back(code);
                     reduce = ret = true;
                  }
                  else if((code.op == "mov" || code.op == "mov_word") && (code2.op == "push")
                     && is_register(code.arg1) && is_register(code2.arg1)
                     && (code.arg1 == code2.arg1)) {
                     i++;
                     string arg = code.arg2;
                     code = f.instruction_vector[i];
                     code.arg1 = arg;
                     temp_vec.push_back(code);
                     reduce = ret = true;
                  }
                  else if((code.op == "mov") && (code2.op == "mov")
                     && is_register(code.arg2) && is_register(code2.arg2)
                     && (code.arg1 == code2.arg2)) {
                     i++;
                     string arg = code.arg2;
                     code = f.instruction_vector[i];
                     code.arg2 = arg;
                     temp_vec.push_back(code);
                     reduce = ret = true;
                  }
                  else if((code.op == "mov") && (code2.op == "mov_word")
                     && is_register(code.arg1) && is_register(code2.arg2)
                     && is_const(code.arg2) 
                     && (code.arg1 == code2.arg2)) {
                     i++;
                     string arg = code.arg2;
                     code = f.instruction_vector[i];
                     code.op = "mov_dword";
                     code.arg2 = arg;
                     temp_vec.push_back(code);
                     reduce = ret = true;
                  }
                  else if((code.op == "mov_dword") && (code2.op == "mov_dword")
                     && is_register(code.arg1) && is_register(code2.arg2)
                     && is_const(code.arg2)
                     && (code.arg1 == code2.arg2)) {
                     i++;
                     string arg = code.arg2;
                     code = f.instruction_vector[i];
                     code.arg2 = arg;
                     temp_vec.push_back(code);
                     reduce = ret = true;
                  }
                  else if((code.op == "mov_dword") && (code2.op == "mov_dword")
                     && is_register(code.arg2) && is_register(code2.arg1)
                     && code.arg2 == code2.arg1
                     && code.arg1 == code2.arg2) {
                     string arg = code.arg2;
                     code = f.instruction_vector[i];
                     code.arg2 = arg;
                     temp_vec.push_back(code);
                     reduce = ret = true;
                     i++;
                  }
                  else {
                     temp_vec.push_back(code);
                  }
               }
               else {
                  temp_vec.push_back(code);
               }
            }
         }

         f.instruction_vector = temp_vec;

         temp_vec.clear();
      }
   }

   return ret;
}

bool
c_compiler::is_type(const string & type_string, 
                    const string & type)
{
   bool ret = false;

   char * buffer = new char [type_string.size()+1];

   memset(buffer, 0, type_string.size()+1); 

   memcpy(buffer, type_string.c_str(), type_string.size());

   char * tok = strtok(buffer, " \n\r\t");

   while(tok != NULL) {
      if(strcmp(tok, type.c_str()) == 0) {
         ret = true;
   
         break;
      }
 
      tok = strtok(NULL, " \r\n\t");
   }
   
   delete [] buffer;

   return ret;
}

int
is_pointer(string type_string) 
{
   int ret = 0;

   for(size_t index = 0; 
       index < type_string.size(); 
       index++) {
      if(type_string[index] == '*') {
         ret++;
      }
   }
 
   return ret;
}

bool
c_compiler::is_param(const string & param)
{
    bool ret = false;

    function_rep * p_function = get_function(_current_function.name);

    if(p_function) {
       if(p_function->has_param(param)) {
          ret = true;
       }
    }

    return ret;
}

bool
c_compiler::is_function(const string & str, int & index)
{
    bool ret = false;

    for(size_t i = 0; i < _function_vector.size(); i++) {
       if(_function_vector[i].name == str) {
          ret = true;
          index = i;
 
          break;
       }
    }

    return ret;
}

bool 
is_register(const string & str)
{
   bool ret;

   // get them all from the NASM manual
   if(str == "edx" 
      || str == "ebx"
      || str == "ecx"
      || str == "esp"
      || str == "ebp"
      || str == "eax"
      || str == "sp"
      || str == "bp"
      || str == "dx"
      || str == "bx"
      || str == "bi"
      || str == "di"
      || str == "al") {
      ret = true;
   }
   else {
      ret = false;
   }

   return ret;
}

int
get_int(float f)
{
   int ret = 0;

   int * ptr = reinterpret_cast< int* >(&f);

   ret = (*ptr);

   return ret;
}

string
c_compiler::to_hex(const float f)
{
   string ret;

   char buffer[1024];

   memset(buffer, 0, 1024);

   sprintf(buffer, "%d", *(int *)&f);

   ret = buffer;

   return ret;
}

string
c_compiler::to_code2(const string & str)
{
   string ret;
   int index = -1;

   if(str.size() == 0) {
      return "";
   }

   if(str[0] == 'C') {
      ret = str;
   }
   else if(str[0] == '.') {
      ret = str;
   }
   else if(isdigit(str[0])) {
      if(is_int(str)) {
         ret = "$" + str;
      }
      else if(is_float(str)) {
         ret = "$" + str;
      }
      else {
         ret = "$" + str;
      }
   }
   else if(is_register(str)) {
      ret = "%" + str;
   }
   else if(is_function(str, index)) {
      ret = str;
   }
   else {
      symbol_table_type::iterator p = _symbol_table.find(str);
      if(p != _symbol_table.end()) {
         ret = (p->second.get_address()) + "(%ebp)";
      }
      else {
         ret = str + "(%ebp)";
      }
   }

   return ret;
}

string
c_compiler::to_code(const string & str) 
{
   string ret;
   int index = -1;

   if(str.size() == 0) {
      return "";
   }

   if(str[0] == '\'') {
      char ch = str[1];
      char buffer[1024];
      memset(buffer, 0, 1024);
      int i = ch;
      sprintf(buffer, "%d", i);   
      ret = string(buffer);
   }
   else if(str[0] == 'C') {
      if(is_type(_type_map[str], "long") || is_type(_type_map[str], "double")) {
         ret = "dword [" + str + "]";
      }
      else {
         ret = str;
      }
   }
   else if(str[0] == 'T') {
      ret = "dword [ebp+" + str + "]";
   }
   else if(is_function(str, index)) {
      ret = str;
   }
   else if(isdigit(str[0])) {
      if(is_double(str)) {
         ret = "dword " + str;
      }
      else if(is_float(str)) {
         ret = "dword " + str;
      }
      else if(is_int(str)) {
         ret = "dword " + str;
      }
      else if(is_short(str)) {
         ret = "dword " + str;
      }
      else if(is_long(str)) {
         ret = "dword " + str;
      }
      else if(is_char(str)) {
         ret = "dword " + str;
      }
      else {
         ret = "dword " + str;
      }
   }
   else if(is_register(str)) {
      ret = str;
   }
   else {
      symbol_table_type::iterator p = _symbol_table.find(str);
      if(p != _symbol_table.end()) {
         //ret = "dword [ebp+" + (p->second.get_address()) + "]";

         if(is_global(str)) {
            ret = "dword [" + str + "]";
         }
         if(is_double(str)) {
            ret = "dword [ebp+" + str + "]";
         }
         else if(is_float(str)) {
            ret = "dword [ebp+" + str + "]";
         }
         else if(is_int(str)) {
            ret = "dword [ebp+" + str + "]";
         }
         else if(is_char(str)) {
            ret = "dword [ebp+" + str + "]";
         }
         else if(is_long(str)) {
            ret = "dword [ebp+" + str + "]";
         }
         else if(is_short(str)) {
            ret = "dword [ebp+" + str + "]";
         }
         else {
            ret = "dword [ebp+" + str + "]";
         }
      }
      else {
         ret = "dword [ebp+" + str + "]";
      }
   }

   return ret;
}

bool
c_compiler::emit_gas()
{
   bool ret = false;

   _out << ".text" << endl << flush;

   for(size_t i = 0; i < _function_vector.size(); i++) {
      if(_function_vector[i].instruction_vector.size() == 0) {
         _out << ".extern " << _function_vector[i].name << endl << flush;
         _out << "\t.type " << _function_vector[i].name << ", @function" << endl << flush;
      }
      else {
         _out << ".global " << _function_vector[i].name << endl << flush;
         _out << "\t.type " << _function_vector[i].name << ", @function" << endl << flush;
      }
   }

   for(size_t i = 0; i < _function_vector.size(); i++) {
      if(_function_vector[i].instruction_vector.size() == 0) {
         continue;
      }

      _out << _function_vector[i].name << ":" << endl << flush;

      for(size_t j = 0; j < _function_vector[i].instruction_vector.size(); j++) {
         instruction & code = _function_vector[i].instruction_vector[j];
         if(code.label != "") {
            _out << (code.label) << ":" << endl << flush;
            continue;
         }

         if(code.arg1.size() && code.arg2.size()) {
            _out << "\t" << code.op << " " << to_code2(code.arg2) << "," << to_code2(code.arg1) << endl << flush;
         }
         else if(code.arg1.size()) {
            _out << "\t" << code.op << " " << to_code2(code.arg1) << endl << flush;
         }
         else {
            _out << "\t" << code.op << endl << flush;
         }
      }
   }

   return ret;
}

string
c_compiler::to_nasm(const string & str) 
{
   string ret;

   for(size_t i = 0; i < str.size(); i++) {
      if((i+1) < str.size()) {
         if(str[i] == '\\' && str[i+1] == '\"') {
            ret += "\",\"" + to_string((int)('\"')) + "\"";
         }
         else {
            ret += str[i];
         }
      }
      else {
         ret += str[i];
      }
   }

   return ret;
}

size_t
c_compiler::get_size(const string & id)
{
   size_t ret = 0U;

   symbol_table_type::iterator p;

   p = _symbol_table.find(id);

   if(p != _symbol_table.end()) {
      ret = p->second.get_size();
   }

   return ret;
}

string
c_compiler::to_struct(const string & str)
{
   string ret;

   char buffer[1024];

   memset(buffer, 0, 1024);

   strcpy(buffer, str.c_str());

   char * ptr = strtok(buffer, " *");

   while(ptr != NULL) {
      if(strcmp(ptr, "struct") == 0) {

      }
      else {
         if(ret.size()) {
            ret += " ";
         }

         ret += string(ptr);
      }

      ptr = strtok(NULL, " *");
   }

   return ret;
}

bool 
c_compiler::is_global(const string & id)
{
   bool ret = false;

   string temp;

   for(size_t i = 0; i < id.size(); i++) {
      if(id[i] == '+' || id[i] == '-') {
         break;
      }
      temp += id[i];
   }

   for(size_t i = 0; i < _variable_vector.size(); i++) {
      if(_variable_vector[i].label == temp) {
         if(_variable_vector[i].global) {
            ret = true;
            break;
         }
      }
   }

   return ret;
}

bool
c_compiler::is_function(const string & f)
{
   bool ret = false;

   for(size_t i = 0; i < _function_vector.size(); i++) {
      if(_function_vector[i].name == f) {
         ret = true;
         break;
      }
   }

   return ret;
}

string
c_compiler::to_param(const string_map_type & param_map, const string & param)
{
   char * buffer = new char[param.size()+1];

   memset(buffer, 0, param.size()+1);

   sprintf(buffer, "%s", param.c_str());

   char * ptr = strtok(buffer, "+-");

   string_vector_type vec;

   while(ptr != NULL) {
      vec.push_back(ptr);

      ptr = strtok(NULL, "+-");
   }

   string ret = param;

   if(vec.size() > 1) {
      if(param_map.find(vec[0]) != param_map.end()) {
         ret += "+4";
      }
   }

   delete [] buffer;

   return ret;
}

bool
c_compiler::emit_nasm()
{
   int count = 0;
   string int_type = "int";
   string float_type = "float";
   string long_type = "long";
   string double_type = "double";
   string char_type = "char";
   string short_type = "short";

   _out << "section .data" << endl << flush;
   _out << endl << flush;
 
   size_t size = 0;

   for(size_t i = 0; i < _struct_vector.size(); i++) {
      struct_rep * p = &(_struct_vector[i]);
      size = p->get_size();

      _out << p->name << " equ " << size << endl << endl << flush;

      for(size_t j = 0; j < p->var_vec.size(); j++) {
         _out << p->name << "_" << p->var_vec[j].label << " equ " << size
              << endl << flush;

         size -= (p->var_vec[j].get_size());
      }

      _out << endl << flush;
   }

   _out << "section .bss" << endl << flush;
   
   _out << endl << flush;

   for(size_t i = 0; i < _variable_vector.size(); i++) {
      variable_rep & var = _variable_vector[i];
      if(var.global) {
         if(var.pointer_count) {
            _out << var.label << "\tresd 1"
                 << endl << flush;
         }
         else if(is_type(var.type, "int")) {
            _out << var.label << "\tresd 1"
                 << endl << flush;
         }
         else if(is_type(var.type, "float")) {
            _out << var.label << "\tresd 1"
                 << endl << flush;
         }
         else if(is_type(var.type, "long")) {
            _out << var.label << "\tresq 1"
                 << endl << flush;
         }
         else if(is_type(var.type, "double")) {
            _out << var.label << "\tresq 1"
                 << endl << flush;
         }
         else if(is_type(var.type, "short")) {
            _out << var.label << "\tresw 1"
                 << endl << flush;
         }
         else if(is_type(var.type, "char")) {
            _out << var.label << "\tresb 1"
                 << endl << flush;
         }
         else {
            _out << var.label << "\tresb " << var.get_size()
                 << endl << flush;
         }
      }
   }

   _out << endl << flush;
   _out << "section .data" << endl << endl << flush;

   size_t z, x, sz;

   map< string, int > function_map;

   for(x = 0; x < _function_vector.size(); x++) {
   count = 0;
   size = 0;
   //for(size_t i = 0; i < _variable_vector.size(); i++) {
      //variable_rep var = _variable_vector[i];
   for(size_t i = 0; i < _function_vector[x].variable_vector.size(); i++) {
      variable_rep var = _function_vector[x].variable_vector[i];

      if(var.label == "" || var.global) {
         continue;
      }
cout << "var: " << var.label << endl << flush;
cout << "type: " << var.type << endl << flush;
cout << "default_value: " << var.default_value << endl << flush;
//cin.get();
//cin.get();

      if(var.array.size() && !is_type(var.type, "struct")) {
         int temp = 1; 
         for(size_t k = 0; k < var.array.size(); k++) {
            temp *= var.array[k];    
         }
         temp *= DATA_WORD_SIZE;
         count += temp;
         _out << var.label << " equ " << -count
              << endl << flush;
      }
      else if(var.array.size() && is_pointer(var.type) && is_type(var.type, "struct")) {
         int temp = 1; 
         for(size_t k = 0; k < var.array.size(); k++) {
            temp *= var.array[k];    
         }
         temp *= DATA_WORD_SIZE;
         count += temp;
         _out << var.label << " equ " << -count
              << endl << flush;
      }
      else if(is_pointer(var.type) == 1 && var.label[0] == 'C') {
cout << "GOT DB 2" << endl << flush;
         if(is_type(var.type, "int"))  {
            _out << var.label << ":"
                 << " dd "
                 << var.default_value
                 << endl << flush;
         }
         else {
            _out << var.label << ":"
                 << " db "
                 << var.default_value
                 << endl << flush;
         }
      }
      else if(is_pointer(var.type) == 1) {
         count += DATA_WORD_SIZE;
         _out << var.label << " equ " <<  -count
              << endl << flush;
      }
      else if(is_type(var.type, int_type)
         || is_type(var.type, float_type)) { 
         count += DATA_WORD_SIZE;

         _out << var.label << " equ " <<  -count
              << endl << flush;

         //count += 4;
      }
      // long and double 64 bit.
      else if(is_type(var.type, long_type)
         || is_type(var.type, double_type)) {
         if(var.default_value != "") {
            _out << var.label << ": dq " << var.default_value
                 << endl << flush;
cout << "GET DQ 3" << endl << flush;
         }
         else {
            count += 8;

            _out << var.label << " equ " << -count
                 << endl << flush;

            //count += 8;
         }
      }
      // short is only 16 bits.
      else if(is_type(var.type, short_type)) {
         count += 2;
         //count += 4;

         _out << var.label << " equ " << -count
              << endl << flush;

         //count += 2;
      }
      else if(is_type(var.type, char_type) 
              && is_pointer(var.type) == 1) {
cout << "GOT DB 4" << endl << flush;
         _out << var.label << ":" 
              << " db " 
              << to_nasm(var.default_value)
              << endl << flush;
      }
      // char 1 byte 8 bits.
      else if(is_type(var.type, char_type)) {
         if(var.default_value.size()) {
cout << "GOT DB 5" << endl << flush;
            _out << var.label << ": db " << var.default_value << endl << flush;
         }
         else {
            count++;

            _out << var.label << " equ " << -count
                 << endl << flush;
         }

         //count++;
      }
      else if(is_type(var.type, "struct")) {
         struct_rep * p = get_struct(var.type);

         if(p != 0) {
            //count += p->get_size();

            size_t temp = p->get_size();

            for(size_t x = 0; x < var.array.size(); x++) {
               temp *= var.array[x];
            }

            count += temp;

            _out << var.label << " equ " << -count
                 << endl << flush;
         }
         else if(var.type.find("*") != string::npos) {
            count += DATA_WORD_SIZE;
            _out << var.label << " equ " << -count
                 << endl << flush;
         }
         else {
            error(string(string("unknown struct: ") + var.type).c_str());
         }
      }
      else {
         error(("unknown variable size: \"" + var.type + "\"").c_str());
      }

      function_map[_function_vector[x].name] = count;
   }

   //bool found = false;
   //size_t z, x, sz;

   _out << endl << flush;

   //for(x = 0; x < _function_vector.size(); x++) {
      if(_function_vector[x].instruction_vector.size()) {
         size = DATA_WORD_SIZE;

         for(z = 0; z < _function_vector[x].param_vector.size(); z++) {
            sz = get_size(_function_vector[x].param_vector[z].first);

            if(_function_vector[x].param_vector[z].first != "...") {
               //if(!(is_type(_function_vector[x].return_type, "struct") && _function_vector[x].return_type.find("*") == string::npos)) {
                  size += sz;
               //}

               //if(size == 8 && _function_vector[x].param_vector.size() == 1)
               //   size -= 4;

               _out << _function_vector[x].param_vector[z].first << " equ "
                    << size << endl << flush;

               //if(is_type(_function_vector[x].return_type, "struct") && _function_vector[x].return_type.find("*") == string::npos) {
               //   size += sz;
               //}
            }
         }

         if(is_type(_function_vector[x].return_type, "struct") && _function_vector[x].return_type.find("*") == string::npos) {
            struct_rep * p_struct = get_struct(_function_vector[x].return_type);

            param_type rv("retval", _function_vector[x].return_type + " *");

            _function_vector[x].param_vector.push_back(rv);
            
            if(p_struct) {
               sz = DATA_WORD_SIZE;

               //sz = get_size(_function_vector[x].return_type);

               size += sz;

               _out << "retval equ " << size << endl << flush;

               //size += sz;
            }  
         }     
      }
   }

   _out << endl << flush;
   _out << "section .text" << endl << flush;

   for(z = 0; z < _function_vector.size(); z++) {
      if(_function_vector[z].name == "array_put"
         || _function_vector[z].name == "array_get") {
         continue;
      }
      else if(is_type(_function_vector[z].return_type, "extern")) {
         _out << "\textern " << (_function_vector[z].name)
              << endl << flush;
      }
      else {
         _out << "\tglobal " << (_function_vector[z].name)
              << endl << flush;
      }
   }

   string_map_type param_map;

   _out << endl << flush;

   _out << "%include \"rt.asm\"" << endl << endl << flush;

   _out << "%include \"" << "i386_macro.asm\"" << endl << flush;

   _out << endl << flush;

   for(z = 0; z < _function_vector.size(); z++) {
      count = function_map[_function_vector[z].name];

      _current_function.name = _function_vector[z].name;

      if(is_type(_function_vector[z].return_type, "extern")) {
         continue;
      }

      param_map.clear();

      if(_function_vector[z].instruction_vector.size()) {
         size = 4;

         for(x = 0; x < _function_vector[z].param_vector.size(); x++) {
            sz = get_size(_function_vector[z].param_vector[x].first);

            if(_function_vector[z].param_vector[x].first != "...") {
               size += sz;

               param_map[_function_vector[z].param_vector[x].first] = "true";
            }
         }
      }

/*
      if(is_type(_function_vector[z].return_type, "struct"))  {
         param_type param;
         param.first = "retval";
         param.second = _function_vector[z].return_type+" *";
         _function_vector[z].param_vector.push_back(param);
      }
*/

      size = 0;

      for(x = 0; x < _function_vector[z].param_vector.size(); x++) {
         sz = get_size(_function_vector[z].param_vector[x].first);

         if(_function_vector[z].param_vector[x].first != "...") {
            size += sz;
         }
      }

      count += size;

cout << endl << flush;
cout << "Z: " << z << endl << flush;
cout << "EMIT FUNCTION: " << (_function_vector[z].name) << endl << flush;
cout << "INSTRUCTION COUNT: " << (_function_vector[z].instruction_vector.size()) << endl << flush;
cout << endl << flush;

      _instruction_vector = _function_vector[z].instruction_vector;

      _current_function = _function_vector[z];

      if(_instruction_vector.size() > 2) {
         if(_instruction_vector[2].op == "sub") {
            _instruction_vector[2].arg2 = to_string(function_map[_function_vector[z].name]);
         }
      }

      _out << endl << flush;
   
      _out << _current_function.name << ":" << endl << flush;

      for(size_t i = 0; i < _instruction_vector.size(); i++) {
         instruction & code = _instruction_vector[i];

cout << code.op << " " << code.arg1 << "," << code.arg2 << "," << code.arg3 << endl << flush;

         if(code.label != "") {
            _out << code.label << ":" << endl << flush;
            continue;
         }

         if(code.op == "float_add" || code.op == "float_sub"
            || code.op == "float_mul" || code.op == "float_div"
            || code.op == "double_add" || code.op == "double_sub"
            || code.op == "double_mul" || code.op == "double_div"
            || code.op == "int_add" || code.op == "int_sub"
            || code.op == "int_mul" || code.op == "int_div"
            || code.op == "ulong_mul" || code.op == "ulong_div"
            || code.op == "ulong_add" || code.op == "ulong_add"
            || code.op == "long_add" || code.op == "long_sub"
            || code.op == "long_mul" || code.op == "long_div") {
            if(code.op == "double_add" || code.op == "double_sub"
               || code.op == "double_mul" || code.op == "double_div") {
               _out << "\t" << code.op << "\t" << (code.arg1) << "," << (code.arg2) << ","
                    << (code.arg3)
                    << endl << flush;
            }
            else if(code.op == "long_mul" || code.op == "long_div" 
                    || code.op == "ulong_mul" || code.op == "ulong_div") {
               _out << "\t" << code.op << "\t" << code.arg1 << "," << code.arg2 << "," << code.arg3 
                    << endl << flush;
            }
            else if(code.op == "long_add" || code.op == "long_sub"
                    || code.op == "ulong_add" || code.op == "ulong_sub") {
               _out << "\t" << code.op << "\t" << (code.arg1) << "," << (code.arg2) << ","
                    << (code.arg3)
                    << endl << flush;
            }
            else {
               _out << "\t" << code.op << "\t" << to_code(to_param(param_map, code.arg1)) << "," << to_code(to_param(param_map, code.arg2))
                    << endl << flush;
            }
         }
         else if(code.op == "mov_struct3") {
            _out << "\tmov_struct3\t" << code.arg1 << "," << code.arg2 << "," << code.arg3 << endl << flush;
         }
         else if(code.op == "mov_struct2") {
            _out << "\tmov_struct2\t[ebp+" << code.arg1 << "]," << code.arg2 << "," << code.arg3 << endl << flush;
         }
         else if(code.op == "mov_struct") {
            _out << "\tmov_struct\t[ebp+" << code.arg1 << "],[ebp+" << code.arg2 << "]," << code.arg3 << endl << flush;
         }
         else if(code.op == "mov3") {
            _out << "\tmov\t[" << code.arg1 << "],dword " << code.arg2 << endl << flush;
         }
         else if(code.op == "mov2") {
            _out << "\tmov\t" << code.arg1 << ",dword [" << code.arg2 << "]" << endl << flush;
         }
         else if(code.op == "push" || code.op == "push_dword") {
            if(is_register(code.arg1)) {
               _out << "\tpush\t" << code.arg1 << endl << flush;
            }
            else if(is_const(code.arg1)) {
               _out << "\tpush\tdword " << code.arg1 << endl << flush;
            }
            else if(is_global(code.arg1)) {
               _out << "\tpush\tdword [" << code.arg1 << "]" << endl << flush;
               //_out << "\tmov\tedx,dword [" << code.arg1 << "]" << endl << flush;
               //_out << "\tpush\tedx" << endl << flush;
            }
            else {
               _out << "\tpush\tdword [ebp+" << to_param(param_map, code.arg1) << "]" << endl << flush;
               //if(!is_param(code.arg1)) {
                  //_out << "\tmov\tedx,dword [ebp+" << to_param(param_map, code.arg1) << "]" << endl << flush;
                  //_out << "\tpush\tedx" << endl << flush;
               //}
               //else {
               //   _out << "\tlea\teax,[ebp+" << to_param(param_map, code.arg1) << "]" << endl << flush;
               //   _out << "\tadd\teax,dword " << DATA_WORD_SIZE_STR << endl << flush;
               //   _out << "\tmov\tedx,dword [eax]" << endl << flush;
               //   _out << "\tpush\tedx" << endl << flush;
               //}
            }
         }
         else if(code.op == "push2") {
            _out << "\tpush\tdword " << code.arg1 << endl << flush;
         }
         else if(code.op == "add2") {
            _out << "\tadd\t" << code.arg1 << "," << code.arg2 << endl << flush;
         }
         else if(code.op == "sub2") {
            _out << "\tsub\t" << code.arg1 << "," << code.arg2 << endl << flush;
         }
         else if(code.op == "push_qword") {
            if(is_const(code.arg1)) {
               _out << "\t" << code.op << "\t" << (code.arg1) << endl << flush;
            }
            else {
               if(is_global(code.arg1)) {
                  _out << "\t" << code.op << "\t" << (code.arg1) << endl << flush;
               }
               else {
                  _out << "\t" << code.op << "\tebp+" << (code.arg1) << endl << flush;
               }
            }
         }
         else if(code.op == "lea") {
            if(is_global(code.arg1)) {
               _out << "\tlea\t" << code.arg1 << "," << "[" << code.arg2 << "]" << endl << flush; 
            }
            else {
               _out << "\tlea\t" << code.arg1 << "," << "[ebp+" << to_param(param_map, code.arg2) << "]" << endl << flush; 
            }
         }
         else if(code.op == "r_val") {
            if(is_register(code.arg2)) {
               _out << "\tr_val2\tdword " << code.arg1 << ",dword [" << code.arg2 << "]" << endl << flush;
            }
            else if(is_global(code.arg2)) {
               _out << "\tr_val\tdword " << code.arg1 << ",dword [" << code.arg2 << "]" << endl << flush;
            }
            else {
               _out << "\tr_val\tdword " << code.arg1 << ",dword [ebp+" << to_param(param_map, code.arg2) << "]" << endl << flush;
            }
         }
         else if(code.op == "l_val") {
            if(is_const(code.arg2) && !is_register(code.arg1)) {
               if(is_global(code.arg1)) {
                  _out << "\tl_val\tdword [" << code.arg1 << "],dword " << code.arg2 << endl << flush;
               }
               else {
                  _out << "\tl_val\tdword [ebp+" << to_param(param_map, code.arg1) << "],dword " << code.arg2 << endl << flush;
               }
            }
            else if(is_const(code.arg2) && is_register(code.arg1)) {
               _out << "\tl_val2\tdword [" << code.arg1 << "],dword " << code.arg2 << endl << flush;
            }
            else if(is_register(code.arg1) && is_register(code.arg2)) {
               _out << "\tl_val2\tdword [" << code.arg1 << "],dword " << code.arg2 << endl << flush;
            }
            else if(is_register(code.arg1)) {
               if(is_global(code.arg2)) {
                  _out << "\tl_val2\tdword [" << code.arg1 << "],dword [" << code.arg2 << "]" << endl << flush;
               }
               else {
                  _out << "\tl_val2\tdword [" << code.arg1 << "],dword [ebp+" << to_param(param_map, code.arg2) << "]" << endl << flush;
               }
            }
            else {
               if(is_global(code.arg1) && is_global(code.arg2)) {
                  _out << "\tl_val\tdword [" << code.arg1 << "],dword [" << code.arg2 << "]" << endl << flush;
               }
               else if(is_global(code.arg1)) {
                  _out << "\tl_val\tdword [" << code.arg1 << "],dword [ebp+" << to_param(param_map, code.arg2) << "]" << endl << flush;
               }
               else if(is_global(code.arg2)) {
                  _out << "\tl_val\tdword [ebp+" << to_param(param_map, code.arg1) << "],dword [" << code.arg2 << "]" << endl << flush;
               }
               else {
                  _out << "\tl_val\tdword [ebp+" << to_param(param_map, code.arg1) << "],dword [ebp+" 
                       << to_param(param_map, code.arg2) << "]" << endl << flush;
               }
            }
         }
         else if(code.op == "fld") {
            if(code.arg1[0] == 'C') {
               _out << "\tfld\tqword [" << code.arg1 << "]" << endl << flush; 
            }
            else {
               if(is_global(code.arg1)) {
                  _out << "\tfld\tqword [" << code.arg1 << "]" << endl << flush; 
               }
               else {
                  _out << "\tfld\tqword [ebp+" << to_param(param_map, code.arg1) << "]" << endl << flush; 
               }
            }
         }
         else if(code.op == "fstp") {
            if(code.arg1[0] == 'C') {
               _out << "\tfstp\tqword [" << code.arg1 << "]" << endl << flush; 
            }
            else {
               if(is_global(code.arg1)) {
                  _out << "\tfstp\tqword [" << code.arg1 << "]" << endl << flush; 
               }
               else {
                  _out << "\tfstp\tqword [ebp+" << to_param(param_map, code.arg1) << "]" << endl << flush; 
               }
            }
         }
         else if(code.op == "fstp_dword") {
            if(is_global(code.arg1)) {
               _out << "\tfstp\tdword [" << code.arg1 << "]" << endl << flush; 
            }
            else {
               _out << "\tfstp\tdword [ebp+" << to_param(param_map, code.arg1) << "]" << endl << flush; 
            }
         }
         else if(code.op == "fld_dword") {
            if(is_global(code.arg1)) {
               _out << "\tfld\tdword [" << code.arg1 << "]" << endl << flush; 
            }
            else {
               _out << "\tfld\tdword [ebp+" << to_param(param_map, code.arg1) << "]" << endl << flush; 
            }
         }
         else if(code.op == "mov_dword" || code.op == "mov") {
            if(is_global(code.arg1) && is_global(code.arg2)) {
               _out << "\tmov_dword dword [" << (code.arg1) << "],dword [" << (code.arg2) << "]" << endl << flush; 
            }
            else if(is_global(code.arg1) && is_register(code.arg2)) {
               _out << "\tmov\t[" << (code.arg1) << "],dword " << (code.arg2) << endl << flush; 
            }
            else if(is_global(code.arg1) && is_const(code.arg2)) {
               _out << "\tmov\t[" << (code.arg1) << "],dword " << (code.arg2) << endl << flush; 
            }
            else if(is_global(code.arg1) && !is_function(code.arg2)) {
               _out << "\tmov\tdword [" << (code.arg1) << "],dword [ebp+" << to_param(param_map, code.arg2) << "]" << endl << flush; 
            }
            else if(is_global(code.arg2)) {
               _out << "\tmov\t" << (code.arg1) << ",dword [" << (code.arg2) << "]" << endl << flush; 
            }
            else {
               if(!is_register(code.arg1) && !is_const(code.arg1)) {
                  //if(!is_param(code.arg1)) {
                     _out << "\tmov\t[ebp+" << to_param(param_map, code.arg1) << "],dword " << (code.arg2) << endl << flush; 
                  //}
                  //else {
                  //   _out << "\tlea\teax,[ebp+" << code.arg1 << "]" << endl << flush; 
                  //   _out << "\tadd\teax,dword " << DATA_WORD_SIZE_STR << endl << flush; 
                  //   _out << "\tmov\tdword [eax],dword " << code.arg2 << endl << flush; 
                  //}
               }
               else if(!is_register(code.arg2) && !is_const(code.arg2) && !is_function(code.arg2)) {
                  //if(!is_param(code.arg2)) {
                     _out << "\tmov\t" << (code.arg1) << ",dword [ebp+" << to_param(param_map, code.arg2) << "]" << endl << flush; 
                  //}
                  //else {
                  //   _out << "\tlea\teax,[ebp+" << code.arg2 << "]" << endl << flush; 
                  //   _out << "\tadd\teax,dword " << DATA_WORD_SIZE_STR << endl << flush; 
                  //   _out << "\tmov\t" << (code.arg1) << ",dword [eax]" << endl << flush; 
                  //} 
               }
               else if(is_register(code.arg1) && is_register(code.arg2)) {
                  _out << "\tmov\t" << (code.arg1) << "," << (code.arg2) << endl << flush; 
               }
               else if(is_register(code.arg1)) {
                  _out << "\tmov\t" << (code.arg1) << ",dword " << (code.arg2) << endl << flush; 
               }
               else if(is_register(code.arg2)) {
                  _out << "\tmov\t" << (code.arg1) << "," << (code.arg2) << endl << flush; 
               }
               else {
                  _out << "\tmov\t" << (code.arg1) << ",dword " << (code.arg2) << endl << flush; 
               }
            }
         }
         else if(code.op == "mov_qword") {
            if(!is_register(code.arg1) && !is_const(code.arg1) && is_register(code.arg2)) {
               _out << "\tmov_qword\t" << (code.arg1) << "," << (code.arg2) << endl << flush; 
            }
            else if(is_register(code.arg1) && !is_const(code.arg2) && !is_register(code.arg2)) {
               _out << "\tmov_qword\t" << (code.arg1) << "," << (code.arg2) << endl << flush; 
            }
            else if(is_const(code.arg2) && is_register(code.arg1)) {
               _out << "\tmov_qword\t" << (code.arg1) << "," << (code.arg2) << endl << flush; 
            }
            else if(is_const(code.arg2)) {
               if(is_global(code.arg1)) {
                  _out << "\tmov_qword\t" << (code.arg1) << "," << (code.arg2) << endl << flush; 
               }
               else {
                  _out << "\tmov_qword\t" << "ebp+" << (code.arg1) << "," << (code.arg2) << endl << flush; 
               }
            }
            else {
               if(is_global(code.arg1) && is_global(code.arg2)) {
                  _out << "\tmov_qword\t" << (code.arg1) << "," << (code.arg2) << endl << flush; 
               }
               else if(is_global(code.arg1)) {
                  _out << "\tmov_qword\t" << (code.arg1) << ",ebp+" << (code.arg2) << endl << flush; 
               }
               else if(is_global(code.arg2)) {
                  _out << "\tmov_qword\tebp+" << (code.arg1) << "," << (code.arg2) << endl << flush; 
               }
               else {
                  _out << "\tmov_qword\t" << "ebp+" << (code.arg1) << ",ebp+" << (code.arg2) << endl << flush; 
               }
            }
         }
         else if(code.op == "mov_word") {
            if(code.arg1 == "esp") {
               _out << "\tmov\tword [" << (code.arg1) << "]," << (code.arg2) << endl << flush; 
            } 
            else if(is_register(code.arg1) && is_register(code.arg2)) {
               _out << "\tmov\t" << (code.arg1) << "," << (code.arg2) << endl << flush; 
            }
            else if(is_register(code.arg1) && is_const(code.arg2)) {
               _out << "\tmov\t" << (code.arg1) << "," << (code.arg2) << endl << flush; 
            }
            else if(!is_register(code.arg1) && is_const(code.arg2)) {
               if(is_global(code.arg1)) {
                  _out << "\tmov\t[" << (code.arg1) << "],word " << (code.arg2) << endl << flush; 
               }
               else {
                  _out << "\tmov\t[ebp+" << to_param(param_map, code.arg1) << "],word " << (code.arg2) << endl << flush; 
               }
            }
            else if(is_register(code.arg1)) {
               if(is_global(code.arg2)) {
                  _out << "\tmov\tword " << (code.arg1) << ",[" << (code.arg2) << "]" << endl << flush; 
               }
               else { 
                  _out << "\tmov\tword " << (code.arg1) << ",[ebp+" << to_param(param_map, code.arg2) << "]" << endl << flush; 
               }
            }
            else if(is_register(code.arg2)) {
               if(is_global(code.arg1)) {
                  _out << "\tmov\t[" << (code.arg1) << "]," << (code.arg2) << endl << flush; 
               }
               else {
                  _out << "\tmov\t[ebp+" << to_param(param_map, code.arg1) << "],word " << (code.arg2) << endl << flush; 
               }
            }
            else {
               if(is_global(code.arg1)) {
                  _out << "\tmov\t[" << (code.arg1) << "],word " << (code.arg2) << endl << flush; 
               }
               else {
                  _out << "\tmov\t[ebp+" << to_param(param_map, code.arg1) << "]," << (code.arg2) << endl << flush; 
               }
            }
         }
/**************
         else if(code.op == "mov") {
            if(is_global(code.arg1)) {
               _out << "\tmov " << "[" << (code.arg1) << "]," << (code.arg2) << endl << flush; 
            }
            else if(is_global(code.arg2)) {
               _out << "\tmov " << (code.arg1) << ",[" << (code.arg2) << "]" << endl << flush; 
            }
            else if(!is_register(code.arg1) && !is_const(code.arg1) && is_register(code.arg2)) {
               _out << "\tmov [ebp+" << to_param(param_map, code.arg1) << "]," << (code.arg2) << endl << flush; 
            }
            else if(!is_register(code.arg1) && !is_const(code.arg1) && is_const(code.arg2)) {
               _out << "\tmov [ebp+" << to_param(param_map, code.arg1) << "]," << (code.arg2) << endl << flush; 
            }
            else if(is_const(code.arg2) && is_register(code.arg1)) {
               _out << "\tmov " << (code.arg1) << "," << (code.arg2) << endl << flush; 
            }
            else if(is_const(code.arg2)) {
               _out << "\tmov [ebp+" << to_param(param_map, code.arg1) << "]," << (code.arg2) << endl << flush; 
            }
            else {
               _out << "\tmov " << (code.arg1) << ",[ebp+" << to_param(param_map, code.arg2) << "]" << endl << flush; 
            }
         }
**************/
         else if(code.op == "push_word") {
            if(code.arg1 == "al") {
               _out << "\tpush\tal" << endl << flush; 
            }
            else if(is_const(code.arg1) || is_register(code.arg1)) {
               _out << "\tpush\tword " << (code.arg1) << endl << flush; 
            }
            else {
               if(is_global(code.arg1)) {
                  _out << "\tpush\tword [" << (code.arg1) << "]" << endl << flush; 
                  //_out << "\tmov\tword edx,word [" << (code.arg1) << "]" << endl << flush; 
                  //_out << "\tpush\tword edx" << endl << flush; 
               }
               else {
                  _out << "\tpush\tword [ebp+" << to_param(param_map, code.arg1) << "]" << endl << flush; 
                  //_out << "\tmov_word\tdx,word [ebp+" << to_param(param_map, code.arg1) << "]" << endl << flush; 
                  //_out << "\tpush\tword edx" << endl << flush; 
               }
            }
         }
         else if(code.op == "mov_long") {
            _out << "\tmov\t" << to_code(code.arg1) << "," << to_code(code.arg2) << endl << flush;
         }
         else if(code.op == "mov_byte") {
            if(code.arg1 == "esp") {
               _out << "\tmov\t[esp],byte " << (code.arg2) << endl << flush; 
            }
            else if(is_register(code.arg1) && is_register(code.arg2)) {
               _out << "\tmov\tbyte " << (code.arg1) << ",byte " << (code.arg2) << endl << flush; 
            }
            else if(is_register(code.arg1) && is_const(code.arg2)) {
               _out << "\tmov\tbyte " << (code.arg1) << ",byte " << to_code(code.arg2) << endl << flush; 
            }
            else if(is_register(code.arg1)) {
               if(is_global(code.arg2)) {
                  _out << "\tmov\tbyte " << (code.arg1) << ",[" << (code.arg2) << "]" << endl << flush; 
               }
               else {
                  _out << "\tmov\tbyte " << (code.arg1) << ",[ebp+" << to_param(param_map, code.arg2) << "]" << endl << flush; 
               }
            }
            else if(is_register(code.arg2)) {
               if(is_global(code.arg1)) {
                  _out << "\tmov\t" << "[" << (code.arg1) << "],byte " << (code.arg2) << endl << flush; 
               }
               else {
                  _out << "\tmov\t" << "[ebp+" << to_param(param_map, code.arg1) << "],byte " << (code.arg2) << endl << flush; 
               }
            }
            else if(is_const(code.arg2) && code.arg2[0] == 'C') {
               if(is_global(code.arg1)) {
                  _out << "\tmov\t[" << (code.arg1) << "],dword " << (code.arg2) << endl << flush; 
               }
               else {
                  _out << "\tmov\t[ebp+" << to_param(param_map, code.arg1) << "],dword " << (code.arg2) << endl << flush; 
               }
            }
            else {
               if(is_global(code.arg1)) {
                  _out << "\tmov\t[" << (code.arg1) << "],byte " << (code.arg2) << endl << flush; 
               }
               else {
                  _out << "\tmov\t[ebp+" << to_param(param_map, code.arg1) << "],byte " << (code.arg2) << endl << flush; 
               }
            }
         }
         else if(code.op == "push_byte") {
            _out << "\tpush\tbyte " << (code.arg1) << endl << flush; 
         }
         else if(code.op == "call") {
            _out << "\tcall\t" << code.arg1 << endl << flush; 
         }
         else if(code.op == "call2") {
            _out << "\tcall\t[ebp+" << to_param(param_map, code.arg1) << "]" << endl << flush; 
         }
         else if(code.arg1.size() && code.arg2.size()) {
            if(is_register(code.arg1) && is_register(code.arg2)) {
               _out << "\t" << code.op << "\t" << code.arg1 << "," << code.arg2 << endl << flush;
            }
            else if(is_register(code.arg2)) {
               _out << "\t" << code.op << "\t" << to_code(code.arg1) << "," << code.arg2 << endl << flush;
            }
            else if(is_register(code.arg1)) {
               _out << "\t" << code.op << "\t" << code.arg1 << "," << to_code(code.arg2) << endl << flush;
            }
            else {
               _out << "\t" << code.op << "\t" << to_code(code.arg1) << "," << to_code(code.arg2) << endl << flush;
            }
         }
         else if(code.arg1.size()) {
            if(is_register(code.arg1) || code.arg1[0] == '.') {
               _out << "\t" << code.op << "\t" << code.arg1 << endl << flush;
            }
            else {
               _out << "\t" << code.op << "\t" << to_code(code.arg1) << endl << flush;
            }
         }
         else {
            _out << "\t" << code.op << endl << flush;
         }
      }
   }

   return true;
}

void
die2(const char * msg,
    const char * file,
    const int line) 
{
   cerr << msg << endl
        << "file: " << file << endl
        << "line: " << line << endl;

   exit(1);
}

#define die(msg) die2(msg,__FILE__,__LINE__)

int
main(int argc, 
     char ** argv, 
     char ** argenvp)
{
   char * ifile_name;
   char * ofile_name;

   if(argc < 3) 
      die("cc <input file>.c <output file>.asm\n");

   bool opt = false;

   for(int i = 0; i < argc; i++) {
      if(strcmp(argv[i], "-O") == 0
         || strcmp(argv[i], "-O1") == 0
         || strcmp(argv[i], "-O2") == 0
         || strcmp(argv[i], "-O3") == 0) {
         opt = true; 
      }
   }

   file_name = ifile_name = argv[1];
 
   lineno = 1;

   ofile_name = argv[2];

   c_compiler cc(ifile_name, ofile_name);

   //p_c_compiler = &cc;

   int ret = 1;

   if(cc.lex()) {
      if(cc.parse()) {
         cc.optimize();

         if(opt) {
            cc.strong_optimize();
         } 

         if(cc.emit()) {
            ret = 0;
         }
      }
   }

   return ret;
}
