/* SPDX-License-Identifier: GPL-2.0 WITH Linux-syscall-note */
/* This file is derived from the GAS 2.1.4 assembler control file.
   The GAS product is under the GNU General Public License, version 2 or later.
   As such, this file is also under that license.
   If the file format changes in the COFF object, this file should be
   subsequently updated to reflect the changes.
   The actual loader module only uses a few of these structures. The full
   set is documented here because I received the full set. If you wish
   more information about COFF, then O'Reilly has a very excellent book.
*/

#ifndef _UAPI_LINUX_H
#define _UAPI_LINUX_H



#define  E_SYMNMLEN  8   /* Number of characters in a symbol name         */
#define  E_FILNMLEN 14   /* Number of characters in a file name           */
#define  E_DIMNUM    4   /* Number of array dimensions in auxiliary entry */

/*
#define        SYMENT        struct external_syment
#define        SYMESZ        sizeof(SYMENT)
#define        AUXENT        union external_auxent
#define        AUXESZ        sizeof(AUXENT)
*/

#define C_EFCN                0xff        /* physical end of function        */
#define C_NULL                0
#define C_AUTO                1        /* automatic variable                */
#define C_EXT                2        /* external symbol                */
#define C_STAT                3        /* static                        */
#define C_REG                4        /* register variable                */
#define C_EXTDEF        5        /* external definition                */
#define C_LABEL                6        /* label                        */
#define C_ULABEL        7        /* undefined label                */
#define C_MOS                8        /* member of structure                */
#define C_ARG                9        /* function argument                */
#define C_STRTAG        10        /* structure tag                */
#define C_MOU                11        /* member of union                */
#define C_UNTAG                12        /* union tag                        */
#define C_TPDEF                13        /* type definition                */
#define C_USTATIC        14        /* undefined static                */
#define C_ENTAG                15        /* enumeration tag                */
#define C_MOE                16        /* member of enumeration        */
#define C_REGPARM        17        /* register parameter                */
#define C_FIELD                18        /* bit field                        */
#define C_AUTOARG        19        /* auto argument                */
#define C_LASTENT        20        /* dummy entry (end of block)        */
#define C_BLOCK                100        /* ".bb" or ".eb"                */
#define C_FCN                101        /* ".bf" or ".ef"                */
#define C_EOS                102        /* end of structure                */
#define C_FILE                103        /* file name                        */
#define C_LINE                104        /* line # reformatted as symbol table entry */
#define C_ALIAS                 105        /* duplicate tag                */
#define C_HIDDEN        106        /* ext symbol in dmert public lib */


/*
 * These defines are byte order independent. There is no alignment of fields
 * permitted in the structures. Therefore they are declared as characters
 * and the values loaded from the character positions. It also makes it
 * nice to have it "endian" independent.
 */
 
/* Load a short int from the following tables with little-endian formats */
#define SHORT_L(ps) ((short)(((unsigned short)((unsigned char)ps[1])<<8)|\
				  ((unsigned short)((unsigned char)ps[0]))))

/* Load a long int from the following tables with little-endian formats */
#define LONG_L(ps) (((long)(((unsigned long)((unsigned char)ps[3])<<24) |\
				 ((unsigned long)((unsigned char)ps[2])<<16) |\
				 ((unsigned long)((unsigned char)ps[1])<<8)  |\
				 ((unsigned long)((unsigned char)ps[0])))))
 
/* Load a short int from the following tables with big-endian formats */
#define SHORT_H(ps) ((short)(((unsigned short)((unsigned char)ps[0])<<8)|\
				  ((unsigned short)((unsigned char)ps[1]))))

/* Load a long int from the following tables with big-endian formats */
#define LONG_H(ps) (((long)(((unsigned long)((unsigned char)ps[0])<<24) |\
				 ((unsigned long)((unsigned char)ps[1])<<16) |\
				 ((unsigned long)((unsigned char)ps[2])<<8)  |\
				 ((unsigned long)((unsigned char)ps[3])))))

/* These may be overridden later by brain dead implementations which generate
   a big-endian header with little-endian data. In that case, generate a
   replacement macro which tests a flag and uses either of the two above
   as appropriate. */

#define LONG(v)   LONG_L(v)
#define SHORT(v)  SHORT_L(v)

/*** coff information for Intel 386/486.  */

/********************** FILE HEADER **********************/

struct filehdr {
	short f_magic;	/* magic number			*/
	short f_nscns;	/* number of sections		*/
	int f_timdat;	/* time & date stamp		*/
	int f_symptr;	/* file pointer to symtab	*/
	int f_nsyms;	/* number of symtab entries	*/
	short f_opthdr;	/* sizeof(optional hdr)		*/
	short f_flags;	/* flags			*/
};

#define FILHDR struct filehdr
#define FILSZ sizeof(FILHDR)

/*
 *   Bits for f_flags:
 *
 *	F_RELFLG	relocation info stripped from file
 *	F_EXEC		file is executable  (i.e. no unresolved external
 *			references)
 *	F_LNNO		line numbers stripped from file
 *	F_LSYMS		local symbols stripped from file
 *	F_MINMAL	this is a minimal object file (".m") output of fextract
 *	F_UPDATE	this is a fully bound update file, output of ogen
 *	F_SWABD		this file has had its bytes swabbed (in names)
 *	F_AR16WR	this file has the byte ordering of an AR16WR
 *			(e.g. 11/70) machine
 *	F_AR32WR	this file has the byte ordering of an AR32WR machine
 *			(e.g. vax and iNTEL 386)
 *	F_AR32W		this file has the byte ordering of an AR32W machine
 *			(e.g. 3b,maxi)
 *	F_PATCH		file contains "patch" list in optional header
 *	F_NODF		(minimal file only) no decision functions for
 *			replaced functions
 */

#define  F_RELFLG		0000001
#define  F_EXEC		0000002
#define  F_LNNO		0000004
#define  F_LSYMS		0000010
#define  F_MINMAL		0000020
#define  F_UPDATE		0000040
#define  F_SWABD		0000100
#define  F_AR16WR		0000200
#define  F_AR32WR		0000400
#define  F_AR32W		0001000
#define  F_PATCH		0002000
#define  F_NODF		0002000

#define	I386MAGIC	        0x14c   /* Linux's system    */

#if 0   /* Perhaps, someday, these formats may be used.      */
#define 386PTXMAGIC	0x154
#define I386AIXMAGIC	0x175   /* IBM's AIX system  */
#define I386BADMAG(x) ((SHORT((x).f_magic) != I386MAGIC) \
			  && SHORT((x).f_magic) != I386PTXMAGIC \
			  && SHORT((x).f_magic) != I386AIXMAGIC)
#else
#define I386BADMAG(x) (SHORT((x).f_magic) != I386MAGIC)
#endif

#define	FILHDR	struct filehdr
#define	FILHSZ	sizeof(FILHDR)

/********************** AOUT "OPTIONAL HEADER" **********************/

/* Linux COFF must have this "optional" header. Standard COFF has no entry
   location for the "entry" point. They normally would start with the first
   location of the .text section. This is not a good idea for linux. So,
   the use of this "optional" header is not optional. It is required.
   Do not be tempted to assume that the size of the optional header is
   a constant and simply index the next byte by the size of this structure.
   Use the 'f_opthdr' field in the main coff header for the size of the
   structure actually written to the file!!
*/

typedef struct 
{
  char 	magic[2];		/* type of file				 */
  char	vstamp[2];		/* version stamp			 */
  char	tsize[4];		/* text size in bytes, padded to FW bdry */
  char	dsize[4];		/* initialized   data "   "		 */
  char	bsize[4];		/* uninitialized data "   "		 */
  char	entry[4];		/* entry pt.				 */
  char 	text_start[4];		/* base of text used for this file       */
  char 	data_start[4];		/* base of data used for this file       */
}
AOUTHDR;

#define AOUTSZ (sizeof(AOUTHDR))

#define STMAGIC	0401
#define OMAGIC     0404
#define JMAGIC     0407    /* dirty text and data image, can't share  */
#define DMAGIC     0410    /* dirty text segment, data aligned        */
#define ZMAGIC     0413    /* The proper magic number for executables  */
#define SHMAGIC	0443	/* shared library header                   */

/********************** SECTION HEADER **********************/

struct scnhdr {
  int s_name;	/* section name			    */
  int s_paddr;	/* physical address, aliased s_nlib */
  int s_vaddr;	/* virtual address		    */
  int s_size;	/* section size			    */
  int s_scnptr;	/* file ptr to raw data for section */
  int s_relptr;	/* file ptr to relocation	    */
  int s_lnnoptr;	/* file ptr to line numbers	    */
  short s_nreloc;	/* number of relocation entries	    */
  short s_nlnno;	/* number of line number entries    */
  int s_flags;	/* flags			    */
};

#define	SCNHDR	struct scnhdr
#define	SCNHSZ	sizeof(SCNHDR)

/*
 * names of "special" sections
 */

#define TEXT	".text"
#define DATA	".data"
#define BSS	".bss"
#define COMMENT    ".comment"
#define LIB        ".lib"

#define SECT_TEXT  0      /* Section for instruction code             */
#define SECT_DATA  1      /* Section for initialized globals          */
#define SECT_BSS   2      /* Section for un-initialized globals       */
#define SECT_REQD  3      /* Minimum number of sections for good file */

#define STYP_REG     0x00 /* regular segment                          */
#define STYP_DSECT   0x01 /* dummy segment                            */
#define STYP_NOLOAD  0x02 /* no-load segment                          */
#define STYP_GROUP   0x04 /* group segment                            */
#define STYP_PAD     0x08 /* .pad segment                             */
#define STYP_COPY    0x10 /* copy section                             */
#define STYP_TEXT    0x20 /* .text segment                            */
#define STYP_DATA    0x40 /* .data segment                            */
#define STYP_BSS     0x80 /* .bss segment                             */
#define STYP_INFO   0x200 /* .comment section                         */
#define STYP_OVER   0x400 /* overlay section                          */
#define STYP_LIB    0x800 /* library section                          */

/*
 * Shared libraries have the following section header in the data field for
 * each library.
 */

struct external_slib {
  int sl_entsz;	/* Size of this entry               */
  int sl_pathndx;	/* size of the header field         */
};

#define	SLIBHD	struct slib
#define	SLIBSZ	sizeof(SLIBHD)

/********************** LINE NUMBERS **********************/

/* 1 line number entry for every "breakpointable" source line in a section.
 * Line numbers are grouped on a per function basis; first entry in a function
 * grouping will have l_lnno = 0 and in place of physical address will be the
 * symbol table index of the function name.
 */

struct external_lineno {
  union {
    char l_symndx[4];	/* function name symbol index, iff l_lnno == 0*/
    char l_paddr[4];	/* (physical) address of line number	*/
  } l_addr;
  char l_lnno[2];	/* line number		*/
};

#define	LINENO	struct lineno
#define	LINESZ	6

/********************** SYMBOLS **********************/

#define E_SYMNMLEN	 8	/* # characters in a short symbol name	*/
#define E_FILNMLEN	14	/* # characters in a file name		*/
#define E_DIMNUM	 4	/* # array dimensions in auxiliary entry */

/*
 *  All symbols and sections have the following definition
 */

struct external_syment 
{
  union {
    char e_name[E_SYMNMLEN];    /* Symbol name (first 8 characters) */
    struct {
      int e_zeroes;         /* Leading zeros */
      int e_offset;         /* Offset if this is a header section */
    } e;
  } e;
  int e_value;              /* Value (address) of the segment */
  short e_scnum;              /* Section number */
  short e_type;               /* Type of section */
  char e_sclass;             /* Loader class */
  char e_numaux;             /* Number of auxiliary entries which follow */
};

#define N_BTMASK	(0xf)   /* Mask for important class bits */
#define N_TMASK	(0x30)  /* Mask for important type bits  */
#define N_BTSHFT	(4)     /* # bits to shift class field   */
#define N_TSHIFT	(2)     /* # bits to shift type field    */

/*
 *  Auxiliary entries because the main table is too limiting.
 */
  
union external_auxent {

/*
 *  Debugger information
 */

  struct {
    int x_tagndx;	        /* str, un, or enum tag indx */
    union {
      struct {
	short x_lnno;        /* declaration line number */
	short x_size;        /* str/union/array size */
      } x_lnsz;
      int x_fsize;	        /* size of function */
    } x_misc;

    union {
      struct {		        /* if ISFCN, tag, or .bb */
	int x_lnnoptr;	/* ptr to fcn line # */
	int x_endndx;	/* entry ndx past block end */
      } x_fcn;

      struct {		        /* if ISARY, up to 4 dimen. */
	char x_dimen[E_DIMNUM][2];
      } x_ary;
    } x_fcnary;

    short x_tvndx;	/* tv index */
  } x_sym;

/*
 *   Source file names (debugger information)
 */

  union {
    char x_fname[E_FILNMLEN];
    struct {
      int x_zeroes;
      int x_offset;
    } x_n;
  } x_file;

/*
 *   Section information
 */

  struct {
    int x_scnlen;	/* section length */
    short x_nreloc;	/* # relocation entries */
    short x_nlinno;	/* # line numbers */
  } x_scn;

/*
 *   Transfer vector (branch table)
 */
  
  struct {
    int x_tvfill;	/* tv fill value */
    short x_tvlen;	/* length of .tv */
    char x_tvran[2][2];	/* tv range */
  } x_tv;		/* info about .tv section (in auxent of symbol .tv)) */
};

#define	SYMENT	struct external_syment
#define	SYMESZ	18	
#define	AUXENT	union external_auxent
#define	AUXESZ	18

#define ETEXT	"etext"

/********************** RELOCATION DIRECTIVES **********************/

struct external_reloc {
  int r_vaddr;        /* Virtual address of item    */
  int r_symndx;       /* Symbol index in the symtab */
  short r_type;         /* Relocation type            */
};

#define RELOC struct external_reloc
#define RELSZ 10

#define DEF_DATA_SECTION_ALIGNMENT  4
#define DEF_BSS_SECTION_ALIGNMENT   4
#define DEF_TEXT_SECTION_ALIGNMENT  4

/* For new sections we haven't heard of before */
#define DEF_SECTION_ALIGNMENT       4

#endif /* _UAPI_LINUX_H */
