/**
 *
 * Hand written lexer for the qP+ grammer.
 *
 * Author: Matthew W. Coan
 * Tue Mar  4 17:02:40 EST 2003
 */

#include "y.tab.h"
#include <string.h>
#include <ctype.h>
#include <stdio.h>

/* Input buffer max size. */
#define MAX_BUFFER 1024

/* Input stream. */
FILE * yyin;

/* Line number. */
int yylineno = 1;

/* The input buffer. */
char yytext[MAX_BUFFER];

/*
 * The scanner:
 *    Read from the input stream
 *    and output the next token.
 */
int 
yylex()
{
   int ch;
   int i;

   ch = fgetc(yyin);

   if(ch == EOF)
      return EOF;

   while(ch != EOF) {
      /* Clear the buffer. */
      memset(yytext, 0, MAX_BUFFER);

      /* Read an ID. */
      if(isalpha(ch) || ch == '_') {
         yytext[0] = ch;
         i = 1;
         while((ch = fgetc(yyin)) != EOF 
               && (isalnum(ch) || ch == '_')
               && (i < (MAX_BUFFER - 1))) {
            yytext[i] = ch;
            i++;
         }
         if(ch != EOF)
            ungetc(ch, yyin); 

         /* Check to see if it is a key word. */
         if(strcmp("MAIN", yytext) == 0)
            return QP_MAIN;
         else if(strcmp("DO", yytext) == 0)
            return QP_DO;
         else if(strcmp("VAR", yytext) == 0)
            return QP_VAR;
         else if(strcmp("NEW", yytext) == 0)
            return QP_NEW;
         else if(strcmp("READ", yytext) == 0)
            return QP_READ;
         else if(strcmp("WRITE", yytext) == 0)
            return QP_WRITE;
         else if(strcmp("IF", yytext) == 0)
            return QP_IF;
         else if(strcmp("WHILE", yytext) == 0)
            return QP_WHILE;
         else if(strcmp("STRING", yytext) == 0)
            return QP_STRING_NAME;
         else if(strcmp("CHAR", yytext) == 0)
            return QP_CHAR_NAME;
         else if(strcmp("INT", yytext) == 0)
            return QP_INT_NAME;
         else if(strcmp("BOOLEAN", yytext) == 0)
            return QP_BOOLEAN_NAME;
         else if(strcmp("TRUE", yytext) == 0)
            return QP_TRUE;
         else if(strcmp("FALSE", yytext) == 0)
            return QP_FALSE;
         else if(strcmp("NOT", yytext) == 0)
            return QP_NOT;
         else {
            fprintf(stdout, "%s\n", yytext);
            fflush(stdout);
            return QP_NAME;
         }
      }
      /* Read an integer. */
      else if(isdigit(ch)) {
         yytext[0] = ch;
         i = 1;
         while((ch = fgetc(yyin)) != EOF
               && isdigit(ch)
               && (i < (MAX_BUFFER - 1))) {
            yytext[i] = ch;
            i++;
         }
         if(ch != EOF)
            ungetc(ch, yyin);
         fprintf(stdout, "%s\n", yytext);
         fflush(stdout);
         return QP_INT;
      }
      /* Read a character. */
      else if(ch == '\'') {
         ch = fgetc(yyin);
         i = fgetc(yyin);
         if(i != '\'') 
            return -1;
         fprintf(stdout, "%c\n", ch);
         fflush(stdout);
         return QP_CHAR;
      }
      /* Read a string constant. */
      else if(ch == '\"') {
         i = 0;
         while(((ch = fgetc(yyin)) != EOF)
               && (ch != '\"')
               && (i < (MAX_BUFFER - 1))) {
            yytext[i] = ch;
            i++;
         }
         if(ch != '\"')
            return -1;
         fprintf(stdout, "%s\n", yytext);
         fflush(stdout);
         return QP_STRING;
      }
      /* Read an operator. */
      else if(ispunct(ch)) {
         switch(ch) {
            case '<':
               ch = fgetc(yyin);
               if(ch == '=')
                  return QP_LTEQ;
               if(ch == '>')
                  return QP_NEQ;
               ungetc(ch, yyin);
               return '<';
               break;
 
            case '>':
               ch = fgetc(yyin);
               if(ch == '=')
                  return QP_LTEQ;
               ungetc(ch, yyin);
               return '>';
               break;

            case '=':
               ch = fgetc(yyin);
               if(ch == '=')
                  return QP_EQ;
               ungetc(ch, yyin);
               return '=';
               break;

            case '/':
               ch = fgetc(yyin);
               if(ch == '/') {
                  while((ch = fgetc(yyin)) != '\n' && ch != EOF)
                     ;
                  ch = fgetc(yyin);
                  yylineno++;
                  break;
               }
               return '/';

            case '{':
            case '}':
            case '-':
            case '+':
            case '*':
            case '%':
            case ';':
            case ':':
            case '$':
            case ',':
            case '(':
            case ')':
               return ch;

            default:
               return -1;
         }
      }
      /* Skip white space. */
      else {
         if(ch == '\n')
            yylineno++;
         ch = fgetc(yyin);
      }
   }

   return EOF;
}

/**
 * Main entry point:
 */
int 
main(int argc, char ** argv)
{
   FILE * fin;

   /* Check the command line. */
   if(argc != 2) {
      fprintf(stderr, "Usage: %s <input file>\n", argv[0]);
      return -1;
   }

   /* Open the file for reading. */
   fin = fopen(argv[1], "r");

   /* Check for an error. */
   if(fin == NULL) {
      fprintf(stderr, "Unable to open file for reading: %s\n", argv[1]);
      return -1;
   }

   /* Set the global file pointer. */
   yyin = fin;

   /* Parse the file. */
   yyparse();

   /* Close the input stream. */
   fclose(fin);

   return 0;
}

/* Handle an error. */
int 
yyerror(const char * msg) 
{
   printf("%d: %s at '%s'\n", yylineno, msg, yytext);
}
