#include "CList.h"
#include "CSVFileReader.h"
#include "_assert.h"

int CSVFileReaderColNameComp(const void * ptr1, const void * ptr2) 
{
   CSVFileReaderColName * pName1 = *(CSVFileReaderColName**)(ptr1);
   CSVFileReaderColName * pName2 = *(CSVFileReaderColName**)(ptr2);
   return strcmp(pName1->name, pName2->name);
}

/* Create a CSVFileReader. This function returns NULL if an error occurs. */
CSVFileReader * CSVFileReaderCreate(char * fileName)
{
   size_t i;
   CSVFileReaderColName * pColName;
   CSVFileReader * pRet = (CSVFileReader*)malloc(sizeof(CSVFileReader));
   Assert(pRet != NULL);
   memset(pRet, 0, sizeof(CSVFileReader));
   pRet->file = fopen(fileName, "r");
   if(pRet->file == NULL) {
      free(pRet);
      return NULL;
   }
   pRet->colValueArray = NULL;
   if(!(pRet->numCols = CSVFileReaderReadLine(pRet))) {
      fclose(pRet->file);
      free(pRet);
      return NULL;
   }
   pRet->colNameArray = (CSVFileReaderColName**)malloc(sizeof(CSVFileReaderColName*) * pRet->numCols);
   Assert(pRet->colNameArray != NULL);
   memset(pRet->colNameArray, 0, sizeof(CSVFileReaderColName*) * pRet->numCols);
   for(i = 0; i < pRet->numCols; i++) {
      pColName = (CSVFileReaderColName*)malloc(sizeof(CSVFileReaderColName));
      Assert(pColName != NULL);
      memset(pColName, 0, sizeof(CSVFileReaderColName));
      pColName->name = pRet->colValueArray[i];
      pColName->index = i;
      pRet->colNameArray[i] = pColName;
   }
   free(pRet->colValueArray);
   pRet->colValueArray = NULL;
   qsort(pRet->colNameArray, pRet->numCols, sizeof(CSVFileReaderColName*), CSVFileReaderColNameComp);
   return pRet;
}

/* Destory a CSVFileReader. */
void CSVFileReaderDestroy(CSVFileReader * pReader)
{
   size_t i;
   CSVFileReaderColName * pColName;
   
   fclose(pReader->file);
   if(pReader->colValueArray != NULL) {
      for(i = 0; i < (pReader->numCols); i++) {
         free(pReader->colValueArray[i]);
      }
      free(pReader->colValueArray);
      pReader->colValueArray = NULL;
   }   
   if(pReader->colNameArray != NULL) {
      for(i = 0; i < pReader->numCols; i++) {
         pColName = (CSVFileReaderColName*)pReader->colNameArray[i];
         free(pColName->name);
         free(pColName);
      }
      free(pReader->colNameArray);
      pReader->colNameArray = NULL;
   }
   free(pReader);
}

/* Free a strings memory. */
void CSVFileReaderValueDestroy(void * pValue)
{
   free(pValue);
}

int peek(FILE * file)
{
   int ch = fgetc(file);
   ungetc(ch, file);
   return ch;
}

/* Read in the value list.  Return the number of cols read. */
int CSVFileReaderReadValueList(CSVFileReader * pReader, 
                               CList * pValueList)
{
   int ret;
   char ch, * p, * p2, oldCh;
   CList * pList;
   CListNode * pNode;
   int gotQ;
   int doBreak;
   int gotOneMore;
   int addOneMore;
   addOneMore = ret = doBreak = 0;

   do {
      pList = CListCreate(CSVFileReaderValueDestroy);
      if(feof(pReader->file)) {
         CListDestroy(pList);
         break;
      }
      ch = fgetc(pReader->file);
      if(feof(pReader->file)) {
         CListDestroy(pList);
         addOneMore = 1;
         continue;
      }
      if(ch == '\"') {
         gotQ = 1;
         ch = fgetc(pReader->file);
      }
      else
         gotQ = 0;

      while(!feof(pReader->file)) {
         if(gotQ && ch == '\"' && peek(pReader->file) != '\"')
            break;
         else if(gotQ && ch == '\"' && peek(pReader->file) == '\"')
            ch = fgetc(pReader->file);
         else if(!gotQ && (ch == ',' || ch == '\r' || ch == '\n')) {
            break;
         }
         p = (char*)malloc(sizeof(char));
         Assert(p != NULL);
         *p = ch;
         CListPushTail(pList, p);
         ch = fgetc(pReader->file);
      }
      if(gotQ && !feof(pReader->file)) {
         ch = fgetc(pReader->file);
         if(ch != ',') {
            while(!feof(pReader->file) && (ch == '\r' || ch == '\n')) {
               oldCh = ch;
               ch = fgetc(pReader->file);
               if(ch != '\r' && ch != '\n') {
                  ungetc(ch, pReader->file);
                  ch = oldCh;
                  break;
               }
            }
            if(ch != '\r' && ch != '\n')
               ungetc(ch, pReader->file);
            doBreak = 1;
         }
      }
      else if(!gotQ && !feof(pReader->file)) {
         gotOneMore = 0;
         if(ch == ',') {
            ch = fgetc(pReader->file);
            if(ch == '\r' || ch == '\n')
               gotOneMore = 1;
            else if(feof(pReader->file)) {
               gotOneMore = addOneMore = 1;
            }
         }
         if(gotOneMore == 0) {
            while(ch != ',' && !feof(pReader->file) && (ch == '\r' || ch == '\n')) {
               oldCh = ch;
               ch = fgetc(pReader->file);
               if(ch != '\r' && ch != '\n') {
                  ungetc(ch, pReader->file);
                  ch = oldCh;
                  break;
               }
            }
            if(ch != '\r' && ch != '\n')
               ungetc(ch, pReader->file);
            else
               doBreak = 1;
         }
         else {
            ungetc(ch, pReader->file);
         }
      }

      p = (char*)malloc(CListSize(pList)+1);
      Assert(p != NULL);
      memset(p, 0, CListSize(pList)+1);
      p2 = p;
      for(pNode = CListBegin(pList); 
          pNode != CListEnd(pList); 
          pNode = pNode->next) {
         *p2 = (*((char*)(pNode->data)));
         p2++;
      }
      CListPushTail(pValueList, p);
      ret++;
      CListDestroy(pList);
   }
   while(!feof(pReader->file) && !doBreak);

   if(addOneMore) {
      p = (char*)malloc(1);
      Assert(p != NULL);
      *p = '\0';
      CListPushTail(pValueList, p);
      ret++;
   }
   return ret;
}

/* Read the next line of the CSV file.  Return 1 if OK else 0. */
int CSVFileReaderReadLine(CSVFileReader * pReader)
{
   size_t i;
   size_t ret;
   CList * pValueList;
   CListNode * pNode;
   
   ret = 0;
   if(pReader->colValueArray != NULL) {
      for(i = 0; i < pReader->numCols; i++) {
         free(pReader->colValueArray[i]);
      }
      free(pReader->colValueArray);
      pReader->colValueArray = NULL;
   }

   pValueList = CListCreate(CSVFileReaderValueDestroy);
   ret = CSVFileReaderReadValueList(pReader, pValueList);
   if(ret > 0) {
      pReader->colValueArray = (char**)malloc(sizeof(char*) * CListSize(pValueList));
      Assert(pReader->colValueArray != NULL);
      memset(pReader->colValueArray, 0, sizeof(char*) * CListSize(pValueList));
      for(i = 0, pNode = CListBegin(pValueList); 
          i < CListSize(pValueList); 
          i++, pNode = pNode->next) {
         pReader->colValueArray[i] = strdup((char*)pNode->data);
      }
      CListDestroy(pValueList);
   }
   return ret;
}

/* Get a value associated with a column name for the current line. */
char * CSVFileReaderGetValue(CSVFileReader * pReader, char * colName)
{
   void * ptr;
   CSVFileReaderColName * pColName;
   CSVFileReaderColName * pKey;

   if(pReader->colValueArray == NULL || pReader->colNameArray == NULL)
      return NULL;
   pKey = (CSVFileReaderColName*)malloc(sizeof(CSVFileReaderColName));
   Assert(pKey != NULL);
   memset(pKey, 0, sizeof(CSVFileReaderColName));
   pKey->name = colName;
   pKey->index = -1;
   ptr = bsearch(&pKey, pReader->colNameArray, pReader->numCols, sizeof(CSVFileReaderColName*), CSVFileReaderColNameComp);
   free(pKey);
   if(ptr == NULL)
      return NULL;
   pColName = *(CSVFileReaderColName**)ptr;
   return pReader->colValueArray[pColName->index];
}


