package com.mwc.util;

import java.io.*;
import java.util.*;

/**
 * A class to represent a template web page for dynamic HTML 
 * content.
 *
 * @autorh Matthew W. Coan
 * 11/10/2000
 */
public class TemplateWebPage {
   // The template vector of strings (for reset method).
   private Vector _template;

   // The content vector of strings containting 
   // the dynamicly generated HTML.
   protected Vector _content;

   // The file name of the template.
   private String _fileName;

   // The last modifyed date.
   private long _lastModifyed;

   /**
    * Construct a template web page.
    */
   public TemplateWebPage() {
      _template = new Vector();
      _content = new Vector();
      _fileName = null;
   }

   /**
    * Construct a template web page out of
    * a preexisting template web page (copy constructor).
    * @param page a preexisting web page template.
    */
   public TemplateWebPage(TemplateWebPage page) {
      synchronized(page) {
         _template = new Vector();
         _content = new Vector();
         for(int i = 0; i < page._template.size(); i++)
            _template.addElement(new String(page._template.elementAt(i).toString()));
         for(int i = 0; i < page._content.size(); i++)
            _content.addElement(new String(page._content.elementAt(i).toString()));
      }
   }

   /**
    * Load a template file into this template
    * web page.
    * @param fileName the filename of the template
    *                 web page to load.
    */
   public void loadTemplate(String fileName) 
   throws IOException {
      _fileName = new String(fileName);
      _template.removeAllElements();
      _content.removeAllElements();
      File f = new File(fileName);
      _lastModifyed = f.lastModified();
      FileReader fin = new FileReader(fileName);
      BufferedReader bin = new BufferedReader(fin);
      String line = null;
      while((line = bin.readLine()) != null) {
         _template.addElement(line);
         _content.addElement(new String(line));
      }
      fin.close();
   }

   /**
    * Expand a template with a given value.
    * @param name the name of the template to expand 
    *             like &lt;&lt;date>>.
    * @param value the value to replace the template 
    *              name with (the date string).
    */
   public void expandTemplate(String name, String value) {
      StringBuffer content = null;
      String line;
      int index, start;
      boolean expand;
      for(int i = 0; i < _content.size(); i++) {
         content = new StringBuffer();
         line = (String)_content.elementAt(i);
         start = 0;
         do {
            index = line.indexOf(name, start);
            if(index != -1) {
               content.append(line.substring(start, index));
               content.append(value);
               start = index + name.length();
               expand = true;
            }
            else {
               content.append(line.substring(start, line.length()));
               expand = false;
            }
         }
         while(expand);
         _content.setElementAt(content.toString(), i);
      }
   }

   /**
    * Reset the content string to the template string
    * (for re-expansion of the template). 
    */
   public void resetToTemplate() {
      _content = new Vector();
      for(int i = 0; i < _template.size(); i++)
         _content.addElement(new String(_template.elementAt(i).toString()));
   }

   /**
    * Reload the template if its last modifyed date has changed.
    */
   public void reloadTemplate() throws IOException {
      if(_fileName == null)
         return;
      synchronized(this) {
         File f = new File(_fileName);
         long lastModifyed = f.lastModified();
         if(_lastModifyed < lastModifyed)
            loadTemplate(_fileName);
      }
   }

   /**
    * Output the web page to a print writer.
    * @param out a PrintWriter to output the web page to.
    */
   public void sendTo(PrintWriter out) throws IOException {
      for(int i = 0; i < _content.size(); i++) 
         out.println((String)_content.elementAt(i));
      out.flush();
   }

   /**
    * Overides Object.toString()
    */
   public String toString() {
      StringBuffer buf = new StringBuffer();
      for(int i = 0; i < _content.size(); i++)
         buf.append(_content.elementAt(i).toString() + "\n");
      return buf.toString();
   }
}