// httpd.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"
/*

    Copyright (C) 2009  Matthew William Coan

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

This file is the intellectual property of
Matthew William Coan.

Simple ultra small HTTP server for Windows, Linux and BSD
using standard C++...

Author: Matthew William Coan
Date: Fri May  7 23:23:20 EDT 2010

*/

#define WIN32_HTTPD
//#define LINUX_HTTPD

#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <strstream>

#include <cstdlib>
#include <cstring>
#include <cctype>

using namespace std;

#ifdef LINUX_HTTPD
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#define SOCKET int
#define closesocket(a) close(a)
#endif 

#ifdef WIN32_HTTPD

#include <winsock2.h>

WSADATA stWSAData;  // WinSock DLL Info.

#define WSA_MAJOR_VERSION 1
#define WSA_MINOR_VERSION 1
#define WSA_VERSION MAKEWORD(WSA_MAJOR_VERSION, WSA_MINOR_VERSION)

#define popen _popen
#define pclose _pclose

#define socklen_t int

#endif

#include <assert.h>

#ifdef LINUX_HTTPD
#define HTTP_ROOT "/home/mcoan/commands"
#define CAT_PROG "cat"
#endif

#ifdef WIN32_HTTPD
#define HTTP_ROOT "C:\\wwwroot"
#define CAT_PROG "type"
#endif

#define HTTP_PORT 8080
#define HTTP_HOST "127.0.0.1"
#define DEFAULT_PAGE "index.html"

#define HTTP_400 "/response/400.html"
#define HTTP_404 "/response/404.html"
#define HTTP_500 "/response/500.html"

class httpd {
public:
   typedef vector< string > string_vector_type; 

   enum response_code { 
      BAD_REQUEST=400,
      FILE_NOT_FOUND=404,
      OK=200,
      LOCATION=201,
      INTERNAL_SERVER_ERROR=500
   };

private:
   string ip_address;
   int port;

public:
   httpd(const string & addr,
         const int the_port)
   {
      ip_address = addr;
      port = the_port;
   }

   httpd(const httpd & cp)
   {
      ip_address = cp.ip_address;
      port = cp.port;
   }

   httpd()
   {
      port = HTTP_PORT;
      ip_address = HTTP_HOST;
   }

   virtual ~httpd()
   {
   }

   httpd & operator=(const httpd & right)
   {
      ip_address = right.ip_address;
      port = right.port;
      return *this;
   }
/*
   const char * strstr(const char * str, const char * se)
   {
      const char * ret = 0;
      int n;

      for(size_t i = 0; i < strlen(str); i++) {
         n = 0;
         
         for(size_t j = 0; j < strlen(se); j++) {
            if(tolower(str[i]) == tolower(se[j])) {
               n++;
            }
         }

         if(n == strlen(se)) {
            ret = &str[i];
            break;
         }
      }

      return ret;
   }
*/
   bool read_line(int fd, string & line)
   {
      bool ret = false;
      char ch = '\0';
      size_t the_size = 0;
      const int MAX_LENGTH = 1024 * 10;

      while(recv(fd, &ch, 1, 0) == 1) {
          if(ch == '\n') {
             ret = true;
             break;
          }

          if(ch == '\r') {
             continue;
          }

          the_size++;

          if(the_size >= MAX_LENGTH) {
             break;
          }

          line += ch;
      }

      return ret;
   }

   bool is_valid(const string & str)
   {
      bool ret = true;

      for(size_t i = 0; i < str.size(); i++) {
         if((str[i] >= 'a' && str[i] <= 'z')
            || (str[i] >= 'A' && str[i] <= 'Z')
            || (str[i] >= '0' && str[i] <= '9')
            || (str[i] == '-')
            || (str[i] == '.')
            || (str[i] == '_')
            || (str[i] == '.')
            || (str[i] == '&')
            || (str[i] == '%')
            || (str[i] == '=')
            || (str[i] == '?')
            || (str[i] == '/')) {
            if((i+1) < str.size()) {
               if(str[i+1] == '.' && str[i] == '.') {
                  ret = false;
                  break;
               }
            }
         }
         else {
            ret = false;
            break;
         }
      }

      return ret;
   }

   string_vector_type read_request(int fd)
   {
      string line;
      string_vector_type vec;
      size_t the_size = 0;
      const int MAX_LENGTH = 1024 * 10;

      while(read_line(fd, line)) {
         if(line.size() == 0) {
            break;
         }

         the_size += line.length();

         if(the_size >= MAX_LENGTH) {
            break; 
         }

         cout << "HTTP_HEADER[" << line << "]" << endl << flush;

         vec.push_back(line);

         line = "";
      }

      return vec;
   }

   void send_file(int fd, const char * file, const int response_code)
   {
      const char * header;
      
      switch(response_code) {
      case BAD_REQUEST:
         header = "HTTP/1.0 400 Bad Request\r\nContent-type: text/html\r\n\r\n";
         break;

      case FILE_NOT_FOUND:
         header = "HTTP/1.0 404 File Not Found\r\nContent-type: text/html\r\n\r\n";
         break;

      case INTERNAL_SERVER_ERROR:
         header = "HTTP/1.0 500 Internal Server Error\r\nContent-type: text/html\r\n\r\n";
         break;

      case OK:
         header = "HTTP/1.0 200 OK\r\nContent-type: text/html\r\n\r\n";
         break;

      default:
         return;
      }

      send(fd, header, strlen(header), 0);

      ifstream in(file);

      if(in) {
         const size_t BUFFER_SIZE = 1024;
         char buffer[BUFFER_SIZE];

         memset(buffer, 0, BUFFER_SIZE);

         in.read(buffer, BUFFER_SIZE-1);

         while(in) {
            send(fd, buffer, strlen(buffer), 0);

            memset(buffer, 0, BUFFER_SIZE);
         
            in.read(buffer, BUFFER_SIZE-1);
         }

         in.close();
      }
   }

   void error(int fd, int code)
   {
      string temp;

      switch(code) {
      case BAD_REQUEST:
         send_file(fd, HTTP_400, BAD_REQUEST);
         break;

      case FILE_NOT_FOUND:
         send_file(fd, HTTP_404, FILE_NOT_FOUND);
         break;

      case INTERNAL_SERVER_ERROR:
         send_file(fd, HTTP_500, INTERNAL_SERVER_ERROR);
         break;
      }

      exit(0);
   }

   string get_query_string(const string & str)
   {
      string ret;
      int i;

      for(i = 0; i < str.size(); i++) {
         if(str[i] == '?') {
            i++;
            break;
         } 
      }

      while(i < str.size()) {
         ret += str[i];

         i++;
      }

      return ret;
   }

   string get_URI(const string & str)
   {
      string ret;

      for(int i = 0; i < str.size(); i++) {
         if(str[i] == '?') {
            break; 
         }

         ret += str[i];
      }

      return ret;
   }

   void run_CGI(int fd, const string & file)
   {
      string qs = get_query_string(file);

      string prog = get_URI(file);
/*
#ifdef LINUX_HTTPD
      prog = "export QUERY_STRING=\"" + qs + "\";"
             + "export REQUEST_METHOD=\"GET\";"
             + prog;
#endif
*/
      string env = string("QUERY_STRING=") + qs;
      putenv(env.c_str());
      putenv("REQUEST_METHOD=GET");

      FILE * program = popen(prog.c_str(), "rw");

      string data;

      if(program) {
         char ch;

         while((ch = fgetc(program)) != EOF) {
            data += ch;
         }

         pclose(program);
      }
      else {
         error(fd, BAD_REQUEST);
      }

      if(data.find("Location: http://") == 0) {
         data = string("HTTP/1.0 301 Moved Permanently\r\n") 
                + data;
         send(fd, data.c_str(), data.size(), 0);
      }
      else if(data.find("Location: /") == 0) {
         data = string("HTTP/1.0 301 Moved Permanently\r\n") 
                + data;
         send(fd, data.c_str(), data.size(), 0);
      }
      else {
         data = string("HTTP/1.0 200 OK\r\n") 
                + data;
         send(fd, data.c_str(), data.size(), 0);
      }
   }

   string to_string(const size_t size)
   {
      string ret = "";

      const size_t SZ = 1024;
      
      char buffer[SZ];

      memset(buffer, 0, SZ);
      
      strstream sout(buffer, ios::out);

      sout << size;

      ret = string(buffer);

      return ret;
   }

   void get_file(int fd, const string & file, const string & ct)
   {
      string ret;
      char ch;

      ifstream in(file.c_str());

      if(in) {
         ch = in.get();

         while(in) {
            ret += ch;

            ch = in.get();
         }

         in.close();
      }
      else {
         error(fd, FILE_NOT_FOUND);
      }

      ret = string("HTTP/1.0 200 OK\r\n")
            + string("Content-type: ") + ct + string("\r\n\r\n")
            + ret;

      send(fd, ret.c_str(), ret.size(), 0);
   }

   void send_response(int fd, const string & data)
   {
      send(fd, data.c_str(), data.size(), 0);
   }

   string get_request_string(const string & cmd)
   {
      int i = 0;

      while(cmd[i] != '/') {
         i++;

         if(i >= cmd.size()) {
            break;
         }
      }

      string file;

      while(cmd[i] != ' ') {
         file += cmd[i];
         i++;

         if(i >= cmd.size() || cmd[i] == '?') {
            break;
         }
      }

      return file;
   }

   string to_win32(const string & s)
   {
      string ret;

	  for(int i = 0; i < s.size(); i++) {
		  if(s[i] == '/') {
	         ret += "\\";
		  }
		  else {
			 ret += s[i];
		  }
	  }

	  return ret;
   }

   void process_GET(int fd, string & cmd)
   {
      string response;
      string file = get_request_string(cmd);

      if(!is_valid(file)) {
         error(fd, BAD_REQUEST);
         return;
      }

      cout << "URI: \"" << file << "\"" << endl << flush;

      if(file.size()) {
         if(file[file.size()-1] == '/') {
            file += DEFAULT_PAGE;
         }
      }
 
#ifdef WIN32_HTTPD
      file = HTTP_ROOT + to_win32(file);
#endif

#ifdef LINUX_HTTPD
      file = HTTP_ROOT + file;
#endif

      cout << "FULL PATH: \"" << file << "\"" << endl << flush;

      if(strstr(file.c_str(), "?") != NULL
        || strstr(file.c_str(), ".pl") != NULL
        || strstr(file.c_str(), ".php") != NULL
        || strstr(file.c_str(), ".jsp") != NULL
        || strstr(file.c_str(), ".asp") != NULL
		  || strstr(file.c_str(), ".cgi") != NULL
		  || strstr(file.c_str(), ".exe") != NULL) {
         run_CGI(fd, file);
      }
      else if(strstr(file.c_str(), ".gif") != NULL) {
         get_file(fd, file, "image/gif");
      }
      else if(strstr(file.c_str(), ".jpg") != NULL
              || strstr(file.c_str(), ".jpeg") != NULL) {
         get_file(fd, file, "image/jpeg");
      }
      else {
         get_file(fd, file, "text/html");
      }
   }

   void process_POST(int fd, string & file, const int n)
   {
      string data;
      char ch;
      string response;

      file = get_request_string(file);
      
      if(!is_valid(file)) {
         error(fd, BAD_REQUEST);
         return;
      }

      putenv("REQUEST_METHOD=POST");
      
      putenv(("CONTENT_LENGTH=" + to_string(n)).c_str());

#ifdef WIN32_HTTPD
      file = HTTP_ROOT + to_win32(file);
#endif

#ifdef LINUX_HTTPD
      file = HTTP_ROOT + file;
#endif

cout << endl << flush;
cout << "file: \"" << file << "\"" << endl << flush;
cout << "length: \"" << n << "\"" << endl << flush;
cout << endl << flush;

      int count = 0;

      while(recv(fd, &ch, 1, 0) == 1) {
         data += ch;

cout << data << endl << flush;

         count++;

         if(count >= n) {
            break;
         }
      }

cout << "write next..." << endl << flush;

      ofstream out("temp.txt", ios::out | ios::ate | ios::trunc);

      if(out) {
         out << data << flush;

         out.close();
      }

      string prog = string(CAT_PROG) + string(" temp.txt | ") + file + string(" > temp2.txt");

      if(system(prog.c_str()) != 0) {
         error(fd, INTERNAL_SERVER_ERROR);
      }

cout << "read next..." << endl << flush;

      ifstream fin("temp2.txt", ios::in);

      if(fin) {
         while(fin) {
            ch = fin.get();

            if(fin) {
               response += ch;
            }
         }

         fin.close();
      }

      unlink("test.txt");

      unlink("test2.txt");

cout << "send next..." << endl << flush;

      string ret = string("HTTP/1.0 200 OK\r\n")
                   + response;

      send(fd, ret.c_str(), ret.size(), 0);
   }

   void process_request(int fd, const string_vector_type & req)
   {
      if(req.size() == 0) {
         return;
      }

      string temp;

      for(size_t i = 0; i < req.size(); i++) {
         if(strstr(req[i].c_str(), "Content-Length:") != NULL) {
            for(size_t j = 0; j < req[i].size(); j++) {
               if(isdigit(req[i][j])) {
                  temp += req[i][j];
               }
            }
            break;
         }
      }

      int count = atoi(temp.c_str());

      string cmd = req[0];

      if(cmd.find("GET") == 0) {
         process_GET(fd, cmd);
      }
      else if(cmd.find("POST") == 0) {
         process_POST(fd, cmd, count);
      }
      else {
         error(fd, BAD_REQUEST);
      }
   }

   void run()
   {
      sockaddr_in svr, cli;
      SOCKET svrSock, cliSock;
      int n;

      svrSock = socket(AF_INET, SOCK_STREAM, 0);

      cout << "socket..." << endl << flush;

      memset((char*)(&svr), 0, sizeof(svr));
      memset((char*)(&cli), 0, sizeof(cli));

      svr.sin_family = AF_INET;
      svr.sin_port = htons(HTTP_PORT);
      svr.sin_addr.s_addr = inet_addr(ip_address.c_str());
      //svr.sin_addr.s_addr = htonl(INADDR_ANY);

      bind(svrSock, (const sockaddr *)(&svr), sizeof(svr));

      cout << "bind..." << endl << flush;

      n = listen(svrSock, SOMAXCONN);

      cout << "listen..." << endl << flush;

      SOCKET fd, svr_fd;
      string_vector_type request;
      string client_ip;

      socklen_t client = sizeof(cli);

      while((fd = accept(svrSock, (struct sockaddr*)(&cli), &client)) > 0) {
         cout << "accept..." << endl << flush;

         request = read_request(fd);

         process_request(fd, request);

         closesocket(fd);
      }
   }
};

#ifdef LINUX_HTTPD
void
handler(int arg)
{
   switch(arg) {
   case SIGCHLD:
      break;

   case SIGINT:
   case SIGTERM:
   case SIGHUP:
      exit(0);
      break;
   }
}
#endif LINUX_HTTPD

int
main(int argc,
     char ** argv,
     char ** envp)
{
#ifdef LINUX_HTTPD
   signal(SIGCHLD, handler);
   signal(SIGHUP, handler);
   signal(SIGINT, handler);
   signal(SIGTERM, handler);
#endif

#ifdef WIN32_HTTPD
   WSAStartup(WSA_VERSION, &stWSAData); 
#endif

   httpd websvr(HTTP_HOST, HTTP_PORT);

   websvr.run();

   return 0;
}
