//
// Author	: Matthew W. Coan
// Date		: 2/12/2000
//
// An assambloy language simulation program.
//
//
#include "stdafx.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <vector>

using namespace std;

// Memory and code rep.
struct Code {
   char opcode;
   int arg;
};

struct Procedure {
	char * memory;
	size_t memorySize;
	Code * code;
	size_t codeSize;
};

// Register rep.
union Register {
	int ival;
	char cval;
	float fval;
};

size_t lineNumber;
int PC;
Register AC, R1, R2;		   

// Typedefs:
typedef void (*ProcType)();
typedef vector< pair< const char *, ProcType > > NativeProcVectorType;
typedef vector< pair< char *, size_t > > IDAddressVectorType;
typedef vector< pair< char *, Procedure > > IDProcVectorType;

// The procedure vector:
IDProcVectorType procVec;

// The bilt in procedures.
NativeProcVectorType nativeProcVec;

// Global memory rep.
char * globalMemory;
size_t globalMemorySize;

// Stack rep.
class RuntimeStack {
	char * _base, 
		 * _tos;
	size_t _size;
public:
	RuntimeStack() : _size(1){
	   _tos = _base = new char[1];
	}

	~RuntimeStack() {
	   delete [] _base;
	}

	void pushi(int i) {
        char * temp = new char[_size+sizeof(int)];
		memcpy(temp, _base, _size);
		_tos = temp + _size;
		memcpy(_tos, &i, sizeof(int));
		delete [] _base;
		_base = temp;
		_tos += sizeof(int);
		_size += sizeof(int);
	}

	void pushc(char c) {
		char * temp = new char[_size+sizeof(char)];
		memcpy(temp, _base, _size);
		_tos = temp + _size;
		memcpy(_tos, &c, sizeof(char));
		delete [] _base;
		_base = temp;
		_tos += sizeof(char);
		_size += sizeof(char);
	}

	void pushf(float f) {
		char * temp = new char[_size+sizeof(float)];
		memcpy(temp, _base, _size);
		_tos = temp + _size;
		memcpy(_tos, &f, sizeof(float));
		delete [] _base;
		_base = temp;
		_tos += sizeof(float);
		_size += sizeof(float);
	}

	int * topi() {
		return reinterpret_cast< int * >(_tos - sizeof(int));
	}

	char * topc() {
		return _tos - sizeof(char);
	}

	float * topf() {
		return reinterpret_cast< float * >(_tos - sizeof(float));
	}

	void popi() {
		_size -= sizeof(int);
		char * temp = new char[_size];
		memcpy(temp, _base, _size);
		delete [] _base;
		_base = temp;
		_tos = _base + _size;
	}

	void popc() {
		_size -= sizeof(char);
		char * temp = new char[_size];
		memcpy(temp, _base, _size);
		delete [] _base;
		_base = temp;
		_tos = _base + _size;
	}

	void popf() {
		_size -= sizeof(float);
		char * temp = new char[_size];
		memcpy(temp, _base, _size);
		delete [] _base;
		_base = temp;
		_tos = _base + _size;
	}
};

// The runtime stack.
RuntimeStack runtimeStack;

// Constant op-codes.
                  // INT's
static const char ADD = 0,
				  SUB = 1,
				  DIV = 2,
				  MUL = 3,
				  RET = 4,
				  MOV = 5,
				  BZ = 6,
				  LT = 7,
				  GT = 8,
				  EQ = 9,
				  LTEQ = 10,
				  GTEQ = 11,
				  NOTEQ = 12,
				  NOT = 13,
				  AND = 14,
				  OR = 15,
				  BR = 16,
				  INC = 17,
				  DEC = 18,
				  LSHIFT = 19,
				  RSHIFT = 20,
				  PUSH = 21,
				  POP = 22,
				  MOD = 23,
				  BAND = 24,
				  BOR = 25,
				  XOR = 26,
				  CALL = 27,
				  NOP = 28, 
				  NEG = 29,
				  ADDR_OF = 30,
				  R_VAL = 31,
				  L_VAL = 32,
                  // CHAR's
				  CADD = 33,
				  CSUB = 34,
				  CDIV = 35,
				  CMUL = 36,
				  CMOV = 37,
				  CBZ = 38,
				  CLT = 39,
				  CGT = 40,
				  CEQ = 41,
				  CLTEQ = 42,
				  CGTEQ = 43,
				  CNOTEQ = 44,
				  CNOT = 45,
				  CAND = 46,
				  COR = 47,
				  CINC = 48,
				  CDEC = 49,
				  CLSHIFT = 50,
				  CRSHIFT = 51,
				  CPUSH = 52,
				  CPOP = 53,
				  CMOD = 54,
				  CBAND = 55,
				  CBOR = 56,
				  CXOR = 57,
				  CNEG = 58,
				  CR_VAL = 59,
				  CL_VAL = 60,
				  // FLOAT's
				  FADD = 61,
				  FSUB = 62,
				  FDIV = 63,
				  FMUL = 64,
				  FMOV = 65,
				  FBZ = 66,
				  FLT = 67,
				  FGT = 68,
				  FEQ = 69,
				  FLTEQ = 70,
				  FGTEQ = 71,
				  FNOTEQ = 72,
				  FNOT = 73,
				  FAND = 74,
				  FOR = 75,
				  FINC = 76,
				  FDEC = 77,
				  FPUSH = 78,
				  FPOP = 79,
				  FNEG = 80,
				  FR_VAL = 81,
				  FL_VAL = 82,
				  // Conversion operations
				  I2C = 83,
				  I2F = 84,
				  C2I = 85,
				  C2F = 86,
				  F2I = 87,
				  F2C = 88;

// Function delarations:
void error(const char * msg);
void execute(Code & c, bool & runBit);
void interpret(Procedure & proc);
int findAddress(char * id, IDAddressVectorType & idVec);
int findProcAddr(char * id);
int findProcAddr2(char * id);
void skipComent(FILE * fin);
void loadNativeProcVec();
void load(const char * fname, int * & mem, Code * & code);
void cleanUp();
void * toAddress(int addr);

void
cleanUp()
{
   delete [] globalMemory;
   int i;
   for(i = 0; i < procVec.size(); i++) {
	   free(procVec[i].first);
	   delete [] procVec[i].second.code;
	   delete [] procVec[i].second.memory;
   }
   procVec.erase(procVec.begin(), procVec.end());
   nativeProcVec.erase(nativeProcVec.begin(), nativeProcVec.end());
}

void
error(const char * msg)
{
   fprintf(stderr, "#%d %s\n", lineNumber, msg);
   exit(EXIT_FAILURE);
}

void *
toAddress(int addr)
{  
   void * ptr = (void*)addr;
   return ptr;
}

void
execute(Code & c, bool & runBit)
{
	int temp;
	switch(c.opcode) {
    // INT operations:
	case ADD:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = R1.ival + R2.ival;
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case SUB:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = R1.ival - R2.ival;
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case DIV:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		if(R1.ival == 0)
			error("divde by zero!");

		AC.ival = R1.ival / R2.ival;
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case MUL:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = R1.ival * R2.ival;
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case RET:
		runBit = false;
		break;

	case MOV:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(toAddress(R2.ival), toAddress(R1.ival), sizeof(int));
		PC++;
		break;

	case BZ:
		if(AC.ival == 0) 
			PC = c.arg;
		else
			PC++;
		break;

    case LT:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival < R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case GT:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival > R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case EQ:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival == R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case LTEQ:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival <= R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case GTEQ:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival >= R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case NOTEQ:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival != R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case NOT:
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (!R1.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case AND:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival && R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case OR:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival || R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case BR:
		PC = c.arg;
		break;

	case INC:
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = R1.ival + 1;
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case DEC:
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = R1.ival - 1;
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case LSHIFT:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival << R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case RSHIFT:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival >> R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

    case PUSH:
		temp = *(reinterpret_cast< int * >(toAddress(c.arg)));
		runtimeStack.pushi(temp);
		PC++;
		break;

	case POP:
		memcpy(toAddress(c.arg), runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		PC++;
		break;

	case MOD:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival % R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;
	
	case BAND:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival & R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case BOR:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival | R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case XOR:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (R1.ival ^ R2.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case CALL:
		if(c.arg < 0) {
			temp = -(c.arg + 1);
			ProcType p = nativeProcVec[temp].second;
			p();
		}
		else {
			temp = PC;
			interpret(procVec[c.arg].second);
			PC = temp;
		}
		PC++;
		break;

	case NOP:
		PC++;
		break;

	case NEG:
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		AC.ival = (-R1.ival);
		runtimeStack.pushi(AC.ival);
		PC++;
		break;

	case ADDR_OF:
		runtimeStack.pushi(c.arg);
		PC++;
		break;

	case R_VAL:
		temp = *((int*)(*(reinterpret_cast< int* >(toAddress(c.arg)))));
		runtimeStack.pushi(temp);
		PC++;
		break;

	case L_VAL:
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		*((int*)(R1.ival)) = *((int*)(c.arg));
		PC++;
		break;
		
	// CHAR's
	case CADD:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval + R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CSUB:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval - R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CDIV:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		if(R1.cval == 0)
			error("divide by zero!");
		AC.cval = (R1.cval / R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CMUL:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval * R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CMOV:		
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(toAddress(R2.ival), toAddress(R1.ival), sizeof(char));
		PC++;
		break;

	case CBZ:
		if(AC.cval == 0)
			PC = c.arg;
		else
			PC++;
		break;

	case CLT:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval < R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CGT:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval > R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CEQ:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval == R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CLTEQ:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval <= R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CGTEQ:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval >= R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CNOTEQ:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval != R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CNOT:
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (!R1.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CAND:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval && R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case COR:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval || R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CINC:
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval + 1);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CDEC:
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval - 1);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CLSHIFT:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval << R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CRSHIFT:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval >> R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CPUSH:
		runtimeStack.pushc(*((char*)(c.arg)));
		PC++;
		break;

	case CPOP:
		memcpy(toAddress(c.arg), runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		PC++;
		break;

	case CMOD:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval % R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CBAND:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval & R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CBOR:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval | R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CXOR:
		memcpy(&R2.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (R1.cval ^ R2.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CNEG:
		memcpy(&R1.cval, runtimeStack.topc(), sizeof(char));
		runtimeStack.popc();
		AC.cval = (-R1.cval);
		runtimeStack.pushc(AC.cval);
		PC++;
		break;

	case CR_VAL:
		temp = *((char*)(*(reinterpret_cast< int* >(toAddress(c.arg)))));
		runtimeStack.pushc(temp);
		PC++;
		break;

	case CL_VAL:
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		*((char*)(R1.ival)) = *((char*)(c.arg));
		PC++;
		break;

	// FLOAT's
	case FADD:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval + R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FSUB:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval - R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FDIV:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		if(R1.fval == 0)
			error("divide by zero!");
		AC.fval = (R1.fval / R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FMUL:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval * R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FMOV:
		memcpy(&R2.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		memcpy(toAddress(R2.ival), toAddress(R1.ival), sizeof(float));
		PC++;
		break;

	case FBZ:
		if(AC.fval == 0)
			PC = c.arg;
		else
			PC++;
		break;

	case FLT:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval < R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FGT:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval > R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FEQ:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval == R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FLTEQ:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval <= R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FGTEQ:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval >= R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FNOTEQ:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval != R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FNOT:
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (!R1.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FAND:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval && R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FOR:
		memcpy(&R2.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval || R2.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FINC:
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval + 1);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FDEC:
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (R1.fval - 1);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FPUSH:
		runtimeStack.pushf(*((float*)(toAddress(c.arg))));
		PC++;
		break;

	case FPOP:
		memcpy(toAddress(c.arg), runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		PC++;
		break;

	case FNEG:
		memcpy(&R1.fval, runtimeStack.topf(), sizeof(float));
		runtimeStack.popf();
		AC.fval = (-R1.fval);
		runtimeStack.pushf(AC.fval);
		PC++;
		break;

	case FR_VAL: {
		float temp = *((float*)(*(reinterpret_cast< int* >(toAddress(c.arg)))));
		runtimeStack.pushf(temp);
		PC++;
				 }
		break;

	case FL_VAL:
		memcpy(&R1.ival, runtimeStack.topi(), sizeof(int));
		runtimeStack.popi();
		*((float*)(R1.ival)) = *((float*)(c.arg));
		PC++;
		break;

	// Conversion operations
	case I2C: 
		runtimeStack.pushc(*((int*)(c.arg)));
		PC++;
		break;

	case I2F: {
		float temp = (float)(*((int*)(c.arg)));
		runtimeStack.pushf(temp);
		PC++;
			  }
		break;

	case C2I: 
		temp = *((char*)(c.arg));
		runtimeStack.pushi(temp);
		PC++;
		break;

	case C2F: {
		float temp = *((char*)(c.arg));
		runtimeStack.pushf(temp);
		PC++;
			  }
		break;

	case F2I:
		temp = *((float*)(c.arg));
		runtimeStack.pushi(temp);
		PC++;
		break;

	case F2C:
		temp = *((float*)(c.arg));
		runtimeStack.pushc(temp);
		PC++;
		break;
	}
}

void
interpret(Procedure & proc)
{
   // Create a local copy of the data and code.
   char * localMem = new char[proc.memorySize];
   memcpy(localMem, proc.memory, proc.memorySize);
   Code * localCode = new Code[proc.codeSize];
   memcpy(localCode, proc.code, (proc.codeSize * sizeof(Code)));

   int temp;
   // Compute memory addresses.
   for(int i = 0; i < proc.codeSize; i++) {
	   switch(localCode[i].opcode) {
	   case PUSH:
	   case POP:
	   case ADDR_OF:
	   case R_VAL:
	   case L_VAL:
	   case CPUSH:
	   case CPOP:
	   case CR_VAL:
	   case CL_VAL:
	   case FPUSH:
	   case FPOP:
	   case FR_VAL:
	   case FL_VAL:
	   case I2C:
	   case I2F:
	   case C2I:
	   case C2F:
	   case F2I:
	   case F2C:
		   temp = localCode[i].arg; 
		   if(temp < 0) {
			   temp = -(temp + 1);
			   temp = (int)(globalMemory + temp);
		   }
		   else 
			   temp =  (int)(localMem + temp);
		   localCode[i].arg = temp;
		   break;
	   }
   }

   // Set the run bit to true and the program 
   // counter to zero.
   bool runBit = true;
   PC = 0;

   // Execute the code.
   while(runBit) 
	   execute(localCode[PC], runBit);

   // Free local memory.
   delete [] localCode;
   delete [] localMem;
}

int
findAddress(char * id, IDAddressVectorType & idVec)
{
	for(int i = 0; i < idVec.size(); i++) {
	   if(strcmp(id, idVec[i].first) == 0)
		   return idVec[i].second;
	}

	return -1;
}

int 
findProcAddr(const char * id)
{
   int i;
   for(i = 0; i < procVec.size(); i++) {
	   if(strcmp(procVec[i].first, id) == 0)
		   return i;
   }

   return -1;
}

int 
findProcAddr2(char * id)
{
   int i;
   for(i = 0; i < nativeProcVec.size(); i++) {
	   if(strcmp(nativeProcVec[i].first, id) == 0)
		   return i;
   }

   return -1;
}

void
skipComent(FILE * fin)
{
	int c;
	c = fgetc(fin);
	while(isspace(c)) {
		c = fgetc(fin);
		if(c == '\n')
			lineNumber++;
	}
	ungetc(c, fin);
	if(c == ';') {
   	    while(c == ';') {
           while((c=fgetc(fin)) != '\n' && !feof(fin))
			  ;

           lineNumber++;
   	       while(isspace(c)) {
		      c = fgetc(fin);
		      if(c == '\n')
			     lineNumber++;
		   }
		}
    	ungetc(c, fin);
	}
}

void
printInt()
{
	fprintf(stdout, "%d", (*(runtimeStack.topi())));
	fflush(stdout);
	runtimeStack.popi();
}

void
printChar()
{
	fprintf(stdout, "%c", (*(runtimeStack.topc())));
	fflush(stdout);
	runtimeStack.popc();
}

void
printFloat()
{
	fprintf(stdout, "%f", (*(runtimeStack.topf())));
	fflush(stdout);
	runtimeStack.popf();
}

void
readInt()
{
	int i;
	fscanf(stdin, "%d", &i);
    runtimeStack.pushi(i);
}

void
readChar()
{
	char c = fgetc(stdin);
	runtimeStack.pushc(c);
}

void
readFloat()
{
	float f;
	fscanf(stdin, "%f", &f);
	runtimeStack.pushf(f);
}

void
loadNativeProcVec()
{
	pair< const char *, ProcType > proc;
	proc.first = "printInt";
	proc.second = printInt;
	nativeProcVec.insert(nativeProcVec.end(), proc);
	proc.first = "printChar";
	proc.second = printChar;
	nativeProcVec.insert(nativeProcVec.end(), proc);
	proc.first = "printFloat";
	proc.second = printFloat;
	nativeProcVec.insert(nativeProcVec.end(), proc);
	proc.first = "readInt";
	proc.second = readInt;
	nativeProcVec.insert(nativeProcVec.end(), proc);
	proc.first = "readChar";
	proc.second = readChar;
	nativeProcVec.insert(nativeProcVec.end(), proc);
	proc.first = "readFloat";
	proc.second = readFloat;
	nativeProcVec.insert(nativeProcVec.end(), proc);
}

void
load(const char * fname)
{
	loadNativeProcVec();
	int temp;
	lineNumber = 0;
	IDAddressVectorType globalAddrVec;
	pair< char*, Procedure > id2Proc;
	FILE * fin = fopen(fname, "r");
	const size_t BUF_MAX = 1024;
	char buf[BUF_MAX];
	if(fin == NULL)
		error("unable to open input file!");
	size_t i, sz;
	char * ptr, * ptr2;
	
	// Skip coments.
	skipComent(fin);

	// Load the global data.
	memset(buf, 0, BUF_MAX);
	fscanf(fin, "%1023s", buf);
    if(strcmp(buf, ".GLOBAL_DATA") != 0)
		error("expected .GLOBAL_DATA");
	fscanf(fin, "%d", &sz);
	fscanf(fin, "%d", &globalMemorySize);
	ptr = ptr2 = globalMemory = new char[globalMemorySize];
	skipComent(fin);
	for(i = 0; i < sz; i++) {
	   memset(buf, 0, BUF_MAX);
	   fscanf(fin, "%s", buf);
	   if(strcmp(buf, "INT") == 0) {
	      memset(buf, 0, BUF_MAX);
          fscanf(fin, "%s", buf);
	      int temp = atoi(buf);
		  memcpy(ptr, &temp, sizeof(int));
	      ptr2 = ptr + sizeof(int);
	      memset(buf, 0, BUF_MAX);
	      fscanf(fin, "%s", buf);
	   }
	   else if(strcmp(buf, "CHAR") == 0) {
		  memset(buf, 0, BUF_MAX);
	      fscanf(fin, "%s", buf);
		  char temp = atoi(buf);
		  memcpy(ptr, &temp, sizeof(char));
		  ptr2 = ptr + sizeof(char);
		  memset(buf, 0, BUF_MAX);
		  fscanf(fin, "%s", buf);
	   }
	   else if(strcmp(buf, "FLOAT") == 0) {
		  memset(buf, 0, BUF_MAX);
	      fscanf(fin, "%s", buf);
		  float temp = atof(buf);
		  memcpy(ptr, &temp, sizeof(float));
		  ptr2 = ptr + sizeof(float);
		  memset(buf, 0, BUF_MAX);
		  fscanf(fin, "%s", buf);
	   }
	   skipComent(fin);
	   globalAddrVec.insert(globalAddrVec.end(), 
		                    pair< char *, size_t >(
			   			    strdup(buf), ptr - globalMemory));
	   ptr = ptr2;
	}

	// Load the procedure declarations.
	memset(buf, 0, BUF_MAX);
	fscanf(fin, "%1023s", buf);
    if(strcmp(buf, ".PROCS") != 0)
		error("expected .PROCS");
	fscanf(fin, "%d", &sz);
	skipComent(fin);
	for(i = 0; i < sz; i++) {
		memset(buf, 0, BUF_MAX);
		fscanf(fin, "%1023s", buf);
		id2Proc.first = strdup(buf);
		procVec.insert(procVec.end(), id2Proc);
		skipComent(fin);
	}

	// While not end of file load procedures.
	while(!feof(fin)) {
	   // Load the procedure name.
	   memset(buf, 0, BUF_MAX);
	   fscanf(fin, "%1023s", buf);
	   if(strcmp(buf, ".PROC") != 0)
		   error("expected .PROC");

	   // Load the procedure data.
	   memset(buf, 0, BUF_MAX);
	   fscanf(fin, "%1023s", buf);
	   for(i = 0; i < procVec.size(); i++) 
		   if(strcmp(buf, procVec[i].first) == 0) {
			   id2Proc.first = procVec[i].first;
			   break;
		   }
       
       if(i == procVec.size())
		   error("procedure not declared!");

	   skipComent(fin);
	   memset(buf, 0, BUF_MAX);
	   fscanf(fin, "%1023s", buf);
       if(strcmp(buf, ".DATA") != 0)
	      error("expected .DATA");
	   fscanf(fin, "%d", &sz);
	   fscanf(fin, "%d", &id2Proc.second.memorySize);
	   id2Proc.second.memory = new char[id2Proc.second.memorySize];
	   ptr2 = ptr = id2Proc.second.memory;
	   IDAddressVectorType idAddrVec;
	   skipComent(fin);
	   for(i = 0; i < sz; i++) {
	      memset(buf, 0, BUF_MAX);
	      fscanf(fin, "%s", buf);
	      if(strcmp(buf, "INT") == 0) {
	         memset(buf, 0, BUF_MAX);
             fscanf(fin, "%s", buf);
	         int temp = atoi(buf);
		     memcpy(ptr, &temp, sizeof(int));
	         ptr2 = ptr + sizeof(int);
	         memset(buf, 0, BUF_MAX);
	         fscanf(fin, "%s", buf);
		  }
	      else if(strcmp(buf, "CHAR") == 0) {
		     memset(buf, 0, BUF_MAX);
	         fscanf(fin, "%s", buf);
		     char temp = atoi(buf);
		     memcpy(ptr, &temp, sizeof(char));
		     ptr2 = ptr + sizeof(char);
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%s", buf);
		  }
	      else if(strcmp(buf, "FLOAT") == 0) {
		     memset(buf, 0, BUF_MAX);
	         fscanf(fin, "%s", buf);
		     float temp = atof(buf);
		     memcpy(ptr, &temp, sizeof(float));
		     ptr2 = ptr + sizeof(float);
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%s", buf);
		  }
		  skipComent(fin);
	      idAddrVec.insert(idAddrVec.end(), 
		                   pair< char *, size_t >(
			   			   strdup(buf), ptr - id2Proc.second.memory));
	      ptr = ptr2;
	   } // end for

	   // Load the procedure code.
	   memset(buf, 0, BUF_MAX);
	   fscanf(fin, "%s", buf);
	   if(strcmp(buf, ".CODE") != 0) 
		   error("expected .CODE");
	   fscanf(fin, "%d", &id2Proc.second.codeSize);
	   skipComent(fin);
	   IDAddressVectorType lableVec;
	   id2Proc.second.code = new Code[id2Proc.second.codeSize];
	   for(i = 0; i < id2Proc.second.codeSize; i++) {
          memset(buf, 0, BUF_MAX);
	      fscanf(fin, "%1023s", buf);
		  if(strcmp(buf, "ADD") == 0) { 
		     id2Proc.second.code[i].opcode = ADD;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "SUB") == 0) {
		     id2Proc.second.code[i].opcode = SUB;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "DIV") == 0) {
		     id2Proc.second.code[i].opcode = DIV;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "MUL") == 0) {
		     id2Proc.second.code[i].opcode = MUL;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "RET") == 0) {
		     id2Proc.second.code[i].opcode = RET;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "MOV") == 0) {
		     id2Proc.second.code[i].opcode = MOV;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "BZ") == 0) {
		     memset(buf, 0, BUF_MAX);
			 fscanf(fin, "%1023s", buf);
		     id2Proc.second.code[i].opcode = BZ;
			 int j;
			 for(j = 0; j < lableVec.size(); j++)
				 if(strcmp(buf, lableVec[j].first) == 0) {
					 id2Proc.second.code[i].arg = lableVec[j].second;
					 break;
				 }
			 if(j == lableVec.size()) { 
				 lableVec.insert(lableVec.end(), 
				                 pair< char*, size_t >(
								 strdup(buf), -1));
		         id2Proc.second.code[i].arg = -(1+(lableVec.size() - 1));
			 }
		  }
          else if(strcmp(buf, "LT") == 0) {
		     id2Proc.second.code[i].opcode = LT;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "GT") == 0) {
		     id2Proc.second.code[i].opcode = GT;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "EQ") == 0) {
		     id2Proc.second.code[i].opcode = EQ;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "LTEQ") == 0) {
		     id2Proc.second.code[i].opcode = LTEQ;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "GTEQ") == 0) {
		     id2Proc.second.code[i].opcode = GTEQ;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "NOTEQ") == 0) {
		     id2Proc.second.code[i].opcode = NOTEQ;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "NOT") == 0)  {
		     id2Proc.second.code[i].opcode = NOT;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "AND") == 0) {
		     id2Proc.second.code[i].opcode = AND;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "OR") == 0) {
		     id2Proc.second.code[i].opcode = OR;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "BR") == 0) {
		     memset(buf, 0, BUF_MAX);
			 fscanf(fin, "%1023s", buf);
		     id2Proc.second.code[i].opcode = BR;
			 int j;
			 for(j = 0; j < lableVec.size(); j++)
				 if(strcmp(buf, lableVec[j].first) == 0) {
					 id2Proc.second.code[i].arg = lableVec[j].second;
					 break;
				 }
			 if(j == lableVec.size()) { 
				 lableVec.insert(lableVec.end(), 
				                 pair< char*, size_t >(
								 strdup(buf), -1));
		         id2Proc.second.code[i].arg = -(1+(lableVec.size() - 1));
			 }
		  }
          else if(strcmp(buf, "INC") == 0) {
		     id2Proc.second.code[i].opcode = INC;
		     id2Proc.second.code[i].arg = -1;
		  }
          else if(strcmp(buf, "DEC") == 0) {
		     id2Proc.second.code[i].opcode = DEC;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "LSHIFT") == 0) {
		     id2Proc.second.code[i].opcode = LSHIFT;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "RSHIFT") == 0) {
		     id2Proc.second.code[i].opcode = RSHIFT;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "PUSH") == 0) {		   
		     id2Proc.second.code[i].opcode = PUSH;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
	      else if(strcmp(buf, "POP") == 0) {
		     id2Proc.second.code[i].opcode = POP;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
	      else if(strcmp(buf, "MOD") == 0) {
		     id2Proc.second.code[i].opcode  = MOD;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "BAND") == 0) {
		     id2Proc.second.code[i].opcode = BAND;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "BOR") == 0) {
		     id2Proc.second.code[i].opcode = BOR;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "XOR") == 0) {
		     id2Proc.second.code[i].opcode = XOR;
		     id2Proc.second.code[i].arg = -1;
		  }
	      else if(strcmp(buf, "CALL") == 0) {
		     id2Proc.second.code[i].opcode = CALL;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
			 id2Proc.second.code[i].arg = 
                         findProcAddr(static_cast< const char * >(buf));
			 if(id2Proc.second.code[i].arg < 0) {
				 id2Proc.second.code[i].arg = findProcAddr2(buf);
				 if(id2Proc.second.code[i].arg < 0)
					 error("bad id!");
				 id2Proc.second.code[i].arg = -(1 + id2Proc.second.code[i].arg);
			 }
		  }
		  else if(strcmp(buf, "NOP") == 0) {
			  id2Proc.second.code[i].opcode = NOP;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "NEG") == 0) {
		     id2Proc.second.code[i].opcode = NEG;
			 id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "ADDR_OF") == 0) {
		     id2Proc.second.code[i].opcode = ADDR_OF;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "R_VAL") == 0) {
		     id2Proc.second.code[i].opcode = R_VAL;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "L_VAL") == 0) {
		     id2Proc.second.code[i].opcode = L_VAL;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "CADD") == 0) {
			  id2Proc.second.code[i].opcode = CADD;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CSUB") == 0) {
			  id2Proc.second.code[i].opcode = CSUB;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CDIV") == 0) {
			  id2Proc.second.code[i].opcode = CDIV;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CMUL") == 0) {
			  id2Proc.second.code[i].opcode = CMUL;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CMOV") == 0) {
			  id2Proc.second.code[i].opcode = CMOV;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CBZ") == 0) {
			  id2Proc.second.code[i].opcode = CBZ;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CLT") == 0) {
			  id2Proc.second.code[i].opcode = CLT;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CGT") == 0) {
			  id2Proc.second.code[i].opcode = CGT;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CEQ") == 0) {
			  id2Proc.second.code[i].opcode = CEQ;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CLTEQ") == 0) {
			  id2Proc.second.code[i].opcode = CLTEQ;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CGTEQ") == 0) {
			  id2Proc.second.code[i].opcode = CGTEQ;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CNOTEQ") == 0) {
			  id2Proc.second.code[i].opcode = CNOTEQ;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CNOT") == 0) {
			  id2Proc.second.code[i].opcode = CNOT;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CAND") == 0) {
			  id2Proc.second.code[i].opcode = CAND;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "COR") == 0) {
			  id2Proc.second.code[i].opcode = COR;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CINC") == 0) {
			  id2Proc.second.code[i].opcode = CINC;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CDEC") == 0) {
			  id2Proc.second.code[i].opcode = CDEC;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CLSHIFT") == 0) {
			  id2Proc.second.code[i].opcode = CLSHIFT;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CRSHIFT") == 0) {
			  id2Proc.second.code[i].opcode = CRSHIFT;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CPUSH") == 0) {
		     id2Proc.second.code[i].opcode = CPUSH;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "CPOP") == 0) {
		     id2Proc.second.code[i].opcode = CPOP;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "CMOD") == 0) {
			  id2Proc.second.code[i].opcode = CMOD;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CBAND") == 0) {
			  id2Proc.second.code[i].opcode = CBAND;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CBOR") == 0) {
			  id2Proc.second.code[i].opcode = CBOR;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CXOR") == 0) {
			  id2Proc.second.code[i].opcode = CXOR;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CNEG") == 0) {
			  id2Proc.second.code[i].opcode = CNEG;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "CR_VAL") == 0) {
		     id2Proc.second.code[i].opcode = CR_VAL;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "CL_VAL") == 0) {
		     id2Proc.second.code[i].opcode = CL_VAL;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "FADD") == 0) {
			  id2Proc.second.code[i].opcode = FADD;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FSUB") == 0) {
			  id2Proc.second.code[i].opcode = FSUB;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FDIV") == 0) {
			  id2Proc.second.code[i].opcode = FDIV;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FMUL") == 0) {
			  id2Proc.second.code[i].opcode = FMUL;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FMOV") == 0) {
			  id2Proc.second.code[i].opcode = FMOV;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FBZ") == 0) {
			  id2Proc.second.code[i].opcode = FBZ;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FLT") == 0) {
			  id2Proc.second.code[i].opcode = FLT;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FGT") == 0) {
			  id2Proc.second.code[i].opcode = FGT;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FEQ") == 0) {
			  id2Proc.second.code[i].opcode = FEQ;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FLTEQ") == 0) {
			  id2Proc.second.code[i].opcode = FLTEQ;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FGTEQ") == 0) {
			  id2Proc.second.code[i].opcode = FGTEQ;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FNOTEQ") == 0) {
			  id2Proc.second.code[i].opcode = FNOTEQ;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FNOT") == 0) {
			  id2Proc.second.code[i].opcode = FNOT;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FAND") == 0) {
			  id2Proc.second.code[i].opcode = FAND;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FOR") == 0) {
			  id2Proc.second.code[i].opcode = FOR;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FINC") == 0) {
			  id2Proc.second.code[i].opcode = FINC;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FDEC") == 0) {
			  id2Proc.second.code[i].opcode = FDEC;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FPUSH") == 0) {
			  id2Proc.second.code[i].opcode = FPUSH;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FPOP") == 0) {
			  id2Proc.second.code[i].opcode = FPOP;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FNEG") == 0) {
			  id2Proc.second.code[i].opcode = FNEG;
			  id2Proc.second.code[i].arg = -1;
		  }
		  else if(strcmp(buf, "FR_VAL") == 0) {
		     id2Proc.second.code[i].opcode = FR_VAL;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "FL_VAL") == 0) {
		     id2Proc.second.code[i].opcode = FL_VAL;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "I2C") == 0) {
		     id2Proc.second.code[i].opcode = I2C;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "I2F") == 0) {
		     id2Proc.second.code[i].opcode = I2F;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "C2I") == 0) {
		     id2Proc.second.code[i].opcode = C2I;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "C2F") == 0) {
		     id2Proc.second.code[i].opcode = C2F;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "F2I") == 0) {
			 id2Proc.second.code[i].opcode = F2I;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else if(strcmp(buf, "F2C") == 0) {
		     id2Proc.second.code[i].opcode = F2C;
		     memset(buf, 0, BUF_MAX);
		     fscanf(fin, "%1023s", buf);
		     temp = findAddress(buf, idAddrVec);
			 if(temp < 0) {
			    temp = findAddress(buf, globalAddrVec);
				if(temp < 0)
			       error("bad id!");
				temp = -(1 + temp);
			 }
			 id2Proc.second.code[i].arg = temp;
		  }
		  else {
			 int j;
			 for(j = 0; j < lableVec.size(); j++)
				 if(strcmp(buf, lableVec[j].first) == 0) {
					 lableVec[j].second = i;
					 break;
				 }
			 if(j == lableVec.size()) {
				 lableVec.insert(lableVec.end(), 
				                 pair< char*, size_t >(
								 strdup(buf), i));
			 }
			 i--;
		  }

		  // Skip the coments:
		  skipComent(fin);
	   } // end for

	   // Fix the lable addresses in the code.
       for(i = 0; i < id2Proc.second.codeSize; i++) { 
		   if(id2Proc.second.code[i].opcode == BR
			   && id2Proc.second.code[i].arg < 0) {
			   id2Proc.second.code[i].arg = lableVec[-(id2Proc.second.code[i].arg + 1)].second;
		   }
		   else if(id2Proc.second.code[i].opcode == BZ
			   && id2Proc.second.code[i].arg < 0) { 
			   id2Proc.second.code[i].arg = lableVec[-(id2Proc.second.code[i].arg + 1)].second;
		   }
	   }

	   for(i = 0; i < lableVec.size(); i++)
		   free(lableVec[i].first);

	   for(i = 0; i < sz; i++) 
		   free(idAddrVec[i].first);
	   
	   // Add the new procedure.
	   for(i = 0; i < procVec.size(); i++) 
		   if(strcmp(procVec[i].first, id2Proc.first) == 0) {
			   procVec[i] = id2Proc;
			   break;
		   }
	} // end while.

    fclose(fin);
}

//
// Main entry point:
//
int
main(int argc, char* argv[])
{ 
	if(argc != 2) {
		fprintf(stderr, "Usage: %s <filename.asm>\n", argv[0]);
		exit(1);
	}

	// Load the program.
	load(argv[1]);

	// Get the address of the main entry point.
	int mainAddr = findProcAddr("main");

	// Check for no main.
	if(mainAddr < 0)
		error("can\'t find main entry point!");
	
	// Execute the program.
	interpret(procVec[mainAddr].second);

	// Free memory.
	cleanUp();

	// return value
	return 0;
}

