@*A MIX SIMULATOR.
This will simulate a binary  computer with 64 distinct
values per byte. 
Each byte will be represented by 6 bits, with a value range of
|000| to |077|.
Signs will be a single bit, |0| for plus or |1| for minus in the first position of 
the real word.
A word will then occupy 4 actual 8-bit hardware bytes of the real machine
and leave one bit (the second bit of the real word) over for possible future use
(garbage collection- - with new operations to set the bit on and off?)

@ The overall structure will be
@c
@<Header files to include@>@/
@<Global variables@>@/
@<Functions@>@/
@<The main program@>

@ Definitions
@d SIGN(word) (word & MINUS)	/* sign of word */
@d MAG(word) (word & FIVEBYTES)  /* magnitude of word */
@d ADDRESS(word) (word & TWOBYTES << 3*BYTESIZE) >> 3*BYTESIZE	/* magnitude of address */
@d INDEX(word) (word & ONEBYTE << 2*BYTESIZE) >> 2*BYTESIZE  /* index specification */
@d FIELD(word) (word & ONEBYTE << BYTESIZE) >> BYTESIZE /* F-specification */
@d CODE(word) (word & ONEBYTE)  /* operation code */
@d MEMSIZE 4000
@d BYTESIZE 6
@d BYTESPERWORD 5
@d ONEBYTE 077
@d TWOBYTES 07777
@d FIVEBYTES 07777777777
@d PLUS  0
@d MINUS (1<<(WORDSIZE+1))
@d WORDSIZE BYTESIZE*BYTESPERWORD
@d LDA   8                          
@d LDAN 16                         
@d STA  24                         
@d IOC  35                       
@d IN   36                       
@d OUT  37                        
@d JA   40                         
@d INCA 48                         
@d CMPA 56                         
@d ON      1
@d OFF     0
@d LESS   -1
@d EQUAL   0
@d GREATER 1



@ Define 10 registers and 10 signs, all initialized  to plus zero.
@d areg  reg[0]   /* Magnitude of A-register */
@d i1reg reg[1]   /* Magnitude of index registers */
@d i2reg reg[2]
@d i3reg reg[3]
@d i4reg reg[4]
@d i5reg reg[5]
@d i6reg reg[6]
@d xreg  reg[7]   /* Magnitude of X-register */
@d jreg  reg[8]   /* Magnitude of J-register */
@d zero  reg[9]   /* Constant zero, for "STZ" */
@d signa sign[0]       /* Sign of A-register */
@d sign1 sign[1]       /* Sign of index registers */
@d sign2 sign[2]
@d sign3 sign[3]
@d sign4 sign[4]
@d sign5 sign[5]
@d sign6 sign[6]
@d signx sign[7]      /* Sign of X-register */
@d signj sign[8]      /* Sign of J-register */
@d signz sign[9]      /* Sign stored by ``STZ'' */
 
@<Global variables@>=
float version = 1.01;
char  *progname;
long cell[MEMSIZE];
long reg[10] = {0,0,0,0,0,0,0,0,0,0};
long sign[10] = {0,0,0,0,0,0,0,0,0,0};
@ An array of 64 structures of pointers to functions, execution times
 and flags to check index registers.
@d T 1
@d F 0

@<Global variables@>=
typedef void (*PFV)();
typedef int BOOLEAN;

void nop(), add(), sub(), mul(), div(), spec(), shift(), move(), 
     load(), loadn(), store(), jbus(), ioc(), in(), out(), jred(), 
     jump(), regjump(), addrop(), compare();

struct operation { PFV fn;
                   int time;
                   BOOLEAN iflag;
                  };

struct operation optable[64] = { @/
                                { nop,     1, F},@/
                                { add,     2, F},@/
                                { sub,     2, F},@/
                                { mul,    10, F},@/
                                { div,    12, F},@/
                                { spec,    1, F},@/
                                { shift,   2, F},@/
                                { move,    1, F},@/
                                { load,    2, F},@/
                                { load,    2, T},@/
                                { load,    2, T},@/
                                { load,    2, T},@/
                                { load,    2, T},@/
                                { load,    2, T},@/
                                { load,    2, T},@/
                                { load,    2, F},@/
                                { loadn,   2, T},@/
                                { loadn,   2, F},@/
                                { loadn,   2, F},@/
                                { loadn,   2, F},@/
                                { loadn,   2, F},@/
                                { loadn,   2, F},@/
                                { loadn,   2, F},@/
                                { loadn,   2, T},@/
                                { store,   2, F},@/
                                { store,   2, F},@/
                                { store,   2, F},@/
                                { store,   2, F},@/
                                { store,   2, F},@/
                                { store,   2, F},@/
                                { store,   2, F},@/
                                { store,   2, F},@/
                                { store,   2, F},@/
                                { store,   2, F},@/
                                { jbus,    1, F},@/
                                { ioc,     1, F},@/
                                { in,      1, F},@/
                                { out,     1, F},@/
                                { jred,    1, F},@/
                                { jump,    1, F},@/
                                { regjump, 1, F},@/
                                { regjump, 1, F},@/
                                { regjump, 1, F},@/
                                { regjump, 1, F},@/
                                { regjump, 1, F},@/
                                { regjump, 1, F},@/
                                { regjump, 1, F},@/
                                { regjump, 1, F},@/
                                { addrop,  1, F},@/
                                { addrop,  1, T},@/
                                { addrop,  1, T},@/
                                { addrop,  1, T},@/
                                { addrop,  1, T},@/
                                { addrop,  1, T},@/
                                { addrop,  1, T},@/
                                { addrop,  1, F},@/
                                { compare, 2, T},@/
                                { compare, 2, T},@/
                                { compare, 2, T},@/
                                { compare, 2, T},@/
                                { compare, 2, T},@/
                                { compare, 2, T},@/
                                { compare, 2, T},@/
                                { compare, 2, T}}; @/

@ An array of device structures describing the each of the available     
I/O devices.
The typewriter terminal and paper tape unit have been discontinued.

Setting setting the clock field of a unit to -1 indicates that
no I/O operation is outstanding for this unit.
@d TAPE7     7                         
@d DISK8     8                         
@d DISK15   15                         
@d READER   16                        
@d PUNCH    17                        
@d PRINTER  18                         
@d MAXUNITS 19                        


@<Global variables@>=
void tape(), disk(), reader(), punch(), printer();


struct io_device { PFV device;    /* device function  */
                   int io_time;   /* transmission time per block */
                   int position;  /* current disk or tape head position */
                   int seek_pos;  /* seek position on disk */
                   int tape_end;  /* end of file on tape */
                   int seek_time; /* skip or seek time */  
                   int address;   /* memory address for current i/o */
                   int io_op;     /* IOC, IN or OUT */ 
                   int clock;     /* clock time when i/o will occur */
                   int complete;  /* flag for I/O complete */
                   int fd;        /* file descriptor */
                  };

struct io_device *unitp, unit[MAXUNITS] = { @/
                   {tape,      50, 0, 0, 0,    50, 0, 0, -1, 1, 0}, @/
                   {tape,      50, 0, 0, 0,    50, 0, 0, -1, 1, 0}, @/
                   {tape,      50, 0, 0, 0,    50, 0, 0, -1, 1, 0}, @/
                   {tape,      50, 0, 0, 0,    50, 0, 0, -1, 1, 0}, @/
                   {tape,      50, 0, 0, 0,    50, 0, 0, -1, 1, 0}, @/
                   {tape,      50, 0, 0, 0,    50, 0, 0, -1, 1, 0}, @/
                   {tape,      50, 0, 0, 0,    50, 0, 0, -1, 1, 0}, @/
                   {tape,      50, 0, 0, 0,    50, 0, 0, -1, 1, 0}, @/
                   {disk,     250, 0, 0, 0,     1, 0, 0, -1, 1, 0}, @/
                   {disk,     250, 0, 0, 0,     1, 0, 0, -1, 1, 0}, @/
                   {disk,     250, 0, 0, 0,     1, 0, 0, -1, 1, 0}, @/
                   {disk,     250, 0, 0, 0,     1, 0, 0, -1, 1, 0}, @/
                   {disk,     250, 0, 0, 0,     1, 0, 0, -1, 1, 0}, @/
                   {disk,     250, 0, 0, 0,     1, 0, 0, -1, 1, 0}, @/
                   {disk,     250, 0, 0, 0,     1, 0, 0, -1, 1, 0}, @/
                   {disk,     250, 0, 0, 0,     1, 0, 0, -1, 1, 0}, @/
                   {reader, 10000, 0, 0, 0,     0, 0, 0, -1, 1, 0}, @/
                   {punch,  20000, 0, 0, 0,     0, 0, 0, -1, 1, 0}, @/
                   {printer, 7500, 0, 0, 0, 10000, 0, 0, -1, 1, 0}}; @/


@  Allocate some variables.
@<Global variables@>=
int loc;          /* Location of the next instruction */
int mem;          /* Address of the present instruction, plus indexing */
int op;           /* Operation code of the present instruction */
int fld;          /* F-field of the present instruction */ 
long inst;        /* Instruction being simulated */
int compi = EQUAL; /* Comparison indicator */
int ovtog = OFF;  /* Overflow toggle */
int uclock;        /* Simulated execution time */
int time = 0;   	/* Time of previous instruction */
long rA, rX;      /* Used as work registers to pass data */
int  rI1, rI2;
void inc(), dec(), sizechk(), ovcheck(), jmp(), jsj(), jumps(), 
			addrerror(), indexerror(), operror(), memerror(), fielderror(), sizeerror(),
      dump();
PFV interrupt = dump;
@*Control Routine.
This does things common to all instructions, unpacks the instruction
into its various parts, and puts the parts into convenient places
for later use.
@<The main...@>=
main(int argc, char *argv[])
{
	int i, t;
  progname = argv[0];

  if (signal(SIGINT, SIG_IGN) != SIG_IGN) signal(SIGINT, interrupt); 
 while (argc > 1 && argv[1][0] == '-') {
       switch (argv[1][1]) {
       case 'V':  /* show version */
           printf("%s: version %4.2f\n", progname, version);
           exit(0);
       default:
           fprintf(stderr,"%s: unknown arg %s\n", progname, argv[1]);
           exit(1);
       }
       argc--, argv++;
    }

	go();
	loc = 0;    	/* Take first instruction from location zero. */
	while(T) {
			uclock += time;	
			for (fld=0;fld<MAXUNITS;fld++) {  /* Check each unit. */
        unitp = &unit[fld];
				if (unitp->clock > 0 && unitp->clock <= uclock) { 
                                   /* Did one finish? */
					do_io();
        }
      }
	    inst = cell[loc]; /* Instruction to simulate */
			loc++;	/* Advance the location counter */
      if (loc >= MEMSIZE) addrerror();
	    mem = ADDRESS(inst);	/*  magnitude of address */
	    if (SIGN(inst)) mem = -mem; /* attach sign of address */
			rI2 = INDEX(inst); /*Examine index field */
			if (rI2) {	/* Is it zero? */ 
				if (rI2 > 6) indexerror(); /* Illegal index specified */
	      mem += sign[rI2] ? -reg[rI2] : reg[rI2]; /* Attach sign */
	      if (abs(mem) > TWOBYTES) addrerror(); /* Is the result too large */
	    }
	    fld = FIELD(inst); /* F-field */ 
	    op = CODE(inst);	/* C-field */
	    if (op >= 64) operror(op);  /* Can't happen on this machine */ 
			time = optable[op].time;  /* Get execution time from table */
      unitp = &unit[fld];
	    (*optable[op].fn)();	/* Call routine for this opcode */
	}	
}
@ The error routines write to standard error so they need stdio.h
@<Header...@>=
#include <stdio.h>
#include <fcntl.h>
#include <strings.h>
#include <signal.h>
@ Memory fetch routine
Returns contents of memory location |mem|.
@<Functions@>=
void memory() 
{
	if (mem < 0 || mem >= MEMSIZE) memerror();
	rA = MAG(cell[mem]); /* rA $\leftarrow$ magnitude of word */
	rX = SIGN(cell[mem]); /* rX $\leftarrow$ sign of word */
}
@ The |fcheck| subroutine process a partial field specification,
making sure that it has the form $8L+R$ with $L\le R\le 5$.
Returns value of L.
@<Functions@>=
void fcheck()
{
	rI1 = rA = fld / 8;  /* rA $\leftarrow$  L */
	rI2 = rX = fld % 8;  /* rX $\leftarrow$  R */
	if (rA > rX || rX > 5) fielderror();  /* Error if L $>$ R or R $>$ 5 */
}  
@ |sizechk|
@<Functions@>=
void sizechk()
{
	if (optable[op].iflag) {  /* Have we just loaded an index register? */ 
		if (abs(rA) > TWOBYTES) sizeerror(); /* Make sure the result fits in two bytes */
	} else ovcheck();
}

@ |ovcheck|
@<Functions@>=
void ovcheck()
{
		if (rA & FIVEBYTES+1) ovtog = ON; /* Did overflow just occur? */
		rA &= FIVEBYTES;  /* reset overflow bit */
} 
@ Normally called by |getv| to extract a field from memory.
|getav| is called directly only in comparison operations
to extract a field from a register.
@<Functions@>=
void getav()
{
	if (rI1) { /* Is the sign included in the field? */
		rX = PLUS;  /* If not, set the sign positive. */
		rA <<= (BYTESIZE * (rI1 - 1)); /* Zero out all bytes to the left */
    rA &= FIVEBYTES;
		rA >>= (BYTESIZE * (rI1 - 1)); /* of the field. */
	}
	rA >>= (BYTESIZE * (BYTESPERWORD - rI2));	/* Shift right into the proper position. */		
}
@ |getv| finds the quantity V (namely, the appropriate field
of location M) used in various MIX operators.
@<Functions@>=
void getv()
{
	fcheck();  /* Process the field and set |rI1| $\leftarrow$ |L| */
	memory();	 /* |rA| $\leftarrow$ memory magnitude, |rX| $\leftarrow$ sign.  */
	getav();   /* Extract the field */
}
@*Individual Operators.
Use NOP to reset the clock if F is 63.
@<Functions@>=
void nop()
{
/* after adding 1 for the NOP the clock will be zero. */
	if (fld == 63) uclock = -1; 
}
@ |add| 
@<Functions@>=
void add()
{
	getv();  /* Get the value V in rA and rX. */
	rI1 = 0; /* Let rI1 indicate the A register */
	inc(); /* Go to ``increase'' routine. */
}
@ |sub| 
@<Functions@>=
void sub()
{
	getv();  /* Get the value V in rA and rX. */
	rI1 = 0; /* Let rI1 indicate the A register */
	dec(); /* Go to ``decrease'' routine. */
}
@ |mul|
@<Functions@>=
void mul()
{
	long long work; /* 64-bit work area */

	getv();  /* Get the value V in rA and rX. */
	if (rX == signa)   /* Are the signs the same? */
		rX = PLUS;   /* Set rX to the result sign */
	else
		rX = MINUS;
  signa = signx = rX;  /* Put in both simulated registers */
	work = rA * (long long int) areg;
	xreg = work & FIVEBYTES;
  work >>= WORDSIZE;
	areg = work & FIVEBYTES;
}		

@ |div|
@<Functions@>=
void div()
{
	long long work; /* 64-bit work area */

	signx = signa; /* Set the sign of the remainder */
	getv();  /* Get the value V in rA and rX. */
	if (rX == signa)   /* Are the signs the same? */
		rX = PLUS;   /* Set rX to the result sign */
	else
		rX = MINUS;
	signa = rX; /* Put it in the simulated rA */
	if (areg > rA) ovtog = ON;  /* The quotient will not fit in 5 bytes */
	work = areg;  /* Divide the operands */
	work <<= WORDSIZE; 
	work |= xreg;
	areg = (work / rA) & FIVEBYTES;
	xreg = work % rA;
} 

@ |load|
@<Functions@>=
void loadn() 
{
	getv();  /* Get the value V in rA and rX. */
	rI1 = op - LDAN; /* Register */
	rX = rX ? PLUS : MINUS; /* Negate the sign. */
	reg[rI1] = rA & FIVEBYTES;
	sign[rI1] = rX;
	sizechk();
}
void load() 
{
	getv();  /* Get the value V in rA and rX. */
	rI1 = op - LDA; /* Register */
	reg[rI1] = rA & FIVEBYTES;
	sign[rI1] = rX;
	sizechk();
}
	
@ |store|
@<Functions@>=
void store()
{
	long work;

	fcheck();  /* rI1 $\leftarrow$ L. */
	memory();  /* Get the contents of the memory location. */
	if (!rI1) {  /* Is the sign included in the field */
		rI1 = 1; /* If so, change L to 1 */
		rX = sign[op - STA]; /*  and ``store'' the register's sign */
	}
	rA <<= (BYTESIZE * rI2);
  rA &= FIVEBYTES; 
	rA >>= (BYTESIZE * rI2);
	work = rA; /* save the area to the field's right */
	rA = cell[mem];
  rA &= FIVEBYTES; 
	rA >>= (BYTESIZE * (BYTESPERWORD - (rI1 - 1)));
	rA <<= (BYTESIZE * (BYTESPERWORD - (rI1 - 1)));
	work |= rA; /* OR in area to field's left */
	rA = reg[op - STA];
	rA <<= (BYTESIZE * (BYTESPERWORD - rI2 + (rI1 - 1)));  /* Truncate register */
  rA &= FIVEBYTES; 
	rA >>= (BYTESIZE * (rI1 - 1));  /* Right adjust register */
	work |= rA;  /* OR in register */
	work |= rX;  /* OR in sign */
	cell[mem] = work;
}
@ |jump|
@<Functions@>=
void jump()
{
	if (fld > 9) fielderror();  /* Is F too large? */
 	rA = compi;  /* rA $\leftarrow$ comparison indicator. */
	jumps(fld);
}
void jumps(int fld)
{
  switch (fld)
	{
		case 0:  /* jmp */
			jmp();
			break;
		case 1:	 /* jsj */
			jsj();
			break;
		case 2:	 /* jov */
			rX = ovtog;
			ovtog = OFF; /* Shut off overflow toggle. */
			if (rX) jmp();
			break;
		case 3:	 /* jnov */
			rX = ovtog; 
			ovtog = OFF; /* Shut off overflow toggle. */
			if (!rX) jmp();
			break;
		case 4:  /* ls */
			if (rA < 0) jmp();  /* Jump if rA negative. */
			break;
		case 5:  /* eq */
			if (rA == 0) jmp();  /* Jump if rA zero. */
			break;
		case 6:  /* gr */
			if (rA > 0) jmp();  /* Jump if rA positive. */
			break;
		case 7:  /* ge */
			if (rA >= 0) jmp(); /* Jump if rA zero or positive. */
			break;
		case 8:  /* ne */
			if (rA != 0) jmp(); /* Jump if rA negative or positive. */
			break;
		case 9:  /* le */
			if (rA <= 0) jmp(); /* Jump if rA zero or negative. */
			break;
	}
}
			
void jmp()
{	
	jreg = loc; /* Set the simulated J-register */
	jsj();
}
void jsj()
{
 memory();  /* Check for valid memory address. */
 loc = mem;
}

@ |regjump|
@<Functions@>=
void regjump()
{
    rA = reg[op - JA];  /* Register jumps */
    rA = sign[op - JA] ? -rA : rA;
	if (fld > 5) fielderror();
  jumps(fld+4);
}

@ |addrop|
@<Functions@>=
void addrop()  /* Address transfer operations */
{
	if (fld > 3) fielderror();  /* Is F too large? */
	if (!mem) 	rX = SIGN(inst); /*  When M is zero ENTA loads the sign of */
	else  rX = mem<0 ? MINUS : PLUS;  /*  the instruction and ENNA loads the opposite sign. */
	rA = mem>0 ? mem : -mem; /* rX $\leftarrow$ sign of M, and rA $\leftarrow$ the magnitude. */
	rI1 = op - INCA; /* rI1 indicates the register. */ 
	switch (fld) {
		case 0:
			inc();
			break;
		case 1:
			dec();
			break;
		case 2: /* enta */
			reg[rI1] = rA & FIVEBYTES;
			sign[rI1] = rX;
			sizechk();
			break;
		case 3: /* enna */
			reg[rI1] = rA & FIVEBYTES;
			sign[rI1] = rX ? PLUS : MINUS; /* Negate the sign. */
			sizechk();
			break;
	}	
}
void dec()
{
	rX = rX ? PLUS : MINUS; /* Reverse the sign. */
	inc();	/* Reduce dec to inc */
} 
void inc() /* Addition routine: */
{
	if (rX != sign[rI1]) {  /* Are the signs the same? */
		rA -= reg[rI1];  /* No; subtract the magnitudes */
		if (rA > 0)   /* Sign change needed? */
			sign[rI1] = rX; /* Change the register's sign */
		else
			rA = -rA;  /* rA is the magnitude of the difference */
	}
	else rA += reg[rI1]; /* Add magnitudes */
	reg[rI1] = rA & FIVEBYTES; /* Store the magnitude of the result */
  sizechk();
}

@ |compare|
@<Functions@>=
void compare()
{
		long v;

		getv();
		v = rX ? -rA : rA; /* Attach the sign */
		rA = reg[op - CMPA]; /* Get field F of the appropriate register */
		rX = sign[op - CMPA];
		getav();
		rX = rX ? -rA : rA; /* Attach the sign */
		if (rX == v) compi = EQUAL; /* Set comparison indicator to either */
		else compi = rX > v ? GREATER : LESS; /* zero, plus one, or minus one. */
}
		
@ |shift|
@<Functions@>=
void shift()
{
	unsigned long long work, work1;
	if (mem < 0) addrerror();
	if (fld > 5) fielderror();
	rA = areg;
	work = areg;
	work <<= WORDSIZE;
	work |= xreg; 
	
	switch (fld) {
		case 0: /* sla */
			if (mem > BYTESPERWORD) mem = BYTESPERWORD;
 			rA <<= mem * BYTESIZE;
			areg = rA & FIVEBYTES;
			break;
		case 1: /* sra */
			if (mem > BYTESPERWORD) mem = BYTESPERWORD;
 			rA >>= mem * BYTESIZE;
			areg = rA;
			break;
		case 2: /* slax */
			if (mem > (2*BYTESPERWORD)) mem = (2*BYTESPERWORD);
 			work <<= mem * BYTESIZE;
			rX = work & FIVEBYTES;
			work >>= WORDSIZE;
			rA = work & FIVEBYTES;
			areg = rA;
			xreg = rX;
			break;
		case 3: /* srax */
			if (mem > (2*BYTESPERWORD)) mem = (2*BYTESPERWORD);
 			work >>= mem * BYTESIZE;
			rX = work & FIVEBYTES;
			work >>= WORDSIZE;
			rA = work & FIVEBYTES;
			areg = rA;
			xreg = rX;
			break;
		case 4: /* slc */
			mem %= (2*BYTESPERWORD);
			work1 = work;
 			work <<= mem * BYTESIZE;
			work1 >>= (((2*BYTESPERWORD)-mem) * BYTESIZE);	
			work |= work1;  /* restore circulated bits */
			rX = work & FIVEBYTES;
			work >>= WORDSIZE;
			rA = work & FIVEBYTES;
			areg = rA;
			xreg = rX;
			break;
		case 5: /* src */
			 mem %= (2*BYTESPERWORD);
			work1 = work;
 			work >>= mem * BYTESIZE;
			work1 <<= (((2*BYTESPERWORD) - mem) * BYTESIZE);	
			work |= work1;  /* restore circulated bits */
			rX = work & FIVEBYTES;
			work >>= WORDSIZE;
			rA = work & FIVEBYTES;
			areg = rA;
			xreg = rX;
			break;
	}
}
@ |move|
@<Functions@>=
void move(){
		while (fld) {
			memory();
			rX |= rA;
			rI1 = i1reg;
			rA = sign1;
			if (rA) {
				if (rI1) memerror();
				sign1 = 0;
			}
			if (rI1 > MEMSIZE) memerror();
			cell[rI1] = rX;
			uclock += 2;
			rI1++;
			i1reg = rI1;
			mem++;
			fld--;
		}
}

@ |jbus|
@<Functions@>=
void jbus(){
	if (unitp->clock > uclock) { /* Is there incomplete I/O on this unit? */
		if (loc-1 == mem) { /* Check for JBUS * */
      do_outstanding_io(unitp->clock);
      uclock = unitp->clock; 
    }
		jmp();
  }
}

@ |ioc|
@<Functions@>=
void ioc(){
int I;
   if (fld == PUNCH || fld == READER) return;  /* ignore IOC for these units */
   if (unitp->clock > uclock) {  /* Is there i/o  outstanding? */
     do_outstanding_io(unitp->clock);
     uclock = unitp->clock; 
     do_io();
   }
   if (unitp->clock > uclock) { /* Is there still i/o outstanding? */
     do_outstanding_io(unitp->clock);
     uclock = unitp->clock; 
     do_io();
   }
    unitp->io_op = op;
    unitp->address = mem;
    if (fld <= TAPE7) 
      if (mem) unitp->clock = uclock + abs(mem)*unitp->seek_time;  
      else unitp->clock = uclock + unitp->position*unitp->seek_time;
    if (DISK8 <= fld && fld <=DISK15) { /* disk seek */
      unitp->clock = uclock + abs(xreg - unitp->position)*unitp->seek_time;
      unitp->seek_pos = xreg; /* remember where to seek */
    }
    if (fld == PRINTER) {
      unitp->clock = uclock + unitp->seek_time;   
    } 
}

@ |in|
@<Functions@>=
void in()
{
   if (unitp->clock > uclock) {  /* Is there i/o  outstanding? */
     do_outstanding_io(unitp->clock);
     uclock = unitp->clock; 
     do_io();
   }
   if (unitp->clock > uclock) { /* Is there still i/o outstanding? */
     do_outstanding_io(unitp->clock);
     uclock = unitp->clock; 
     do_io();
   }
  unitp->io_op = op;
	unitp->address = mem;  /* Remember where the input is to go to */
	unitp->clock = uclock + unitp->io_time / 2;
  unitp->complete = F;  /* flag as not done */
  if (fld >=DISK8 && fld <= DISK15) {   /* disk i/o */
    unitp->clock += abs(xreg - unitp->position)*unitp->seek_time; 
    unitp->seek_pos = xreg; /* remember where to seek */
  }
}

@ |jred|
@<Functions@>=
void jred(){
	if (unitp->clock <= uclock) 
	jmp();
}

@ |spec|
@<Functions@>=
void spec(){
	long long n, m;
	char work[11];
	int i, next, device, t;
  int table[10] = {30,31,32,33,34,35,36,37,38,39};
	
	if (fld > 2) fielderror();	
	switch (fld) {
		case 0:  /* num */
			n = 0;
			m = 1;
      for (i=0; i<BYTESPERWORD; i++) {  /* shift right and take modulus 64 to */
				n += (xreg>>BYTESIZE*i)%64%10*m; /* isolate the byte, take modulus 10 */           
				m *= 10;    /* finally raise to the appropriate power of 10. */
			}
      for (i=0; i<BYTESPERWORD; i++) {
				n += (areg>>BYTESIZE*i)%64%10*m;
				m *= 10;
			}
			areg = n & FIVEBYTES;
			if (areg != n) ovtog = ON;
			uclock += 9;
			break;
		case 1: /* char */
			sprintf(work, "%010d", areg);
			areg = xreg = 0;
			for (i = 0; i < BYTESPERWORD; i++) {
				areg |= table[work[i]-'0']  << ((BYTESPERWORD-(i+1))*BYTESIZE);
				xreg |= table[work[i+BYTESPERWORD]-'0']  << ((BYTESPERWORD-(i+1))*BYTESIZE); 
			}
			uclock += 9;
			break;
		case 2: /* hlt */
			do_outstanding_io(FIVEBYTES);         /* tag with a large value */               
			dump();
			exit(-1);
 	}
}

@ |out|
@<Functions@>=

void out()
{
   if (unitp->clock > uclock) {  /* Is there i/o  outstanding? */
     do_outstanding_io(unitp->clock);
     uclock = unitp->clock; 
     do_io();
   }
   if (unitp->clock > uclock) { /* Is there still i/o outstanding? */
     do_outstanding_io(unitp->clock);
     uclock = unitp->clock; 
     do_io();
   }
  unitp->io_op = op;
	unitp->address = mem;  /* Remember where the output is to come from */
	unitp->clock = uclock + unitp->io_time / 2;
  unitp->complete = F;  /* flag as not done */
  if (fld >=DISK8 && fld <= DISK15)  {  /* disk i/o */
    unitp->clock += abs(xreg - unitp->position)*unitp->seek_time; 
    unitp->seek_pos = xreg; /* remember where to seek */
  }
}

@* I/O routines.
@<Functions@>=
 do_io()  /* check if I/O still needs to be done */
{
    if (!unitp->complete) { /* if I/O not done then do it */
       (*unitp->device)();
       unitp->complete = T;  /* flag as done and set clock */
       unitp->clock = uclock + unitp->io_time / 2;
    }   
    else unitp->clock = -1;
}

do_outstanding_io(int time)
{
      int i, next, save_fld;

      save_fld = fld;
			next = 0;
			while (next != time) { /* Keep on till no more outstanding i/o */
				next = time;         /* up until time */                        
				for (i=0;i<MAXUNITS;i++)  /* Check all units. */
					if (unit[i].clock > 0 && unit[i].clock < next) {
						next = unit[i].clock; /* Find which should complete next. */
						fld = i;
					}
				if (next != time) {
					unitp = &unit[fld];
          uclock = unitp->clock;
          do_io();
				}
			}   /* Keep on till no more outstanding i/o */
      fld = save_fld;
			unitp = &unit[fld];
}

@ The card reader is connected to stdin.
@<Functions@>=
void reader()
{
	char buffer[82];
	char table[] = " ABCDEFGHI~JKLMNOPQR~~STUVWXYZ0123456789.,()+-*/=";
	int i, j, l;
	char *p;

	fgets(buffer, 82, stdin);
	l = strlen(buffer);
	for (i=0; i<16; i++) {   
		cell[unit[READER].address] = 0;
    for (j=0; j<BYTESPERWORD; j++)
		  if (i*BYTESPERWORD+j < l && (p = strchr(table,buffer[i*BYTESPERWORD+j])))
        cell[unit[READER].address] |= (p-table)<<(BYTESIZE*(BYTESPERWORD-(j+1)));
    unit[READER].address++;
	}
}
@ The printer is connect to stdout.         
@<Functions@>=
void printer()
{
	char table[] = " ABCDEFGHI~JKLMNOPQR~~STUVWXYZ0123456789.,()+-*/=$<>@@;:'~~~~~~~~";
	int i, j;

  switch (unitp->io_op) {
  case IOC:
    printf("\f");
    break;
  case OUT:
	  for (i = 0; i < 24; i++) 	
      for (j = BYTESPERWORD - 1; j >= 0; j--)
	    	putchar(table[(cell[unit[PRINTER].address+i] & ONEBYTE << BYTESIZE*j) >> BYTESIZE*j]);
	  putchar('\n');
    break;
  }
}
@ The punch is connected to stderr.
@<Functions@>=
void punch()
{
	char table[] = " ABCDEFGHI~JKLMNOPQR~~STUVWXYZ0123456789.,()+-*/=$<>@@;:'~~~~~~~~";
	int i, j;

	for (i = 0; i < 15; i++) 	
          for (j = BYTESPERWORD - 1; j >= 0; j--)
		putc(table[(cell[unit[PUNCH].address+i] & ONEBYTE << BYTESIZE*j) >> BYTESIZE*j],stderr);
	putc('\n',stderr);
}
@ The tape units use files tape0 to tape7 in the current directory.
@<Functions@>=
void tape()
{
    char path[7]; 

    sprintf(path,"tape%d",fld);
    if (!unitp->fd) {   /* not open */
      unitp->position = unitp->tape_end = 0;
      if (unitp->io_op == OUT) {  /* open for write */
        if ((unitp->fd = open(path,O_RDWR|O_CREAT|O_TRUNC,0777)) < 0) 
          fprintf(stderr,"Write open failed for %s\n", path);
      } else { /* open for read */
        if (!unitp->fd && (unitp->fd = open(path,O_RDWR)) < 0) 
          fprintf(stderr,"Read open failed for %s\n", path);
      }
    } 
   /* OK now it's open */
    switch (unitp->io_op) {
      case OUT:  /* write */
        if(write(unitp->fd,&cell[unitp->address],400) < 0) 
          fprintf(stderr,"Write failed for %s\n", path); 
        unitp->position++;
        unitp->tape_end = unitp->position;
        break; 
      case IN:  /* read */
        if (read(unitp->fd,&cell[unitp->address],400) <= 0) 
          fprintf(stderr,"Read failed for %s\n", path); 
        unitp->position++;
        break;
      case IOC:  /* control */
        if (unitp->address == 0) {
          close(unitp->fd);  /* rewind tape by closing it */
          unitp->fd = NULL;  /* reset the file descriptor for reuse */
          unitp->position = 0;
        } else {
          if (unitp->position + unitp->address < 0) {
            close(unitp->fd);
            unitp->position = 0;
          } else {
            if (lseek(unitp->fd, unitp->address*400, SEEK_CUR) < 0) 
              fprintf(stderr,"IOC %d on tape %d failed\n",unitp->address,fld); 
            unitp->position += unitp->address;
          }
        }
    }
}  
@ The disk units use files disk8 to disk15 in the current directory.
They are always opened for update without truncating so that old data
is not destroyed until it is overwritten.
Note that big disk files will never shrink and the only way to recover
space is to delete them outside of MIX.
@<Functions@>=
void disk()
{
    char path[8]; 
    int i, offset;

    sprintf(path,"disk%d",fld);
    if (!unitp->fd) {  
      if ((unitp->fd = open(path,O_RDWR|O_CREAT,0777)) < 0) 
        fprintf(stderr,"Open failed for %s\n", path);
    } 
    switch (unitp->io_op) {
      case IOC:
        unitp->position =  unitp->seek_pos; /* No need to actually do a seek */
        break;  
      case OUT:
        if (lseek(unitp->fd, unitp->seek_pos*400, SEEK_SET) < 0) 
          fprintf(stderr,"Seek failed for %s\n", path);
        else unitp->position = unitp->seek_pos;
        if(write(unitp->fd,&cell[unitp->address],400) < 0) 
          fprintf(stderr,"Write failed for %s\n", path);
        else unitp->position++;
        break;
      case IN:
        if (lseek(unitp->fd, unitp->seek_pos*400, SEEK_SET) < 0) 
          fprintf(stderr,"Seek failed for %s\n", path);
        else unitp->position = unitp->seek_pos;
        if (read(unitp->fd,&cell[unitp->address],400) <= 0) 
          fprintf(stderr,"Read failed for %s\n", path);
        else unitp->position++;
        break;
    }
}	

       	
@*Error routines.
@<Functions@>=
void indexerror()
{
	fprintf(stderr,"Invalid index register: %d\n", rI2);
	dump();
}	
void addrerror()  
{
	fprintf(stderr,"Invalid address: %d\n", mem);
	dump();
}	
void operror()  
{
	fprintf(stderr,"Invalid opcode: %d\n", op);
	dump();
}	
void memerror()  
{
	fprintf(stderr,"Invalid memory location: %d\n", mem);
	dump();
}	
void fielderror()  
{
	fprintf(stderr,"Invalid field: %d\n", fld);
	dump();
}	
void sizeerror()  
{
	fprintf(stderr,"Invalid size: %d\n", rA);
	dump();
}	


@*Dump routines.
@d sch(p) (sign[p] ? '-' : '+' )
@<Functions@>=
dump_status()
{
  char t[4], c[8];
  if (ovtog)   strcpy(t,"ON");
  else  strcpy(t,"OFF");  
	if (!compi) strcpy(c,"EQUAL");
  else if (compi > 0) strcpy(c,"GREATER");
      else strcpy(c,"LESS");
  fprintf(stderr,"Registers A/X    %c0%010o %c0%010o\n", sch(0),reg[0],sch(7),reg[7]);
  fprintf(stderr,"                 %c %10d %c %10d\n\n", sch(0),reg[0],sch(7),reg[7]);
  fprintf(stderr,"Index Registers  %c0%04o  %c0%04o  %c0%04o  %c0%04o  %c0%04o  %c0%04o \n",
    sch(1),reg[1],sch(2),reg[2],sch(3),reg[3],sch(4),reg[4],sch(5),reg[5],sch(6),reg[6]);
  fprintf(stderr,"                 %c %4d  %c %4d  %c %4d  %c %4d  %c %4d  %c %4d \n\n",
    sch(1),reg[1],sch(2),reg[2],sch(3),reg[3],sch(4),reg[4],sch(5),reg[5],sch(6),reg[6]);
  fprintf(stderr,"Jump Register    %c0%04o     Overflow toggle:      %s\n", sch(8),reg[8],t);
  fprintf(stderr,"                 %c %4d     Comparison Indicator: %s\n\n", sch(8),reg[8],c);
  fprintf(stderr,"Clock = %d u. Location = %04d, M %d, I %d, F %d, C %d, inst = %c %010o \n\n",
    uclock,loc,mem,INDEX(inst),fld,op,SIGN(inst)?'-':'+',MAG(inst));
}
  
void dump_memory()
{
  char buff[2][132];
  int a, i, j, k, flag;
  k = 0;
  flag = 0;
  a = 0;
  for (i=0; i<MEMSIZE; i += 8) {
    k = ++k%2;
    sprintf (buff[k],"%04d ",i); 
    for (j=0;j<8;j++) {
      sprintf(buff[k]+5+j*15,"%c%04d %02d %02d %02d ", SIGN(cell[i+j]) ? '-' : '+',
      ADDRESS(cell[i+j]), INDEX(cell[i+j]),FIELD(cell[i+j]), CODE(cell[i+j]));
    }
    if (strcmp(buff[0]+5,buff[1]+5)) {   /* If line changed. */
      if (flag) { /* Was the previous line printed? */
        if (a-i+8) fprintf(stderr,"    Lines %04d to %04d are the same.\n", a, i-16);
        fprintf(stderr,"%s\n",buff[(k+1)%2]);
      }
      flag = 0;
      fprintf(stderr,"%s\n",buff[k]);  
    }
    else {
      if (!flag) a = i;  /* Note start of identical lines */
      flag=1;
    }
  }
  if (flag) { /* Was the previous line printed? */
    if (a-i+8) fprintf(stderr,"    Lines %04d to %04d are the same.\n", a, i-16);
    fprintf(stderr,"%s\n",buff[k%2]);
  }
}

void dump()
{
  dump_status();
  dump_memory();
  exit(-1);
}

@ GO Button. 
@c
go()
{
  unit[READER].address = 0;
  unitp = &unit[READER];
  uclock += unitp->io_time;
  reader();
}
