// httpd.cpp : Defines the entry point for the console application.
//


#ifndef _HTTPD_H
#define _HTTPD_H

/*

This file is the intellectual property of
Matthew William Coan.

Simple ultra small HTTP server for Windows, Linux and BSD
using standard C++...

Author: Matthew William Coan
Date: Fri May  7 23:23:20 EDT 2010

*/

#define WIN32_HTTPD
//#define LINUX_HTTPD

#include <iostream>
#include <fstream>
#include <string>
#include <vector>

#include <cstdlib>
#include <cstdio>
#include <cstring>

using namespace std;

#ifdef LINUX_HTTPD
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#define SOCKET int
#define closesocket(a) close(a)
#endif 

#ifdef WIN32_HTTPD

#include <winsock2.h>

WSADATA stWSAData;  // WinSock DLL Info.

#define WSA_MAJOR_VERSION 1
#define WSA_MINOR_VERSION 1
#define WSA_VERSION MAKEWORD(WSA_MAJOR_VERSION, WSA_MINOR_VERSION)

#define popen _popen
#define pclose _pclose

#define socklen_t int

#endif

#include <assert.h>

#ifdef LINUX_HTTPD
#define HTTP_ROOT "/home/mcoan/commands"
#endif

#ifdef WIN32_HTTPD
const char * HTTP_ROOT = "C:\\wwwroot";
#endif

int HTTP_PORT = 8080;
const char * HTTP_HOST = "127.0.0.1";
const char * DEFAULT_PAGE = "/index.html";

int
System(const string & cmd)
{
	return system(cmd.c_str());
}

class httpd {
public:
   typedef vector< string > string_vector_type; 

   enum response_code { 
      BAD_REQUEST=500,
      FILE_NOT_FOUND=404,
      OK=200
   };

private:
   string ip_address;
   int port;

public:
   httpd(const string & addr,
         const int the_port)
   {
      ip_address = addr;
      port = the_port;
   }

   httpd(const httpd & cp)
   {
      ip_address = cp.ip_address;
      port = cp.port;
   }

   httpd()
   {
      port = HTTP_PORT;
      ip_address = HTTP_HOST;
   }

   virtual ~httpd()
   {
   }

   httpd & operator=(const httpd & right)
   {
      ip_address = right.ip_address;
      port = right.port;
      return *this;
   }

   bool read_line(int fd, string & line)
   {
      bool ret = false;
      char ch = '\0';
      size_t the_size = 0;
      const int MAX_LENGTH = 1024 * 10;

      while(recv(fd, &ch, 1, 0) == 1) {
          if(ch == '\n') {
             ret = true;
             break;
          }

          if(ch == '\r') {
             continue;
          }

          the_size++;

          if(the_size >= MAX_LENGTH) {
             break;
          }

          line += ch;
      }

      return ret;
   }

   bool is_valid(const string & str)
   {
      bool ret = true;

      for(size_t i = 0; i < str.size(); i++) {
         if((str[i] >= 'a' && str[i] <= 'z')
            || (str[i] >= 'A' && str[i] <= 'Z')
            || (str[i] >= '0' && str[i] <= '9')
            || (str[i] == '-')
            || (str[i] == '.')
            || (str[i] == '_')
            || (str[i] == '.')
            || (str[i] == '&')
            || (str[i] == '%')
            || (str[i] == '=')
            || (str[i] == '?')
            || (str[i] == '/')) {
            if((i+1) < str.size()) {
               if(str[i+1] == '.' && str[i] == '.') {
                  ret = false;
                  break;
               }
            }
         }
         else {
            ret = false;
            break;
         }
      }

      return ret;
   }

   string_vector_type read_request(int fd)
   {
      string line;
      string_vector_type vec;
      size_t the_size = 0;
      const int MAX_LENGTH = 1024 * 10;

      while(read_line(fd, line)) {
         if(line.size() == 0) {
            break;
         }

         the_size += line.length();

         if(the_size >= MAX_LENGTH) {
            break; 
         }

         vec.push_back(line);

         line = "";
      }

      return vec;
   }

   void error(int fd, int code)
   {
      switch(code) {
      case BAD_REQUEST:
         break;

      case FILE_NOT_FOUND:
         break;
      }
   }

   string get_query_string(const string & str)
   {
      string ret;
      int i;

      for(i = 0; i < str.size(); i++) {
         if(str[i] == '?') {
            i++;
            break;
         } 
      }

      while(i < str.size()) {
         ret += str[i];

         i++;
      }

      return ret;
   }

   string get_URI(const string & str)
   {
      string ret;

      for(int i = 0; i < str.size(); i++) {
         if(str[i] == '?') {
            break; 
         }

         ret += str[i];
      }

      return ret;
   }

   string to_string(size_t size)
   {
      char buffer[1024];

      memset(buffer, 0, 1024);

      sprintf(buffer, "%d", size);

      return string(buffer);
   }

   void run_CGI(int fd, const string & file)
   {
      string qs = get_query_string(file);

      string prog = get_URI(file);
/*
#ifdef LINUX_HTTPD
      prog = "export QUERY_STRING=\"" + qs + "\";"
             + "export REQUEST_METHOD=\"GET\";"
             + prog;
#endif
*/
      string env = string("QUERY_STRING=") + qs;
      putenv(env.c_str());
      putenv("REQUEST_METHOD=GET");

      FILE * program = popen(prog.c_str(), "r");

      string data;

      if(program) {
         char ch;

         while((ch = fgetc(program)) != EOF) {
            data += ch;
         }

         pclose(program);
      }

      data = string("HTTP/1.0 200 OK\r\n") 
         + string("Content-length: ") + to_string(data.size()) + string("\r\n")
         + data;

      send(fd, data.c_str(), data.size(), 0);
   }

   void get_file(int fd, const string & file)
   {
      string ret;
      char ch;

      ifstream in(file.c_str());

      if(in) {
         ch = in.get();

         while(in) {
            ret += ch;

            ch = in.get();
         }

         in.close();
      }
      else {
         error(fd, FILE_NOT_FOUND);
      }

      ret = string("HTTP/1.0 200 OK\r\n")
            + string("Content-type: text/html\r\n\r\n")
            + ret;

      send(fd, ret.c_str(), ret.size(), 0);
   }

   string get_request_string(const string & cmd)
   {
      int i = 0;

      while(cmd[i] != '/') {
         i++;

         if(i >= cmd.size()) {
            break;
         }
      }

      string file;

      while(cmd[i] != ' ') {
         file += cmd[i];
         i++;

         if(i >= cmd.size()) {
            break;
         }
      }

      return file;
   }

   string to_win32(const string & s)
   {
      string ret;

	  for(int i = 0; i < s.size(); i++) {
		  if(s[i] == '/') {
	         ret += "\\";
		  }
		  else {
			 ret += s[i];
		  }
	  }

	  return ret;
   }

   void process_GET(int fd, string & cmd)
   {
      string file = get_request_string(cmd);

      if(!is_valid(file)) {
         error(fd, BAD_REQUEST);
         return;
      }

	  string uri = file;

      //cout << "URI: \"" << file << "\"" << endl << flush;

      if(file == "/") {
         file = DEFAULT_PAGE;
      }
 
#ifdef WIN32_HTTPD
      file = HTTP_ROOT + to_win32(file);
#endif

#ifdef LINUX_HTTPD
      file = HTTP_ROOT + file;
#endif

      //cout << "FULL PATH: \"" << file << "\"" << endl << flush;

      if(strstr(file.c_str(), "?") != NULL
        || strstr(file.c_str(), ".pl") != NULL
        || strstr(file.c_str(), ".php") != NULL
        || strstr(file.c_str(), ".jsp") != NULL
        || strstr(file.c_str(), ".asp") != NULL
#ifdef LINUX_HTTPD
		  || strstr(file.c_str(), ".cgi") != NULL) {
#endif

#ifdef WIN32_HTTPD
		 || strstr(file.c_str(), ".exe") != NULL) {
#endif
         run_CGI(fd, file);
      }
      else if(strcmp(uri.c_str(), "/shutdown") == 0) {
		 exit(0);
	  }
      else {
         get_file(fd, file);
      }
   }

   void process_POST(int fd, string & cmd)
   {
	   char * ptr = strstr(cmd.c_str(), "?");
	   if(ptr != NULL) {
		   ptr++;
		   int ct = 0;
		   char buffer[64];
		   sprintf(buffer, "%d", ct);
		   char * buf = new char [ ct + 1 ];
		   memset(buf, 0, ct+1);
		   recv(fd, buf, ct, 0);
		   ofstream out("request.txt", ios::out | ios::binary);
		   out << buf << flush;
		   out.close();
		   delete [] buf;
		   if(System("setenv CONENT_TYPE=POST;setenv CONTENT_LENGTH=" 
			   + string(buffer) + ";cat request.txt | " 
			   + string(HTTP_ROOT) + cmd.substr(0, cmd.size()-strlen(ptr))
			   + " > output.dat") == 0) {
				get_file(fd, "output.dat");
		   }
	   }
   }

   void process_request(int fd, const string_vector_type & req)
   {
      if(req.size() == 0) {
         return;
      }

      string cmd = req[0];

      if(cmd.find("GET") == 0) {
         process_GET(fd, cmd);
      }
      else if(cmd.find("POST") == 0) {
         process_POST(fd, cmd);
      }
      else {
         error(fd, BAD_REQUEST);
      }
   }

   void run()
   {
      sockaddr_in svr, cli;
      SOCKET svrSock, cliSock;
      int n;

      svrSock = socket(AF_INET, SOCK_STREAM, 0);

      //cout << "socket..." << endl << flush;

      memset((char*)(&svr), 0, sizeof(svr));
      memset((char*)(&cli), 0, sizeof(cli));

      svr.sin_family = AF_INET;
      svr.sin_port = htons(port);
      //svr.sin_addr.s_addr = inet_addr(ip_address.c_str());
      svr.sin_addr.s_addr = htonl(INADDR_ANY);

      bind(svrSock, (const sockaddr *)(&svr), sizeof(svr));

//      cout << "bind..." << endl << flush;

      n = listen(svrSock, SOMAXCONN);

     // cout << "listen..." << endl << flush;

      SOCKET fd, svr_fd;
      string response;
      string_vector_type request;
      string client_ip;

      socklen_t client = sizeof(cli);

      while((fd = accept(svrSock, (struct sockaddr*)(&cli), &client)) > 0) {
         //cout << "accept..." << endl << flush;

         request = read_request(fd);

         process_request(fd, request);

         closesocket(fd);
      }
   }
};

#ifdef LINUX_HTTPD
void
handler(int arg)
{
   switch(arg) {
   case SIGCHLD:
      break;

   case SIGINT:
   case SIGTERM:
   case SIGHUP:
      exit(0);
      break;
   }
}
#endif LINUX_HTTPD

int
httpd_main(int argc,
          char ** argv,
          char ** envp)
{
#ifdef LINUX_HTTPD
   signal(SIGCHLD, handler);
   signal(SIGHUP, handler);
   signal(SIGINT, handler);
   signal(SIGTERM, handler);
#endif

#ifdef WIN32_HTTPD
   WSAStartup(WSA_VERSION, &stWSAData); 
#endif

   if(argc == 5) {
      HTTP_ROOT = argv[1];
      HTTP_PORT = atoi(argv[2]);
      HTTP_HOST = argv[3];
      DEFAULT_PAGE = argv[4];
   }


   httpd websvr(HTTP_HOST, HTTP_PORT);

   websvr.run();

   return 0;
}

#endif