/*

SIMULATION OF A CENTRAL PROCESSING UNIT IN C++

Authro: Mattthew W. Coan
Date: Tue Mar  7 18:35:37 EST 2017

*/

#ifndef _CPU_H
#define _CPU_H

#include <string>
#include <vector>
#include <iostream>
#include "soft_cpu.h"
#include "memory.h"

namespace cpu_sim {

using namespace std;
using namespace memory_sim;

class Circuit { 
protected:
   Circuit * input1;
   Circuit * input2;
   int i1,i2;
public:
   Circuit() {
      input1 = 0;
      input2 = 0;
      i1 = 0;
      i2 = 0;
   }

   void load1(int value) { i1 = value; }

   void load2(int value) { i1 = value; }

   Circuit * get_input1() {
      return input1;
   }

   Circuit * get_input2() {
      return input2;
   }

   void set_input1(Circuit * c) { input1 = c; }

   void set_input2(Circuit * c) { input2 = c; }

   virtual int execute() {
      return 0;
   }
};

class AND : public Circuit { 
public:
   int execute() {
      if(input1)
         i1 = input1->execute();
      if(input2)
         i2 = input2->execute();
      data_word_type ret = i1 & i2;
      return ret;
   }
};

class OR : public Circuit { 
public:
   int execute() {
      if(input1)
         i1 = input1->execute();
      if(input2)
         i2 = input2->execute();
      data_word_type ret = i1 | i2;
      return ret;
   }
};

class XOR : public Circuit { 
public:
   int execute() {
      if(input1)
         i1 = input1->execute();
      if(input2)
         i2 = input2->execute();
      data_word_type ret = i1 ^ i2;
      return ret;
   }
};

class NOT : public Circuit { 
public:
   int execute() {
      if(input1)
         i1 = input1->execute();
      data_word_type ret = ~i1;
      return ret;
   }
};

class WIRE : public Circuit {
public:
   int execute() {
      if(input1)
         i1 = input1->execute();
      return i1;
   }
};

class Register {
   int data[BITS_SIZE];
public:
   Register() { }
   ~Register() { }

   data_word_type get_value() { 
      data_word_type ret = 0;
      for(int i = 0; i < BITS_SIZE; i++) {
         if(data[i]) {
            ret |= bit_array[i];
         }
      }
      return ret;
   }

   void set_value(data_word_type data_word) {
      for(int i = 0; i < BITS_SIZE; i++) {
         data[i] = (bit_array[i] & data_word)?(1):(0);
      }
   }

   int get_bit(size_t i) { return data[i]; }
   void set_bit(size_t i, int value) {
      data[i] = value;
   }
};

class CPU;

class Instruction {
protected:
   CPU * cpu;
   Memory * memory;
public:
   Instruction() { cpu = 0; memory = 0; }
   virtual ~Instruction() { }
   virtual void run() = 0;
};

class LOAD1 : public Instruction {
public:
   LOAD1(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~LOAD1() { }
   void run(); 
};

class LOAD2 : public Instruction {
public:
   LOAD2(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~LOAD2() { }
   void run(); 
};

class ADD : public Instruction {
   Circuit * _and1[BITS_SIZE];
   Circuit * _and2[BITS_SIZE];
   Circuit * _xor1[BITS_SIZE];
   Circuit * _xor2[BITS_SIZE];
   Circuit * _xor3[BITS_SIZE];;
   Circuit * wire1[BITS_SIZE];
   Circuit * wire2[BITS_SIZE];
public:
   ADD(CPU * cpu, Memory * mem) { 
      this->cpu = cpu; 
      memory = mem; 
/*
A -----+---\\------\
       |    >> XOR1 >-+--\\------\
B --+--|---//------/  |   >> XOR2 >-- S
    |  |          +---|--//------/
    |  |          |   |
C --|--|--------+ |   +--|-----)
    |  |        | |      | AND1 )--\\------\
    |  |        +-+------|-----)    >> XOR3 >-- C
    |  |                      +----//------/
    |  +-- |-----)            |
    |      | AND2 )-----------+
    +------|-----)
*/

      for(size_t i = 0; i < BITS_SIZE; i++) {
         wire1[i] = new WIRE();
         wire2[i] = new WIRE();
         _xor1[i] = new XOR();
         _xor2[i] = new XOR();
         _xor3[i] = new XOR();
         _and1[i] = new AND();
         _and2[i] = new AND();
         _xor1[i]->set_input1(wire1[i]);
         _xor1[i]->set_input2(wire2[i]);
         _and1[i]->set_input1(_xor1[i]);
         if(i == 0)
            _and1[i]->set_input2(0);
         else 
            _and1[i]->set_input2(_xor3[i-1]);
         _xor2[i]->set_input1(_xor1[i]);
         if(i == 0)
            _xor2[i]->set_input2(0);
         else 
            _xor2[i]->set_input2(_xor3[i-1]);
         _xor3[i]->set_input1(_and1[i]);
         _xor3[i]->set_input2(_and2[i]);
         _and2[i]->set_input1(wire1[i]);
         _and2[i]->set_input2(wire2[i]);
      }
   }
   ~ADD() { 
      for(size_t i = 0; i < BITS_SIZE; i++) {
         delete _and1[i];
         delete _and2[i];
         delete _xor1[i];
         delete _xor2[i];
         delete _xor3[i];
         delete wire1[i];
         delete wire2[i];
      }
   }
   void run();
};

class PRINT : public Instruction {
public:
   PRINT(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~PRINT() { }
   void run(); 
};

class HALT : public Instruction {
public:
   HALT(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~HALT() { }
   void run();
};

class SUB : public Instruction {
public:
   SUB(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~SUB() { }
   void run();
};

class MUL : public Instruction {
public:
   MUL(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~MUL() { }
   void run();
};

class DIV : public Instruction {
public:
   DIV(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~DIV() { }
   void run();
};

class MOD : public Instruction {
public:
   MOD(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~MOD() { }
   void run();
};


class LT : public Instruction {
public:
   LT(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~LT() { }
   void run();
};

class GT : public Instruction {
public:
   GT(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~GT() { }
   void run();
};

class EQ : public Instruction {
public:
   EQ(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~EQ() { }
   void run();
};

class NEQ : public Instruction {
public:
   NEQ(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~NEQ() { }
   void run();
};

class LTEQ : public Instruction {
public:
   LTEQ(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~LTEQ() { }
   void run();
};

class GTEQ : public Instruction {
public:
   GTEQ(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~GTEQ() { }
   void run();
};


class JMP : public Instruction {
public:
   JMP(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~JMP() { }
   void run();
};

class JZ : public Instruction {
public:
   JZ(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~JZ() { }
   void run();
};

class JE : public Instruction {
public:
   JE(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~JE() { }
   void run();
};

class CMP : public Instruction {
public:
   CMP(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~CMP() { }
   void run();
};

class PUSH_R2 : public Instruction {
public:
   PUSH_R2(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~PUSH_R2() { }
   void run();
};

class POP_R2 : public Instruction {
public:
   POP_R2(CPU * cpu, Memory * mem) { this->cpu = cpu; memory = mem; }
   ~POP_R2() { }
   void run();
};


class CPU {
public:
   typedef vector< Instruction* > instruction_vec_type;
private:
   Register PC;
   Register SP;
   Register BP;
   Register R1;
   Register R2;
   Register CR;
   Register AL;
   instruction_vec_type code;
   Memory * memory;
public:
   CPU(Memory * memory) {
      code.push_back(new LOAD1(this,memory));
      code.push_back(new LOAD2(this,memory));
      code.push_back(new ADD(this,memory));
      code.push_back(new PRINT(this,memory));
      code.push_back(new HALT(this,memory));
      code.push_back(new SUB(this,memory));
      code.push_back(new MUL(this,memory));
      code.push_back(new DIV(this,memory));
      code.push_back(new MOD(this,memory));
      code.push_back(new LT(this,memory));
      code.push_back(new GT(this,memory));
      code.push_back(new EQ(this,memory));
      code.push_back(new NEQ(this,memory));
      code.push_back(new LTEQ(this,memory));
      code.push_back(new GTEQ(this,memory));
      code.push_back(new JMP(this,memory));
      code.push_back(new JZ(this,memory));
      code.push_back(new JE(this,memory));
      code.push_back(new CMP(this,memory));
      code.push_back(new PUSH_R2(this,memory));
      code.push_back(new POP_R2(this,memory));
      this->memory = memory;
   }
   ~CPU() {
      for(size_t i = 0; i < code.size(); i++) {
         delete code[i];
      }
      code.clear();
   }

   Register & get_PC() { return PC; }

   Register & get_SP() { return SP; }

   Register & get_BP() { return BP; }

   Register & get_R1() { return R1; }

   Register & get_R2() { return R2; }

   Register & get_CR() { return CR; }

   Register & get_AL() { return AL; }

   void set_PC(data_word_type value) {
      PC.set_value(value);
   }

   void set_SP(data_word_type value) {
      SP.set_value(value);
   }

   void set_BP(data_word_type value) {
      BP.set_value(value);
   }

   void set_R1(data_word_type value) {
      R1.set_value(value);
   }

   void set_R2(data_word_type value) {
      R2.set_value(value);
   }

   void set_CR(data_word_type value) {
      CR.set_value(value);
   }

   void set_AL(data_word_type value) {
      AL.set_value(value);
   }

   void execute(data_word_type opcode) {
      if(opcode >= code.size()) {
         throw "bad opcode...";
      }
      else 
         code[opcode]->run();
   }
};

void LOAD1::run() {
   data_word_type arg;
   arg = memory->fetch(cpu->get_PC().get_value() + sizeof(data_word_type));
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type) + sizeof(data_word_type));
   cpu->set_R1(arg);
}

void LOAD2::run() {
   data_word_type arg;
   arg = memory->fetch(cpu->get_PC().get_value() + sizeof(data_word_type));
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type) + sizeof(data_word_type));
   cpu->set_R2(arg);
}

void ADD::run() {
   Register & R1 = cpu->get_R1();
   Register & R2 = cpu->get_R2();
   for(size_t i = 0; i < BITS_SIZE; i++) {
      wire1[i]->load1(R1.get_bit(i));
      wire2[i]->load1(R2.get_bit(i));
   }
   R2.set_value(0);
   for(size_t i = 0; i < BITS_SIZE; i++) {
      R2.set_bit(i, _xor2[i]->execute());
   }
   //cpu->set_R2(cpu->get_R2().get_value() + cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void SUB::run() {
   cpu->set_R2(cpu->get_R2().get_value() -  cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void DIV::run() {
   cpu->set_R2(cpu->get_R2().get_value() /  cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void MUL::run() {
   cpu->set_R2(cpu->get_R2().get_value() *  cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void MOD::run() {
   cpu->set_R2(cpu->get_R2().get_value() %  cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void JMP::run() {
   data_word_type addr = memory->fetch(cpu->get_PC().get_value() + sizeof(data_word_type));
   cpu->set_PC(addr);
}

void CMP::run() {
   data_word_type value = memory->fetch(cpu->get_PC().get_value() + sizeof(data_word_type));
   if(cpu->get_R2().get_value() == value)
      cpu->set_AL(1);
   else 
      cpu->set_AL(0);
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type) + sizeof(data_word_type));
}

void JZ::run() {
   data_word_type addr = memory->fetch(cpu->get_PC().get_value() + sizeof(data_word_type));
   if(cpu->get_AL().get_value() == 0) 
      cpu->set_PC(addr);
   else
      cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type) + sizeof(data_word_type));
}

void JE::run() {
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void LT::run() {
   cpu->set_R2(cpu->get_R2().get_value() <  cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void GT::run() {
   cpu->set_R2(cpu->get_R2().get_value() >  cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void LTEQ::run() {
   cpu->set_R2(cpu->get_R2().get_value() <=  cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void GTEQ::run() {
   cpu->set_R2(cpu->get_R2().get_value() >=  cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void EQ::run() {
   cpu->set_R2(cpu->get_R2().get_value() ==  cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void NEQ::run() {
   cpu->set_R2(cpu->get_R2().get_value() !=  cpu->get_R1().get_value());
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void PRINT::run() {
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
   cout << cpu->get_R2().get_value() << endl;
}

void HALT::run() {
   cpu->set_CR(0);
}

void PUSH_R2::run() {
   memory->store(cpu->get_SP().get_value(), cpu->get_R2().get_value());
   cpu->set_SP(cpu->get_SP().get_value() + sizeof(data_word_type));
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

void POP_R2::run() {
   data_word_type temp = memory->fetch(cpu->get_SP().get_value() - sizeof(data_word_type));
   cpu->set_R2(temp);
   cpu->set_SP(cpu->get_SP().get_value() - sizeof(data_word_type));
   cpu->set_PC(cpu->get_PC().get_value() + sizeof(data_word_type));
}

}

#endif /* _CPU_H */
