package com.mwc.sqld.db;

import java.io.*;
import java.util.*;

import javax.xml.parsers.*;
import javax.xml.transform.stream.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.*;

import org.xml.sax.*;
import org.xml.sax.helpers.*;

import org.w3c.dom.*;
import org.w3c.dom.traversal.*;

import com.mwc.util.*;

/**
 * An XML database implementation.
 * 
 * @author Matthew W. Coan (10:38 PM 7/28/2002) mailto:matthewcoan@hotmail.com
 */
public class XMLDatabase extends SimpleDatabase {
   private String _name;
   private String _dataDirectory;
   private Hashtable _tables = new Hashtable();
   private boolean _mod = false;
   private Object _modLock = new Object();
   private MasterTable _masterTable;
   private DocumentBuilder _db;

   class XMLTable implements Table {
      private ReadWriteLock _rwLock = new ReadWriteLock();
      private String [][] _cols;
      private String _name;
      private String _fileName;
      private Document _doc;
      private Object _modLock = new Object();
      private boolean _mod = false;
      
      public boolean modified() {
         synchronized(_modLock) {
            return _mod;
         }
      }
      
      public XMLTable(String name, 
                      String fileName, 
                      String [][] cols)
      throws DBException {
         File f = new File(fileName);
         if(!f.exists()) {
            try {
               FileOutputStream fout = new FileOutputStream(fileName);
               BufferedOutputStream bout = new BufferedOutputStream(fout);
               PrintStream out = new PrintStream(bout);
               out.println("<?xml version=\"1.0\" encoding=\"utf-8\"?>");
               out.println("<!DOCTYPE table SYSTEM \"table.dtd\">");
               out.println("<table/>");
               out.flush();
               fout.close();
            }
            catch(Throwable th) {
               throw new DBException(th.getMessage());
            }
         }
         _doc = null;
         try {
            _doc = _db.parse(new File(fileName));
         } 
         catch (SAXException se) {
se.printStackTrace();
            throw new DBException(se.getMessage());
         } 
         catch (IOException ioe) {
ioe.printStackTrace();
            throw new DBException(ioe.getMessage());
         }
         _name = name;
         _fileName = fileName;
         _cols = cols;
      }
      
      class XMLCurser implements UpdateCurser {
         private int _recCount;
         private Hashtable _tbl = new Hashtable();
         private Vector _valueVector;
         private NodeIterator _recIterator;
         private Node _currentNode;
         private Hashtable _rmTbl = null;

         XMLCurser(int recCount) {
            _recCount = recCount;
            _rwLock.readLock();
            for(int i = 0; i < _cols.length; i++)
               _tbl.put(_cols[i][0].toLowerCase(), new Integer(i));
            reset();
         }
         
         public boolean remove() {
            if(_currentNode == null)
               return false;
            if(_rmTbl == null)
               _rmTbl = new Hashtable();
            _rmTbl.put(_currentNode, _currentNode);
            synchronized(_modLock) {
               _mod = true;
            }
            return true;
         }
         
         public boolean notNull(String name) {
            int index = name.indexOf('.');
            if(index >= 0) {
               String tableName = name.substring(0,index);
               if(!_name.equalsIgnoreCase(tableName))
                  return false;
               name = name.substring(index+1);
            }
            for(int i = 0; i < _cols.length; i++) {
               if(_cols[i][0].equalsIgnoreCase(name)) {
                  if(_cols[i][2].compareTo("true") == 0)
                     return true;
                  else
                     return false;
               }
            }
            return false;
         }
         
         public boolean setValue(String name, String value) {
            if(_currentNode == null)
               return false;
            int index = name.indexOf('.');
            if(index >= 0) {
               String tableName = name.substring(0,index);
               if(!_name.equalsIgnoreCase(tableName))
                  return false;
               name = name.substring(index+1);
            }
            boolean ret = false;
            synchronized(_doc) {
               DocumentTraversal traversable = (DocumentTraversal)_doc;
               NodeIterator fieldIterator = traversable.createNodeIterator(_currentNode, NodeFilter.SHOW_ELEMENT, new FieldFilter(), true);
               Node fieldNode;
               NodeIterator textIterator;
               Node textNode;
               while((fieldNode = fieldIterator.nextNode()) != null) {
                  if(fieldNode.getNodeName() != null 
                     && fieldNode.getNodeName().equalsIgnoreCase(name)) {
                     Text v = _doc.createTextNode(value.toString());
                     textIterator = traversable.createNodeIterator(fieldNode, 
                                                                   NodeFilter.SHOW_ALL, 
                                                                   null, 
                                                                   true);
                     LinkedList nodeList = new LinkedList();
                     while((textNode = textIterator.nextNode()) != null) {
                        if(textNode.getNodeName() != null) {
                           if(textNode.getNodeName().equalsIgnoreCase(name))
                              continue;
                        }
                        nodeList.add(textNode);
                     }
                     Iterator it = nodeList.iterator();
                     while(it.hasNext()) {
                        textNode = (Node)it.next();
                        textNode.getParentNode().removeChild(textNode);
                     }
                     fieldNode.appendChild(v);
                     ret = true;
                     break;
                  }
               }
            }
            if(ret) {
               synchronized(_modLock) {
                  _mod = true;
               }
            }
            return ret;
         }
         
         class RecFilter implements NodeFilter {
            public short acceptNode(Node n) {
               String nodeName = n.getNodeName();
               short ret;
               if(nodeName != null) {
                  if(nodeName.compareTo("rec") == 0)
                     ret = NodeFilter.FILTER_ACCEPT;
                  else
                     ret = NodeFilter.FILTER_SKIP;
               }
               else 
                  ret = NodeFilter.FILTER_SKIP;
               return ret;
            }
         }
         
         public boolean reset() {
            _valueVector = null;
            DocumentTraversal traversable = (DocumentTraversal)_doc;
            _recIterator = traversable.createNodeIterator(_doc, NodeFilter.SHOW_ELEMENT, new RecFilter(), true);
            _currentNode = null;
            return true;
         }
   
         public String nodeType(short type) {
            switch(type) {
            case Node.ATTRIBUTE_NODE:
               return "ATTRIBUTE_NODE";
            case Node.CDATA_SECTION_NODE:
               return "CDATA_SECTION_NODE";
            case Node.COMMENT_NODE:
               return "COMMENT_NODE";
            case Node.DOCUMENT_FRAGMENT_NODE:
               return "DOCUMENT_FRAGMENT_NODE";
            case Node.DOCUMENT_NODE:
               return "DOCUMENT_NODE";
            case Node.DOCUMENT_TYPE_NODE:
               return "DOCUMENT_TYPE_NODE";
            case Node.ELEMENT_NODE:
               return "ELEMENT_NODE";
            case Node.ENTITY_NODE:
               return "ENTITY_NODE";
            case Node.ENTITY_REFERENCE_NODE:
               return "ENTITY_REFERENCE_NODE";
            case Node.NOTATION_NODE:
               return "NOTATION_NODE";
            case Node.PROCESSING_INSTRUCTION_NODE:
               return "PROCESSING_INSTRUCTION_NODE";
            case Node.TEXT_NODE:
               return "TEXT_NODE";
            default:
               return "<UNKNOWN>";
            }
         }
         
         class FieldFilter implements NodeFilter {
            public short acceptNode(Node n) {
               String nodeName = n.getNodeName();
               short ret;
               if(nodeName != null) {
                  if(nodeName.compareTo("rec") != 0)
                     ret = NodeFilter.FILTER_ACCEPT;
                  else
                     ret = NodeFilter.FILTER_REJECT;
               }
               else 
                  ret = NodeFilter.FILTER_SKIP;
               return ret;
            }
         }

         public boolean next() {
            synchronized(_doc) {
               _currentNode = _recIterator.nextNode();
               if(_currentNode == null)
                  return false;
               _valueVector = new Vector();
               DocumentTraversal traversable = (DocumentTraversal)_doc;
               NodeIterator fieldIterator = traversable.createNodeIterator(_currentNode, NodeFilter.SHOW_ELEMENT, new FieldFilter(), true);
               Node fieldNode;
               NodeIterator textIterator;
               Node textNode;
               StringBuffer value;
               while((fieldNode = fieldIterator.nextNode()) != null) {
                  value = new StringBuffer();
                  textIterator = traversable.createNodeIterator(fieldNode, 
                                                                NodeFilter.SHOW_TEXT 
                                                                | NodeFilter.SHOW_ENTITY_REFERENCE 
                                                                | NodeFilter.SHOW_CDATA_SECTION, 
                                                                null, 
                                                                true);
                  while((textNode = textIterator.nextNode()) != null) {
                     value.append(textNode.getNodeValue());
                  }
                  _valueVector.addElement(value.toString());
               }
               return true;
            }
         }
            
         public boolean close() {
            if(_rmTbl != null) {
               synchronized(_doc) {
                  Enumeration enum = _rmTbl.keys();
                  while(enum.hasMoreElements()) {
                     _currentNode = (Node)enum.nextElement();
                     _currentNode.getParentNode().removeChild(_currentNode);
                  }
               }
            }
            _currentNode = null;
            _rwLock.readUnlock();
            return true;
         }
   
         public int recCount() {
            return _recCount;
         }

         public String getValue(String name) {
            Integer index;
            int i = name.indexOf('.');
            if(i < 0) 
               index = (Integer)_tbl.get(name.toLowerCase());
            else {
               String tblName = name.substring(0,i);
               if(tblName.equalsIgnoreCase(_name)) {
                  name = name.substring(i+1);
                  index = (Integer)_tbl.get(name.toLowerCase());
               }
               else
                  index = null;
            }
            if(index == null)
               return null;
            return (String)_valueVector.elementAt(index.intValue());
         }
            
         public String getValue(int index) {
            return (String)_valueVector.elementAt(index);
         }
            
         public String getType(String colName) {
            int i = colName.indexOf('.');
            if(i >= 0) {
               String tblName = colName.substring(0,i);
               if(!tblName.equalsIgnoreCase(_name))
                  return null;
               colName = colName.substring(i+1);
            }
            for(i = 0; i < _cols.length; i++) {
               if(colName.equalsIgnoreCase(_cols[i][0])) {
                  return _cols[i][1];
               }
            }
            return null;
         }
            
         public int size() {
            return _valueVector.size();
         }
            
         public String[][] cols() {
            return _cols;
         }
      }
      
      public String name() {
         return _name;
      }
      
      public Curser getNewCurser() {
         return new XMLCurser(-1);
      }
      
      public boolean insert(LinkedList values) {
         _rwLock.writeLock();
         try {
            Element rec = _doc.createElement("rec");
            Iterator iterator = values.iterator();
            Object value;
            Element field;
            Text data;
            int index = 0;
            while(iterator.hasNext()) {
               value = iterator.next();
               field = _doc.createElement(_cols[index][0]);
               data = _doc.createTextNode(value.toString());
               field.appendChild(data);
               rec.appendChild(field);
               index++;
            }
            DocumentTraversal traversable = (DocumentTraversal)_doc;
            NodeIterator nodeIterator = traversable.createNodeIterator(_doc, NodeFilter.SHOW_ELEMENT, null, true);
            Node node;
            while((node = nodeIterator.nextNode()) != null) {
               if(node.getNodeName() != null) {
                  if(node.getNodeName().compareTo("table") == 0) {
                     node.appendChild(rec);
                     return true;
                  }
               }
            }
            return false;
         }
         finally {
            synchronized(_modLock) {
               _mod = true;
            }
            _rwLock.writeUnlock();
         }
      }
      
      public boolean save() {
         _rwLock.writeLock();
         try {
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            transformer.setOutputProperty(OutputKeys.INDENT, "yes");
            transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "no");
            transformer.setOutputProperty(OutputKeys.ENCODING, "utf-8");
            transformer.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, "table.dtd");
            transformer.transform(new DOMSource(_doc), new StreamResult(new File(_fileName)));
         }
         catch(Throwable th) {
th.printStackTrace();
            return false;
         }
         finally {
            synchronized(_modLock) {
               _mod = false;
            }
            _rwLock.writeUnlock();
         }
         return true;
      }
      
      public boolean drop() {
         return false;
      }
      
      public String[][] cols() {
         return _cols;
      }
   }
   
   private static class XMLErrorHandler implements ErrorHandler {
      LinkedList errorList = new LinkedList();
      
      /**
       * Returns a string describing parse exception details
       */
      private String getParseExceptionInfo(SAXParseException spe) {
         errorList.add(spe);
         String systemId = spe.getSystemId();
         if (systemId == null) {
             systemId = "null";
         }
         String info = "URI=" + systemId +
             " Line=" + spe.getLineNumber() +
             ": " + spe.getMessage();
         return info;
      }

      // The following methods are standard SAX ErrorHandler methods.
      // See SAX documentation for more info.

      public void warning(SAXParseException spe) throws SAXException {
         errorList.add(spe);
      }
              
      public void error(SAXParseException spe) throws SAXException {
         errorList.add(spe);
      }

      public void fatalError(SAXParseException spe) throws SAXException {
         errorList.add(spe);
      }
   }

   public XMLDatabase(String name, String dataDirectory) 
   throws DBException {
      // Step 1: create a DocumentBuilderFactory and configure it
      DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

      // Optional: set various configuration options
      dbf.setValidating(true);
      dbf.setIgnoringComments(true);
      dbf.setIgnoringElementContentWhitespace(true);
      dbf.setCoalescing(true);
      // The opposite of creating entity ref nodes is expanding them inline
      dbf.setExpandEntityReferences(false);
      
      _db = null;
      try {
          _db = dbf.newDocumentBuilder();
      } 
      catch (ParserConfigurationException pce) {
         throw new DBException(pce.getMessage());
      }
      
      _db.setErrorHandler(new XMLErrorHandler());
      
      _name = name;
      _dataDirectory = dataDirectory;
      File f = new File(_dataDirectory);
      if(!f.exists())
         f.mkdir();
      f = new File(_dataDirectory + "/table.dtd");
      if(!f.exists()) {
         try {
            FileOutputStream fout = new FileOutputStream(f);
            BufferedOutputStream bout = new BufferedOutputStream(fout);
            PrintStream out = new PrintStream(bout);
            out.println("<?xml version='1.0' encoding='us-ascii'?>");
            out.println("<!ELEMENT table (rec*)>");
            out.println("<!ELEMENT rec (ANY+)>");
            out.flush();
            fout.close();
         }
         catch(Throwable th) {
            throw new DBException(th.getMessage());
         }
      }
      f = new File(_dataDirectory + "/data.dat");
      if(f.exists()) {
         _masterTable = new MasterTable(f);
         Curser c = _masterTable.getNewCurser();
         try {
            Table t;
            String tableName, tableFileName, lastTableName = "";
            String cols[][];
            int i;
            while(c.next()) {
               tableName = c.getValue("TableName");
               if(!lastTableName.equalsIgnoreCase(tableName)) {
                  lastTableName = tableName;
                  cols = _masterTable.loadCols(tableName);
                  tableFileName = _dataDirectory + "/" + tableName + ".xml";
                           
System.out.println("LOAD-TABLE("+tableName+")");                           
                  t = new XMLTable(tableName, tableFileName, cols);
                  _tables.put(t.name().toLowerCase(), t);
               }
            }
         }
         finally {
            if(c != null)
               c.close();
         }
      }
   }
   
   public Table createTable(String name, String cols[][]) {
      Table tbl = getTable(name);
      if(tbl != null)
         return null;
      String fileName = _dataDirectory + "/" + name + ".xml";
      try {
         tbl = new XMLTable(name, fileName, cols);
      }
      catch(DBException dbe) {
         return null;
      }
      synchronized(_tables) {
         _tables.put(name.toLowerCase(), tbl);
      }
      synchronized(_modLock) {
         _mod = true;
      }
      return tbl;
   }   
   
   public String name() {
      return _name;
   }
   
   public File getTempFile() {
      File f = new File(_dataDirectory + "/temp");
      if(!f.exists())
         f.mkdir();
      try {
         return File.createTempFile("TEMP_", ".dat", f);
      }
      catch(IOException ioe) {
         return null;
      }
   }
   
   public Table getTable(String table) {
      if(table.equalsIgnoreCase("Master"))
         return _masterTable;
      synchronized(_tables) {
         return (Table)_tables.get(table.toLowerCase());
      }
   }
   
   public String[] tableNames() {
      synchronized(_tables) {
         String ret[] = new String[_tables.size()];
         Enumeration e = _tables.elements();
         int i = 0;
         while(e.hasMoreElements()) {
            ret[i] = ((Table)e.nextElement()).name();
            i++;
         }
         return ret;
      }
   }
   
   public boolean dropTable(String table) {
      return false;
   }
   
   public boolean save() {
      try {
         _masterTable.saveState(_dataDirectory, _tables);
         synchronized(_modLock) {
            _mod = false;
         }
         sync();
      }
      catch(Throwable throwable) {
         return false;
      }
      return true;
   }
   
   public boolean modified() {
      synchronized(_modLock) {
         return _mod;
      }
   }
   
   public void sync() {
      Enumeration enumeration = _tables.elements();
      XMLTable xmlTable;
      while(enumeration.hasMoreElements()) {
         xmlTable = (XMLTable)enumeration.nextElement();
         if(xmlTable.modified())
            xmlTable.save();
      }
   }
   
   public void close() {
   }
}