#ifndef _D_LIST_H 
#define _D_LIST_H

/*

EXTERNAL DOUBLY LINKED LIST 

Author: Matthew W. Coan
Date: Wed Jun 26 16:32:29 EDT 2013
Version: 1.0.0a

*/

#include "bfstream.h"
#include <iostream>
#include <typeinfo>

namespace db_tools {

using namespace std;

extern d_database * p_db;

class d_list_node {
public:
   offset_type next;
   offset_type prev;
   offset_type data;

   d_list_node() { 
      next = -1L; 
      prev = -1L;
      data = -1L;
   }

   d_list_node(const d_list_node & n) {
      data = n.data;
      next = n.next; 
      prev = n.prev;
   }

   d_list_node & operator=(const d_list_node & n) {
      data = n.data;
      next = n.next;
      prev = n.prev;
      return *this;
   }

   void read(bfstream & fin) {
      fin >> next >> prev >> data;
      fin.clear();
   }

   void write(bfstream & fout) {
      fout << next << prev << data;
      fout.flush();
   }
};

template< class T >
class d_list_iterator {
   d_list_node node;
   bfstream * fptr;
   offset_type count;
   offset_type addr;
   T temp;

public:
   d_list_iterator(d_database & db, offset_type start, offset_type size) {
      fptr = (bfstream*)&(db.get_stream());
      addr = start;
      count = size;
      fptr->seek_current(addr);
      node.read(*fptr);
   }

   d_list_iterator(const d_list_iterator & it) {
      node = it.node;
      fptr = it.fptr;
      count = it.count;
      addr = it.addr;
   }

   d_list_iterator() {
      fptr = 0;
      count = 0;
      addr = -1L;
      node.next = -1L;
      node.prev = -1L;
   }

   ~d_list_iterator() {
   }

   d_list_node & get_node() { 
      return node; 
   }

   const offset_type & get_count() {
      return count;
   } 
 
   const offset_type & get_address() { 
      return addr; 
   }

   void read(bfstream & fin) { node.read(fin); }
   void write(bfstream & fout) { node.write(fout); }

   d_list_iterator & operator++(int) {
      addr = node.next;
      fptr->seek_current(node.next);
      node.read(*fptr);
      count++;
      fptr->clear();
      return *this;
   }

   d_list_iterator & operator--(int) {
      addr = node.prev;
      fptr->seek_current(node.prev);
      node.read(*fptr);
      count--;
      fptr->clear();
      return *this;
   }

   T & operator*() {
      fptr->seekg(addr, ios::beg);
      node.read(*fptr);
      fptr->clear();
      fptr->seek_current(node.data);
      *fptr >> temp;
      fptr->clear();
      return temp;
   }

   T * operator->() {
      fptr->seek_current(addr);
      node.read(*fptr);
      fptr->clear();
      fptr->seek_current(node.data);
      *fptr >> temp;
      fptr->clear();
      return &temp;
   }

   void set_address(const offset_type & off) {
      addr = off;
   }

   bool operator==(const d_list_iterator & it) {
      bool ret = it.count == count;
      return ret;
   }

   bool operator!=(const d_list_iterator & it) {
      bool ret = it.count != count;
      return ret;
   }

   d_list_iterator & operator=(const d_list_iterator & it) {
      count = it.count;
      node = it.node;
      fptr = it.fptr;
      addr = it.addr;
      return *this;
   }
};

template< class T >
class d_list_reverse_iterator {
public:
   d_list_node node;
   bfstream * fptr;
   offset_type count;
   offset_type addr;
   T data;

public:
   d_list_reverse_iterator(d_database & db, offset_type start, offset_type size) {
      fptr = (bfstream*)&(db.get_stream());
      addr = start;
      count = size;
      fptr->seek_current(addr);
      node.read(*fptr);
   }

   d_list_reverse_iterator(const d_list_reverse_iterator & cp) {
      fptr = cp.fptr;
      addr = cp.addr;
      count = cp.count;
      node = cp.node;
   }

   d_list_reverse_iterator() {
      fptr = 0;
      count = 0;
      addr = -1;
      node.next = -1L;
      node.prev = -1L;
   }

   ~d_list_reverse_iterator() {
   }

   d_list_reverse_iterator & operator++(int) {
      addr = node.prev;
      fptr->seek_current(addr);
      node.read(*fptr);
      count++;
      fptr->clear();
      return *this;
   }

   d_list_reverse_iterator & operator--(int) {
      addr = node.next;
      fptr->seek_current(addr);
      node.read(*fptr);
      count--;
      fptr->clear();
      return *this;
   }

   T & operator*(int) {
      fptr->seek_current(addr);
      node.read(*fptr);
      fptr->clear();
      T * p = new T;
      fptr->seek_current(node.data);
      *fptr >> data;
      return data;
   }

   T * operator->() {
      fptr->seek_current(addr);
      node.read(*fptr);
      fptr->clear();
      T * p = new T;
      fptr->seek_current(node.data);
      *fptr >> *p; 
      return p;
   }

   bool operator==(const d_list_reverse_iterator & it) {
      bool ret = it.count == count;
      return ret;
   }

   bool operator!=(const d_list_reverse_iterator & it) {
      bool ret = it.count != count;
      return ret;
   }

   d_list_reverse_iterator & operator=(const d_list_reverse_iterator & it) {
      count = it.count;
      node = it.node;
      fptr = it.fptr;
      addr = it.addr;
      return *this;
   }
};

template< class T >
inline
offset_type 
obj_size_default(const T & t) 
{
   offset_type ret = sizeof(t);
   return ret;
}

template< class T >
inline
offset_type 
list_size(const T & t) 
{
   offset_type ret = sizeof(offset_type) + sizeof(offset_type) + sizeof(offset_type);
   return ret;
}

template< class T, class obj_size_type = offset_type (*)(const T & t) >
class d_list {
public:
   typedef d_list_iterator< T > iterator;
   typedef d_list_reverse_iterator< T > reverse_iterator;

private:
   d_database * dptr;
   d_alloc * allocator;
   bfstream * fptr;
   offset_type head,tail;
   offset_type the_size;
   offset_type start;
   obj_size_type obj_size;

   void save_size() {
      fptr->seek_current(start);
      *fptr << the_size << head << tail;
      fptr->flush();
   }

   void swap(iterator & a, iterator & b) {
      offset_type temp = a.get_node().data;
      a.get_node().data = b.get_node().data;
      b.get_node().data = temp;
      fptr->seek_current(a.get_address());
      a.write(*fptr);
      fptr->seek_current(b.get_address());
      b.write(*fptr);
   }

public:
   void set_size(const offset_type & off) { the_size = off; }
   void set_start(const offset_type & s) { start = s; }
   void set_head(const offset_type & h) { head = h; }
   void set_tail(const offset_type & t) { tail = t; }
   void set_db(d_database & db) { 
      dptr = &db; 
      allocator = db.get_allocator();
      fptr = &dptr->get_stream();
   }
   void set_size_function(obj_size_type os) { obj_size = os; } 
   offset_type get_size() const { return the_size; }
   offset_type get_start() const { return start; }
   offset_type get_head() const { return head; }
   offset_type get_tail() const { return tail; }
   d_list(d_database & db, offset_type off, obj_size_type os = obj_size_default) {
      dptr = &db;
      head = -1L;
      tail = -1L;
      the_size = 0L;
      start = off;
      obj_size = os;
      fptr = &dptr->get_stream();
      fptr->seek_current(start);
      *fptr >> the_size >> head >> tail;
      fptr->clear();
      allocator = db.get_allocator();
   }

   d_list(const d_list & cp) {
      dptr = cp.dptr;
      head = cp.head;
      tail = cp.tail;
      the_size = cp.the_size;
      start = cp.start;
      obj_size = cp.obj_size;
      fptr = &dptr->get_stream();
   }

   d_list() {
      dptr = 0;
   }

   void init() {
      fptr->seek_current(start);
      *fptr >> the_size >> head >> tail;
      fptr->clear();
      if(the_size == -1) the_size = 0;
   }

   d_database & get_db() { 
      return *dptr; 
   }


   void pop_front() {
      if(the_size == 0) 
         return;

      d_list_node n;

      fptr->seek_current(head);
      n.read(dptr->get_stream());

      allocator->deallocate(head);

      head = n.next;
      if(head != -1L) {
         fptr->seek_current(head);
         n.read(dptr->get_stream());

         n.prev = -1L;

         fptr->seek_current(head);
         n.write(dptr->get_stream());
      }

      the_size--;

      save_size();
   }

   void pop_back() {
      if(the_size == 0)
         return;

      d_list_node n;

      fptr->seek_current(tail);
      n.read(dptr->get_stream());

      allocator->deallocate(tail);

      tail = n.prev;

      fptr->seek_current(tail);
      n.read(dptr->get_stream());

      n.next = -1L;
      fptr->seekg(tail, ios::beg);
      n.write(dptr->get_stream());

      the_size--;

      save_size();
   }

   void push_front(const T & data) {
      d_list_node n,n2;
      n.data = allocator->allocate(obj_size(data));
      fptr->seek_current(n.data);
      *fptr << data;
      n.next = head;
      n.prev = -1L;

      offset_type size = sizeof(offset_type) + sizeof(offset_type) + sizeof(offset_type);
      //size += obj_size(data);

      offset_type address = allocator->allocate(size);

      fptr->seekg(head + sizeof(offset_type), ios::beg);
      *fptr << address;

      fptr->seek_current(address);
      n.write(dptr->get_stream());

      head = address;

      the_size++;

      if(the_size == 1) {
         tail = address;
      }

      save_size();
   }

   void push_back(const T & data) {
      d_list_node n,n2;
      n.data = allocator->allocate(obj_size(data));
      fptr->seek_current(n.data);;
      *fptr << data;
      n.next = -1L;
      n.prev = tail;

      offset_type size = sizeof(offset_type) + sizeof(offset_type) + sizeof(offset_type);

      offset_type address = allocator->allocate(size);

      if(the_size > 0L) {
         fptr->seekg(tail, ios::beg);
         *fptr << address;
      }

      fptr->seek_current(address);
      n.write(dptr->get_stream());

      tail = address;

      the_size++;


      if(the_size == 1) {
         head = address;
      }

      save_size();
   }

   T front() {
      fptr->seek_current(head);
      d_list_node n;
      n.read(dptr->get_stream());
      fptr->seek_current(n.data);
      T data;
      *fptr >> data;
      return data;
   }
 
   T back() {
      fptr->seek_current(tail);
      d_list_node n;
      n.read(dptr->get_stream());
      T data;
      fptr->seek_current(n.data);
      *fptr >> data;
      return data;
   }

   reverse_iterator rbegin() {
      reverse_iterator it(*dptr, tail, 0L);
      return it;
   }

   reverse_iterator rend() {
      reverse_iterator it(*dptr, head, the_size);
      return it;
   }

   iterator begin() {
      iterator it(*dptr, head, 0L);
      return it;
   }

   iterator end() {
      iterator it(*dptr, tail, the_size);
      return it;
   }

   void erase(iterator ptr) {
      d_list_node n0,n1,n2;

      allocator->deallocate(ptr.get_node().data);

      fptr->seek_current(ptr.get_address());
      n1.read(*fptr);
      fptr->clear();

      if(n1.prev != -1L) {
         fptr->seek_current(n1.prev);
         n0.read(*fptr);
         fptr->clear();
      }

      if(n1.next != -1L) {
         fptr->seek_current(n1.next);
         n2.read(*fptr);
         fptr->clear();
      }

      if(n1.prev != -1L) {
         n0.next = n1.next;
         fptr->seek_current(n1.prev);
         n0.write(*fptr);
      }

      if(n1.next != -1L) {
         n2.prev = n1.prev;
         fptr->seek_current(n1.next);
         n2.write(*fptr);
      }

      allocator->deallocate(ptr.get_address());

      if(head == ptr.get_address()) {
         head = n1.next;
      }

      if(tail == ptr.get_address()) {
         tail = n1.prev;
      }

      the_size--;

      save_size();
   }

   void erase(iterator ptr1, iterator ptr2) {
      d_list_node start,end,n;
      offset_type start_addr,end_addr,temp,temp2 = -1L;

      start_addr = ptr1.get_address();
      end_addr = ptr2.get_address();

      fptr->seek_current(start_addr);
      start.read(*fptr);

      fptr->seek_current(end_addr);
      end.read(*fptr);

      if(end.next != -1L) {
         temp2 = end.next;
      }

      if(start.prev != -1L) {
         temp = start.prev;
         fptr->seek_current(temp);
         n.read(*fptr);
         n.next = temp2;
         fptr->seek_current(temp);
         n.write(*fptr);
         the_size--;
      }

      if(end.next != -1L) {
         temp = end.next;
         fptr->seek_current(temp);
         n.read(*fptr);
         n.prev = start_addr;
         fptr->seek_current(temp);
         n.write(*fptr);
         the_size--;
      }

      fptr->seek_current(start_addr);
      n.read(*fptr);
      //the_size--; 
      
      while(n.next != -1L) {
         if(n.next == end_addr)
            break;
         offset_type off = n.next;
         fptr->seek_current(off);
         n.read(*fptr);
         allocator->deallocate(off);
         allocator->deallocate(n.data);
         the_size--;
      }

      if(head == start_addr)  {
         head = -1L;
         the_size--;
      }

      if(tail == end_addr) {
         tail = -1L;
         the_size--;
      }

      save_size();
   }

   void insert(iterator ptr, const T & data) {
      d_list_node n1,n2,n3;

      if(ptr.get_count() == the_size) {
         push_back(data);
      }
      else {
         fptr->seek_current(ptr.get_address());
         n1.read(*fptr);

         if(n1.prev != -1L) {
            fptr->seek_current(n1.prev);
            n2.read(*fptr);
         }

         offset_type size = sizeof(offset_type)
         + sizeof(offset_type) + sizeof(offset_type);

         offset_type d = allocator->allocate(size);

         offset_type addr = allocator->allocate(obj_size(data));

         fptr->seek_current(addr);
         *fptr << data;

         n3.data = addr;
         n3.next = ptr.get_address();
         n3.prev = n1.prev;
         fptr->seek_current(d);
         n3.write(*fptr);

         if(n1.prev != -1L) {
            n2.next = d;
            fptr->seek_current(n1.prev);
            n2.write(*fptr);
         }

         n1.prev = d;
         fptr->seek_current(ptr.get_address());
         n1.write(*fptr);

         if(head == ptr.get_address()) {
            head = d;
         }

         the_size++;

         save_size();
      }
   }

   template< class Comp > 
   void sort(Comp comp) {
      for(iterator p1 = begin(); p1 != end(); p1++) {
         for(iterator p2 = p1; p2 != end(); p2++) {
            if(p1 != p2) {
               if(comp(*p2,*p1)) {
                  swap(p1,p2);
               }
            }
         }
      }
   }

   void sort() {
      sort(less< T >( ));
   }
void clear() {
      erase(begin(), end());
   }

   const offset_type & size() {
      return the_size;
   }

   d_list< T, obj_size_type > & operator=(d_list< T, obj_size_type > & right) {
      clear();
      for(iterator p = right.begin(); p != right.end(); p++)
         push_back(*p);
      return *this;
   }

   void write(bfstream & fout) const {
      fout << the_size << head << tail;
      fout.flush();
   }

   void read(bfstream & fin, d_database & db) {
      offset_type start = fin.tellg();
      offset_type size,head,tail;
      fin >> size >> head >> tail;
      set_size(size);
      set_start(start);
      set_head(head);
      set_tail(tail);
      set_db(db);
      set_size_function(list_size);
      init();
   }
};

template< class T, class obj_size_type >
inline
bfstream & 
operator<<(bfstream & fout, const d_list< T, obj_size_type > & data) 
{
   data.write(fout);
   return fout;
}


template< class T, class obj_size_type >
inline
bfstream & 
operator>>(bfstream & fin, d_list< T, obj_size_type > & data) 
{
   data.read(fin, *p_db);
   return fin;
}


}

#endif /* _D_LIST_H */
