#define _XOPEN_SOURCE
#include <arpa/inet.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netdb.h>
#include <curses.h>
#include <signal.h>
#include <unistd.h>
#include <time.h>
#include <stdio.h>
#include <cmath>
#include <cctype>
#include <cstdlib>
#include <cstring>
#include <list>
#include <map>
#include "exploit_host.h"
#include "properties_file.h"

//
// Worm Hunter/Killer!
//
// Thursday 12/19/2002
// Matthew W. Coan
//

extern "C" {
char *strptime(const char *s, const char *format, struct tm *tm);
int snprintf(char *str, size_t size, const char *format, ...);
int inet_aton(const char *cp, struct in_addr *inp);
}

class str_less {
public:
   bool operator()(const char * left, const char * right) const {
      return strcmp(left, right) < 0;
   }
};

//
// Web server log file class:
//
class LogFile {
public:
   struct log_file_access_exception { };
   struct log_format_exception { 
      log_format_exception() 
      { }

      log_format_exception(const char * file, const int line) {
         fprintf(stderr, "LOG_FORMAT_EXCEPTION(%s @ %d)\n", file, line);
      }
   };
   struct properties_format_exception { };

   // 
   // Worm definition class
   //
   class WormDefinition {
   public:
      typedef pair< const char *, const size_t > worm_token_type;
      typedef list< worm_token_type > worm_token_list_type;
   private:
      const char * _name;
      const worm_token_list_type * _p_worm_token_list;
   public:
      WormDefinition(const char * name, 
                     const worm_token_list_type * p_worm_token_list)
      : _name(name), 
        _p_worm_token_list(p_worm_token_list)
      { }

      ~WormDefinition() {
         delete _p_worm_token_list;
      }

      const char * worm_name() const { return _name; }

      bool is_worm(const char * request) {
         size_t request_sz = strlen(request);
         for(worm_token_list_type::const_iterator p = _p_worm_token_list->begin();
             p != _p_worm_token_list->end(); p++) {
            if(p->second <= request_sz) {
               if(strstr(request, p->first) != NULL)
                  return true;
            } 
         }
         return false;
      }
   };

   typedef list< WormDefinition * > worm_definition_list_type;

   // 
   // The data we need in each entry.
   // 
   class Entry {
      char * _rec;
      char * _ip;
      char * _request;
      time_t _when;
      char * _when_string;
      int _response_code;
      WormDefinition * _p_def;
   public:
      Entry(char * rec, 
            const char * df, 
            const char * lf)
      throw(log_format_exception);
      ~Entry() { delete [] _rec; }

      const char * ip() const { return _ip; }
      const char * request() const { return _request; }
      const int response_code() const { return _response_code; }
      bool is_worm(const worm_definition_list_type & def_list);
      const time_t & when() const { return _when; }
      const char * worm_name() const { return _p_def->worm_name(); }
      const char * when_string() {
         if(_when_string)
            return _when_string; 
         char buffer[1024];
         memset(buffer, 0, 1023);
         strftime(buffer, 1023, "%b-%d-%Y %r", localtime(&_when));
         return (_when_string = new_strdup(buffer));
      }
   };
 
   typedef list< LogFile::Entry * > entry_list_type;
   typedef map< const char *, entry_list_type *, str_less > entry_map_type;
private:
   const char * _file_name;
   const char * _date_format;
   const char * _log_format;
   entry_map_type _entry_map;
   worm_definition_list_type _worm_def_list;

   void _load_log_data() 
   throw(log_file_access_exception, log_format_exception);

   void _load_worm_def_list(const properties_file & props)
   throw(properties_format_exception);
public:
   LogFile(const char * fn,  // log file name
           const char * df,  // date format
           const char * lf,  // log format
           const properties_file & props) // properties w/ worm defs
   throw(log_file_access_exception, 
         log_format_exception,
         properties_format_exception)
   :_file_name(fn),
    _date_format(df),
    _log_format(lf)
   { _load_worm_def_list(props); _load_log_data(); }

   ~LogFile();

   const char * file_name() const { return _file_name; }

   entry_map_type::iterator begin() { return _entry_map.begin(); }
   entry_map_type::iterator end() { return _entry_map.end(); }
   size_t size() { return _entry_map.size(); }
};

void
LogFile::_load_worm_def_list(const properties_file & props)
throw(properties_format_exception)
{
   const char * temp = props["worm.definition.list.count"];
   if(!temp)
      throw properties_format_exception();
   int worm_list_sz = atoi(temp);
   if(worm_list_sz <= 0)
      throw properties_format_exception();
   const size_t SZ = 1024;
   char buffer[SZ];
   int worm_token_list_sz;
   int i,j;
   WormDefinition::worm_token_list_type * p_token_list;
   WormDefinition * p_worm_def;
   size_t sz;
   const char * name;
   bool prop_error = false;
   
   for(i = 1; i <= worm_list_sz; i++) {
      memset(buffer, 0, SZ);
      snprintf(buffer, SZ - 1, "worm.definition.%d.name", i);
      name = props[buffer];
      if(!name) {
         prop_error = true;
         break;
      }
      memset(buffer, 0, SZ);
      snprintf(buffer, SZ - 1, "worm.definition.%d.token.count", i);
      temp = props[buffer];
      if(!temp) {
         prop_error = true;
         break;
      }
      worm_token_list_sz = atoi(temp);
      if(worm_token_list_sz <= 0)
         throw properties_format_exception();
      p_token_list = new WormDefinition::worm_token_list_type();
      for(j = 1; j <= worm_token_list_sz; j++) {
         memset(buffer, 0, SZ);
         snprintf(buffer, SZ - 1, "worm.definition.%d.token.%d", i, j);
         temp = props[buffer];
         if(!temp) {
            prop_error = true;
            break;
         }
         p_token_list->push_back(LogFile::WormDefinition::worm_token_type(temp, strlen(temp)));
      }
      if(prop_error)
         break;
      p_worm_def = new WormDefinition(name, p_token_list);
      _worm_def_list.push_back(p_worm_def);
   }

   if(prop_error) {
      for(worm_definition_list_type::iterator p = _worm_def_list.begin();
          p != _worm_def_list.end(); p++)
         delete (*p);
      _worm_def_list = LogFile::worm_definition_list_type();
      throw properties_format_exception();
   }
}

bool
LogFile::Entry::is_worm(const worm_definition_list_type & def_list) 
{
   _p_def = 0;

   if(!_request) 
      return false;

   for(worm_definition_list_type::const_iterator p = def_list.begin();
       p != def_list.end(); p++) {
      if((*p)->is_worm(_request)) {
         _p_def = (*p);
         return true;
      }
   }
      
   return false;
}

bool
is_host_char(char ch) 
{
   if((ch >= '0' && ch <= '9') 
      || (ch >= 'a' && ch <= 'z')
      || (ch >= 'A' && ch <= 'Z')
      || (ch == '.')
      || (ch == '-') 
      || (ch == ':'))
      return true;
   return false;
}

bool
is_date_time_char(char ch)
{
   if((ch >= '0' && ch <= '9')
      || (ch >= 'a' && ch <= 'z')
      || (ch >= 'A' && ch <= 'Z')
      || (ch == ':')
      || (ch == '/')
      || (ch == '-')
      || (ch == ' '))
      return true;
   return false;
}

LogFile::Entry::Entry(char * rec,
                      const char * df,
                      const char * lf)
throw(LogFile::log_format_exception)
:_rec(rec), _ip(0), _request(0), 
 _response_code(0), _when_string(0),
 _p_def(0) 
{
   const char * p_log_format = lf;
   char * p_rec = rec;

   const char HOST[] = "<<HOST>>";
   char * p_host = strstr(lf, HOST);

   const char TIME[] = "<<TIME>>";
   char * p_time = strstr(lf, TIME);

   const char REQUEST[] = "<<REQUEST>>";
   char * p_request = strstr(lf, REQUEST);

   const char STATUS_CODE[] = "<<STATUS_CODE>>";
   char * p_status_code = strstr(lf, STATUS_CODE);

   if(p_host == NULL || p_time == NULL 
      || p_request == NULL || p_status_code == NULL)
      throw log_format_exception(__FILE__, __LINE__);

   char * phost = 0;
   char * ptime = 0;
   char * prequest = 0;
   char * pstatus_code = 0;
   char * temp;
   char * temp2;
   size_t length = strlen(p_rec);
   size_t current = 0;

   const char FMT_WORD[] = "<<WORD>>";

   while(*p_rec != '\0' && *p_log_format != '\0') {
      if(p_log_format == p_host) {
         phost = p_rec;
         for(temp = p_rec; is_host_char(*temp); temp++)
            ;
         *temp = '\0';
         current += strlen(phost) + 1;
         p_rec = temp + 1;
         p_log_format += sizeof(HOST);
      }
      else if(p_log_format == p_time) {
         ptime = p_rec;
         size_t space_count = 0;
         for(const char * temp0 = df; *temp0 != '\0'; temp0++) {
            if(*temp0 == ' ')
               space_count++;
         }
         for(temp = p_rec; is_date_time_char(*temp); temp++) {
            if(*temp == ' ') {
               if(!space_count)
                  break;
               space_count--;
            }
         }
         *temp = '\0';
         current += strlen(ptime) + 1;
         p_rec = temp + 1;
         p_log_format += sizeof(TIME);
      }
      else if(p_log_format == p_request) {
         prequest = p_rec;
         if(*p_rec == '\"') {
            for(temp = p_rec + 1; *temp != '\"' && *temp != '\0'; temp++)
               ;
            *temp = '\0';
            current += strlen(prequest) + 1;
            p_rec = temp + 1;
            while(*prequest != '\0' && *prequest != '/')
               prequest++;
            for(temp = prequest; *temp != '\0' 
                && *temp != '\"' && !isspace(*temp); temp++)
               ;
            *temp = '\0';
         }
         else {
            for(temp = p_rec; !isspace(*temp); temp++)
               ;
            *temp = '\0';
            current += strlen(prequest) + 1;
            p_rec = temp + 1;
         }
         p_log_format += sizeof(REQUEST);
      }
      else if(p_log_format == p_status_code) {
         pstatus_code = p_rec;
         for(temp = p_rec; isdigit(*temp); temp++)
            ;
         *temp = '\0';
         current += strlen(pstatus_code) + 1;
         p_rec = temp + 1;
         p_log_format += sizeof(STATUS_CODE);
      }
      else {
         if(strlen(p_log_format) >= (sizeof(FMT_WORD) - 1)) {
            if(strncmp(p_log_format, FMT_WORD, sizeof(FMT_WORD) -1) == 0) {
               while(*p_rec != '\0' && !isspace(*p_rec)) {
                  p_rec++;
                  current++;
               }
               p_log_format += sizeof(FMT_WORD) - 1;
            }
            else {
               p_log_format++;
               p_rec++;
               current++;
            }
         }
         else {
            p_log_format++;
            p_rec++;
            current++;
         }
      }

      if(current >= length)
         break;
   }

   if(!phost || !ptime || !prequest || !pstatus_code)
      throw log_format_exception(__FILE__, __LINE__);

   _ip = phost;

   tm date_time;
   memset(&date_time, 0, sizeof(tm));
   if(strptime(ptime, df, &date_time) == NULL)
      throw log_format_exception();
   _when = mktime(&date_time);

   _request = prequest;

   _response_code = atoi(pstatus_code);
}

LogFile::~LogFile()
{
   for(entry_map_type::iterator p0 = _entry_map.begin(); 
       p0 != _entry_map.end(); p0++) {
      for(entry_list_type::iterator p = p0->second->begin(); 
          p != p0->second->end(); p++) {
         delete (*p);
      }
      delete ((*p0).second);
   }
   for(worm_definition_list_type::iterator p0 = _worm_def_list.begin();
       p0 != _worm_def_list.end(); p0++)
      delete (*p0);
}

int
round(float f)
{
   int i = (int)f;
   float d = f - i;
   if(d > 0.5f)
      return i+1;
   return i;
}

void 
LogFile::_load_log_data()
throw(log_file_access_exception, log_format_exception)
{
   const size_t BUF_SZ = 1024 * 12;
   char buffer[BUF_SZ];
   FILE * fin = fopen(_file_name, "r");
   char * temp;
   size_t temp_sz;
   struct stat statbuf;

   if(fin == NULL)
      throw log_file_access_exception();

   memset(&statbuf, 0, sizeof(struct stat));
   stat(_file_name, &statbuf);

   size_t file_size = statbuf.st_size;
   size_t nread = 0;
   size_t prog_bar_size;
   size_t n_to_write;
   const char MESSAGE[] = "Loading Log File...";
   
   list< char * > rec;
   while(1) {
      memset(buffer, 0, BUF_SZ);
      if(fgets(buffer, BUF_SZ - 1, fin) ==  NULL)
        break;

      temp_sz = strlen(buffer) + 1;

      nread += (temp_sz - 1);

      clear();

      move((LINES >> 1) - 2, (COLS >> 1) - ((sizeof(MESSAGE) - 1) >> 1));
      attron(A_REVERSE);
      addstr(MESSAGE);
      attroff(A_REVERSE);

      prog_bar_size = COLS - round(COLS * ((1.0/((float)COLS)) * 10.0));
      prog_bar_size -= 2;
      n_to_write = round(((float)prog_bar_size) * ((1.0/((float)file_size)) * nread));

      move(LINES >> 1, (COLS >> 1) - (prog_bar_size >> 1) - 1);
      addstr("[");
      for(int i = 0; i < n_to_write; i++) {
         move(LINES >> 1, (COLS >> 1) - (prog_bar_size >> 1) + i);
         addstr("*");
      }
      move(LINES >> 1, (COLS >> 1) + (prog_bar_size >> 1) + 1);
      addstr("]");
      
      refresh();

      if(buffer[0] == '#')
         continue;

      temp = new char[temp_sz];
      memset(temp, 0, temp_sz);
      memcpy(temp, buffer, temp_sz - 1);
      rec.push_back(temp);

      if(buffer[strlen(buffer) - 1] == '\n') {
         size_t sz = 0;
         for(list< char * >::iterator p = rec.begin(); 
             p != rec.end(); p++) 
            sz += strlen(*p);
         sz++;
         char * total_rec = new char[sz];
         memset(total_rec, 0, sz);
         char * strptr = total_rec;
         int i = 0;
         for(list< char * >::iterator p = rec.begin(); 
             p != rec.end(); p++) {
            char * ptr = (*p);
            while((*ptr != '\0') && (i < (sz - 1))) {
               *strptr = *ptr;
               strptr++;
               ptr++;
               i++;
            }

            delete [] (*p);
         }

         rec = list< char * >();
         //rec.erase(rec.begin(), rec.end());

         Entry * p_entry = 0;
         p_entry = new Entry(total_rec, 
                             _date_format,  
                             _log_format);

         if(p_entry->is_worm(_worm_def_list)) {
            entry_list_type * p_entry_list = _entry_map[p_entry->ip()];

            if(p_entry_list)
               p_entry_list->push_back(p_entry);
            else {
               p_entry_list = new entry_list_type();
               p_entry_list->push_back(p_entry);
               _entry_map[p_entry->ip()] = p_entry_list;
            }
         }
      }
   }

   fclose(fin);
}

FILE * global_pipe;

void
die(int)
{
   if(global_pipe != NULL) {
      pclose(global_pipe);
      global_pipe = NULL;
      return;
   }
   signal(SIGINT, SIG_IGN);
   mvcur(0, COLS -1, LINES -1, 0);
   echo();
   curs_set(1);
   attroff(A_REVERSE);
   refresh();
   endwin();
   exit(0);
}

char * 
get_net_bios_name(const char * ip) 
{
   clear();
   move(0, (COLS >> 1) - 6);
   attron(A_REVERSE);
   addstr("nmblookup...");
   attroff(A_REVERSE);
   refresh();

   const size_t SZ = 1024;
   char buffer[SZ];

   memset(buffer, 0, SZ);
   snprintf(buffer, SZ - 1, "nmblookup -A %s", ip);
   
   FILE * p = popen(buffer, "r");
   if(p == NULL)
      return 0;

   int index = 0;
   memset(buffer, 0, SZ);
   while(fgets(buffer, SZ - 1, p) != NULL) {
      if(index == 1)
         break;
      index++;
      memset(buffer, 0, SZ);
   }

   pclose(p);

   size_t sz = strlen(buffer);
   if(sz >= 1) {
      if(buffer[sz - 1] == '\n' || buffer[sz - 1] == '\r')
         buffer[sz - 1] = '\0';
      if(sz >= 2) {
         if(buffer[sz - 2] == '\n' || buffer[sz - 2] == '\r')
            buffer[sz - 2] = '\0';
      }
   }

   char * ptr = buffer;
   while(*ptr != '\0' && *ptr == ' ')
      ptr++;
   char * name = ptr;
   while(*ptr != '\0' && *ptr != ' ')
      ptr++;
   *ptr = '\0';
   
   sz = strlen(name) + 1;
   char * temp = new char[sz];
   memset(temp, 0, sz);
   memcpy(temp, name, sz - 1);
   
   return temp;
}

bool
run_command(const char * cmd)
{
   list< char * > output;
   const size_t SZ = 1024;
   char buffer[SZ];
   clear();
   attron(A_REVERSE);
   const char * msg = "Running Command...";
   move(0, (COLS >> 1) - (strlen(msg) >> 1));
   addstr(msg);
   move(1,0);
   refresh();
   global_pipe = popen(cmd, "r");
   if(pipe == NULL)
      return false;
   memset(buffer, 0, SZ);
   attroff(A_REVERSE);
   int index = 0;
   while(1) {
      if(fgets(buffer, SZ-1, global_pipe) == NULL) {
         msg = "  Command Done...  ";
         break;
      }

      if(global_pipe == NULL)
         break;

      output.push_back(new_strdup(buffer));

      move(index + 1, 0);
      addstr(buffer);
      refresh();
      memset(buffer, 0, SZ);
      index++;

      attron(A_REVERSE);
      move(0, (COLS >> 1) - (strlen(msg) >> 1));
      addstr(msg);
      refresh();
      attroff(A_REVERSE);

      if((index + 1) == LINES) {
         index = 0;
         clear();
      }
   }

   if(global_pipe != NULL) {
      pclose(global_pipe);
      global_pipe = NULL;
   }

   int offset = 0;
   int line = 1;
   index = 0;
   int ch;
   size_t len;
   int i;
   while(output.size() > 0) {
      clear();
      line = 2;
      i = 0;
      attron(A_REVERSE);
      move(0, (COLS >> 1) - (strlen(msg) >> 1));
      addstr(msg);
      attroff(A_REVERSE);
      for(list< char * >::iterator p = output.begin();
          p != output.end(); p++) {
         if(i < offset) {
            i++;
            continue;
         }

         if(i == index)
            attron(A_REVERSE);
         else
            attroff(A_REVERSE);
         move(line, 0);
         len = strlen(*p);
         if(len > (COLS - 1))
            (*p)[COLS - 1] = '\0';
         addstr(*p);

         line++;
         i++;

         if(i >= ((offset + LINES) - 2))
            break;
      }
      refresh();

      ch = getch();
      if(ch == '\n' || ch == 'q' || ch == ERR)
         break;

      switch(ch) {
         case 'k':
         case KEY_UP:
            if(index > 0)
               index--;
            if(offset > index)
               offset--;
         break;
 
         case 'j':
         case KEY_DOWN:
            if(index < (output.size() - 1))
               index++;
            if((LINES - (index - offset)) == 2 
               && index <= (output.size() - 1))
               offset++;
         break;
      }
   }

   for(list< char * >::iterator p = output.begin();
       p != output.end(); p++)
      delete [] (*p);

   return true;
} 

class entry_pair_gt {
   int _order_by;
public:
   entry_pair_gt(int order_by = 1) 
   :_order_by(order_by) 
   { }

   bool operator()(const pair< const char *, LogFile::entry_list_type * > & left,
                   const pair< const char *, LogFile::entry_list_type * > & right) const;
};

bool
ip_gt(const char * l, const char * r)
{
   const size_t IP_SZ = (4 * 3) + 3 + 1;
   char b0[IP_SZ], b1[IP_SZ];
   int lv;
   int rv;
   char * s0, * s1;
   char * p0, * p1;

   memset(b0, 0, IP_SZ);
   memset(b1, 0, IP_SZ);

   size_t lsz = strlen(l);
   size_t rsz = strlen(r);

   if((lsz > (IP_SZ - 1))  || (rsz > (IP_SZ - 1)))
      return false;

   memcpy(b0, l, lsz);
   memcpy(b1, r, rsz); 

   p0 = b0;
   p1 = b1;

   for(int i = 0; i < 4; i++) {
      s0 = p0;
      while(*p0 != '.' && *p0 != '\0')
         p0++;
      if(*p0 != '\0') {
         *p0 = '\0';
         p0++;
      }

      lv = atoi(s0);

      s1 = p1;
      while(*p1 != '.' && *p1 != '\0')
         p1++;
      if(*p1 != '\0') {
         *p1 = '\0';
         p1++;
      }

      rv = atoi(s1);

      if(lv > rv)
         return true;
      if(lv < rv)
         return false;
   }

   return false;
}

bool 
entry_pair_gt::operator()(const pair< const char *, LogFile::entry_list_type * > & left,
                          const pair< const char *, LogFile::entry_list_type * > & right) const 
{
   switch(_order_by) {
      case 1:
         if(left.second->back()->when() == right.second->back()->when())
            return (left.second->size() > right.second->size());
         return (left.second->back()->when() > right.second->back()->when());

      case 2:
         if(left.second->size() == right.second->size())
            return (left.second->back()->when() > right.second->back()->when());
         return (left.second->size() > right.second->size());

      case 3:
         return ip_gt(left.first, right.first);

      default:
         return false;
   }
}


void
show_host_names(const char * ip) 
{
   clear();
   move(0, (COLS >> 1) - 8);
   attron(A_REVERSE);
   addstr("gethostbyaddr...");
   attroff(A_REVERSE);
   refresh();

   hostent * p_host;
   sockaddr_in addr;
 
   memset(&addr, 0, sizeof(addr));
   if(!inet_aton(ip, &addr.sin_addr))
      return;
   
   p_host = gethostbyaddr((const char *)(&addr.sin_addr), 
                          sizeof(addr.sin_addr), AF_INET);

   if(p_host != NULL) {
      int alias_count = 0;
      for(int i = 0; p_host->h_aliases[i] != NULL; i++)
         alias_count++;

      if((alias_count + 1) > LINES)
         alias_count = LINES - 1;

      clear();

      const size_t SZ = 1024;
      char buffer[SZ];

      memset(buffer, 0, SZ);
      snprintf(buffer, SZ - 1, "Worm Host: %s", p_host->h_name);
      move((LINES >> 1) - ((1 + alias_count) >> 1), 
           (COLS >> 1) - (strlen(buffer) >> 1));
      addstr(buffer);

      for(int i = 1; i <= alias_count; i++) {
         memset(buffer, 0, SZ);
         snprintf(buffer, SZ - 1, "Alias: %s", p_host->h_aliases[i - 1]);
         move(((LINES >> 1) - ((1 + alias_count) >> 1)) + 1, 
              (COLS >> 1) - (strlen(buffer) >> 1));
         addstr(buffer);
      }

      move(LINES - 1, (COLS >> 1) - 8);
      addstr("[enter to return]");
      refresh();

      while(getch() != '\n')
         ;

      refresh();
   }
}

void
show_worm_request_list(LogFile::entry_list_type * p_entry_list) 
{
   size_t sz;
   int ch;
   int index = 0;
   int offset = 0;
   int line;
   LogFile::Entry * p_entry;
   const size_t SZ = 1024;
   char buffer[SZ];
   
   while(1) {
      line = 0;
      clear();
      for(LogFile::entry_list_type::iterator p = p_entry_list->begin();
          p != p_entry_list->end(); p++) {
         if(line < offset) {
            line++;
            continue; 
         }

         if(line == index)
            attron(A_REVERSE);
         else
            attroff(A_REVERSE);

         p_entry = (*p);

         memset(buffer, 0, SZ);
         snprintf(buffer, SZ - 1, "%d - %d %s", 
                  line+1,
                  p_entry->response_code(), 
                  p_entry->request());
         sz = strlen(buffer);
         if(sz > (COLS-2) && COLS < SZ) {
            buffer[COLS - 3] = '\0';
            sz = COLS - 2;
         }
         move(line - offset, (COLS >> 1) - (sz >> 1));
         addstr(buffer);

         line++;

         if(line >= (offset + LINES)) 
            break;
      } 
      refresh();

      ch = getch();
      if(ch == '\n' || ch == 'q' || ch == ERR)
         break;

      switch(ch) {
         case 'k':
         case KEY_UP:
            if(index > 0)
               index--;
            if(index < offset)
               offset--;
         break;
      
         case 'j':
         case KEY_DOWN:
            if(index < (p_entry_list->size() - 1))
               index++;
            if((LINES - (index - offset)) == 1 
               && index < (p_entry_list->size() - 1))
               offset++;
         break;
      }
   }
}

bool
get_log_properties(properties_file & props, 
                   const char * & log_date_format,
                   const char * & log_format)
{
   const char * pcount = props["log.file.format.count"];
   if(pcount == 0)
      return false;
   int n = atoi(pcount);
   int start = (LINES >> 1) - (n >> 1);
   const size_t MAX_BUFFER = 1024;
   char buffer[MAX_BUFFER];
   int index;
   size_t length;
   const char * format_name;
   int current_index = 0;
   int offset_index = 0;

   if(start < 1)
      start = 1;

   int ch;
   const char MESSAGE[] = "Please Select The Format For The Log File";
   bool stop = false;

   while(!stop) {
      clear();
      attron(A_REVERSE);
      move(0, (COLS >> 1) - (strlen(MESSAGE) >> 1));
      addstr(MESSAGE);
      attroff(A_REVERSE);

      index = 0;
      for(int i = start; ((index < n) && (i < LINES)); i++) {
         if(index < offset_index) {
            index++;
            continue;
         }
         memset(buffer, 0, MAX_BUFFER);
         snprintf(buffer, MAX_BUFFER-1, "log.file.format.%d.name", (index+1));
         format_name = props[buffer];
         if(!format_name) 
            return false;
         length = strlen(format_name);
         move(i, (COLS >> 1) - (length >> 1));
         if(current_index == index) 
            attron(A_REVERSE);
         else
            attroff(A_REVERSE);
         addstr(format_name);
         index++;

         if(index >= n)
            break;
      }
      refresh();

      ch = getch();
      if(ch == 'q' || ch == ERR)
         break;

      switch(ch) {
         case 'k':
         case KEY_UP:
            if(current_index > 0)
               current_index--;
            if(offset_index < current_index)
               offset_index--;
         break;

         case 'j':
         case KEY_DOWN:
            if(current_index < (n-1))
               current_index++;
            if((current_index - offset_index) >= (LINES - 2))
               offset_index++;
         break;
 
         case '\n':
            stop = true;
         break;
      }
   }

   memset(buffer, 0, MAX_BUFFER);
   snprintf(buffer, MAX_BUFFER-1, "log.file.format.%d.timeformat", (current_index+1));
   log_date_format = props[buffer];

   memset(buffer, 0, MAX_BUFFER);
   snprintf(buffer, MAX_BUFFER-1, "log.file.format.%d.regexpr", (current_index+1));
   log_format = props[buffer];

   if(!log_date_format || !log_format)
      return false;

   return true;
}

void 
main_app(int argc, char ** argv)
{
   int ch = -1;
   int curr = 0;
   LogFile * p_log_file = 0;
   const size_t MENU_SZ = 4;
   const char * MENU_ARRAY[MENU_SZ] = { 
      "Load Log File And Find Worm Hosts",
      "Show Worm Hosts",
      "About Worm Hunter",
      "Quit Worm Hunter"
   };
   const size_t MAX_FILE_NAME_SZ = 1024;
   char buffer[MAX_FILE_NAME_SZ];
   const size_t SZ = 1024;
   char worm_host[SZ];
   LogFile::entry_list_type * p_entry_list;
   int line;
   int offset;
   int index;
   int i;
   LogFile::Entry * p_entry;
   const size_t WORM_HOST_MENU_SZ = 10;
   const char * WORM_HOST_MENU_ARRAY[WORM_HOST_MENU_SZ] = {
      "Send An Email To This Host",
      "Show Worm Request List",
      "Get Host By Address",
      "Send SMB Message",
      "SMB Guest Login",
      "Port Scann Host",
      "Who Is Search",
      "Trace Route",
      "Ping",
      "Back To Worm Host Menu"
   };
   const size_t ABOUT_MSG_SZ = 4;
   const char * ABOUT_MSG[ABOUT_MSG_SZ] = {
      "Worm Hunter (V1.0.0)",
      "Worm Hunter/Killer",
      "Author: Matthew W. Coan",
      "matthewcoan@hotmail.com"
   };
   typedef list< pair< const char *, LogFile::entry_list_type * > > entry_map_list_type;
   entry_map_list_type entry_pair_list;
   char * net_bios_name;
   properties_file * p_props = 0;
   const char * log_date_format;
   const char * log_format;

   try {
      p_props = new properties_file(argv[1]);
   }
   catch(properties_file::file_access_exception) {
      fprintf(stderr, "Unable to load properties file: %s\n", argv[1]);
      die(0);
   }

   if(argc == 3) {
      try {
         if(get_log_properties(*p_props, log_date_format, log_format)) {
            p_log_file = new LogFile(argv[2], log_date_format, log_format, *p_props);
            for(LogFile::entry_map_type::iterator p = p_log_file->begin();
                p != p_log_file->end(); p++)
               entry_pair_list.push_back(*p);
            entry_pair_list.sort(entry_pair_gt());
         }
      }
      catch(LogFile::properties_format_exception) {
         fprintf(stderr, "[Bad Worm Definition List]\n");
         delete p_props;
         die(0);
      }
      catch(LogFile::log_file_access_exception) {
         fprintf(stderr, "Unable to load log file: %s\n", argv[2]);
         delete p_props;
         die(0);
      }
      catch(LogFile::log_format_exception) {
         fprintf(stderr, "[Bad Format] Unable to load log file: %s\n", argv[2]);
         delete p_props;
         die(0);       
      }
   }

   while(1) {
      switch(ch) {
         case 'k':
         case KEY_UP:
            if(curr > 0)
               curr--;
         break;
        
         case 'j':
         case KEY_DOWN:
            if(curr < (MENU_SZ - 1))
               curr++;
         break;

         case '\n': 
            switch(curr) {
               case 0: // Load log file
                  if(p_log_file) {
                     delete p_log_file;
                     p_log_file = 0;
                  }
                  echo();
                  curs_set(1);
                  try {
                     move(LINES - 1, 0);
                     addstr("Log file path: ");
                     memset(buffer, 0, MAX_FILE_NAME_SZ);
                     mvscanw(LINES-1,15,"%s", buffer);
                     noecho();
                     curs_set(0);
                     if(get_log_properties(*p_props, log_date_format, log_format)) {
                        p_log_file = new LogFile(buffer, log_date_format, log_format, *p_props);
                        entry_pair_list.erase(entry_pair_list.begin(), entry_pair_list.end());
                        for(LogFile::entry_map_type::iterator p = p_log_file->begin();
                            p != p_log_file->end(); p++)
                           entry_pair_list.push_back(*p);
                        entry_pair_list.sort(entry_pair_gt());
                     }
                  }
                  catch(LogFile::properties_format_exception) {
                     noecho();
                     curs_set(0);
                     move(LINES - 1, 0);
                     addstr("Your worm definition list has a format error! "
                            "[please press enter to continue]");
                     while(getch() != '\n')
                        ;
                  }
                  catch(LogFile::log_file_access_exception) {
                     noecho();
                     curs_set(0);
                     move(LINES - 1, 0);
                     addstr("Unable to load log file! "
                            "[please press enter to continue]");
                     while(getch() != '\n')
                        ;
                  }
                  catch(LogFile::log_format_exception) {
                     noecho();
                     curs_set(0);
                     move(LINES - 1, 0);
                     addstr("Unable to load log file! Format exception!"
                            "[please press enter to continue]");
                     while(getch() != '\n')
                        ;
                  }
                  noecho();
                  curs_set(0);
               break;

               case 1: // Display worm hosts
                  if(!p_log_file)
                     break;

                  if(p_log_file->size() == 0)
                     break;

                  offset = 0;
                  index = 0;

                  while(1) {
                     clear();

                     attron(A_REVERSE);
                     memset(worm_host, 0, SZ);
                     snprintf(worm_host, SZ-1, "     #       TYPE              IP"
                              "  # WORM REQ    LAST WORM REQUEST DATE",
                              p_log_file->size());
                     mvaddstr(0,(COLS >> 1) - (strlen(worm_host) >> 1), worm_host);
                     attroff(A_REVERSE);

                     line = 2;
                     i = 0;
                     for(entry_map_list_type::iterator p = entry_pair_list.begin();
                         p != entry_pair_list.end(); p++) {
                        if(line < (offset+2)) {
                           line++;
                           i++;
                           continue;
                        }
                        p_entry_list = p->second;
                        memset(worm_host, 0, SZ);

                        if(index == (line - 2))
                           attron(A_REVERSE);
                        else
                           attroff(A_REVERSE);

                        snprintf(worm_host, SZ-1, "%6d %10s %15s %11d %25s", 
                                 i+1, p_entry_list->back()->worm_name(), 
                                 p->first, p_entry_list->size(), 
                                 p_entry_list->back()->when_string());
                        mvaddstr(line - offset, 
                                 (COLS >> 1) - (strlen(worm_host) >> 1), 
                                 worm_host);

                        line++;
                        i++;

                        if((line - 2) > (offset + LINES))
                           break;
                     }

                     refresh();

                     ch = getch();
                     if(ch == 'q' || ch == ERR)
                        break;

                     switch(ch) {
                        case 'k':
                        case KEY_UP:
                           if(index > 0)
                              index--;
                           if(index < offset)
                              offset--;
                        break;

                        case 'j':
                        case KEY_DOWN:
                           if(index < (p_log_file->size() - 1))
                              index++;
                           if((LINES - (index - offset)) == 2 
                              && index < p_log_file->size())
                              offset++;
                        break;

                        case '#':
                           entry_pair_list.sort(entry_pair_gt(2));
                        break;

                        case 'd':
                           entry_pair_list.sort(entry_pair_gt());
                        break;

                        case 'i':
                           entry_pair_list.sort(entry_pair_gt(3));
                        break;

                        case '\n':
                           i = 0;
                           for(entry_map_list_type::iterator p = entry_pair_list.begin();
                               p != entry_pair_list.end(); p++) {
                              p_entry_list = p->second;
                              if(i == index)
                                 break;
                              i++;
                           }
 
                           p_entry = p_entry_list->front();

                           bool running = true;
                           int my_index = 0;

                           while(running) {
                              clear();

                              attroff(A_REVERSE);
                              memset(worm_host, 0, SZ);
                              snprintf(worm_host, SZ-1, "Worm Host: %s", p_entry->ip());
                              move(0, (COLS >> 1) - (strlen(worm_host) >> 1));
                              addstr(worm_host);

                              memset(worm_host, 0, SZ);
                              snprintf(worm_host, SZ-1, "Worm Request Count: %d", 
                                       p_entry_list->size());
                              move(1, (COLS >> 1) - (strlen(worm_host) >> 1));
                              addstr(worm_host);

                              for(i = 0; i < WORM_HOST_MENU_SZ; i++) {
                                 if(my_index == i)
                                   attron(A_REVERSE);
                                 else
                                   attroff(A_REVERSE); 

                                 move((LINES >> 1) - (WORM_HOST_MENU_SZ >> 1) + i, 
                                      (COLS >> 1) - (strlen(WORM_HOST_MENU_ARRAY[i]) >> 1));
                                 addstr(WORM_HOST_MENU_ARRAY[i]);
                              }
 
                              refresh();

                              ch = getch();
 
                              if(ch == 'q' || ch == ERR)
                                 break;

                              switch(ch) {
                                 case 'k':
                                 case KEY_UP:
                                    if(my_index > 0)
                                       my_index--;
                                 break;

                                 case 'j':
                                 case KEY_DOWN:
                                    if(my_index < (WORM_HOST_MENU_SZ - 1))
                                       my_index++;
                                 break;

                                 case '\n':
                                    switch(my_index) {
                                       case 0: // Send Email
                                          
                                       break;

                                       case 1:
                                          show_worm_request_list(p_entry_list);
                                       break;
 
                                       case 2: // Get Host By Address
                                          show_host_names(p_entry->ip());
                                       break;

                                       case 3: // Send SMB Message
                                          net_bios_name = get_net_bios_name(p_entry->ip());
                                          if(net_bios_name == NULL)
                                             break;
                                          if(strlen(net_bios_name) == 0)
                                             break;
                                          memset(worm_host, 0, SZ);
                                          snprintf(worm_host, SZ - 1, 
                                                   "printf \"Your computer has a worm...\\r\\n\\r\\n"
                                                   "Your computer has sent %d worm requests "
                                                   "to my computer...\\r\\n\\r\\n"
                                                   "The last attack occured on: %s\\r\\n\\r\\n"
                                                   "The worm could possibly be %s...\\r\\n\\r\\n"
                                                   "Please fix this problem...\\r\\n\" | "
                                                   "smbclient -M %s -I %s -U \"Epoch\" -N",
                                                   p_entry_list->size(),
                                                   p_entry_list->back()->when_string(),
                                                   p_entry_list->back()->worm_name(),
                                                   net_bios_name,
                                                   p_entry->ip());
                                          delete [] net_bios_name;
                                          run_command(worm_host);
                                       break;

                                       case 4: // SMB Guest Login
                                          memset(worm_host, 0, SZ);
                                          snprintf(worm_host, SZ - 1, 
                                                   "smbclient -L %s -U Guest -N",
                                                   p_entry->ip());
                                          run_command(worm_host);
                                       break;

                                       case 5: // Port Scan Host
                                          memset(worm_host, 0, SZ);
                                          snprintf(worm_host, SZ - 1, 
                                                   "nmap -sS %s", p_entry->ip());
                                          run_command(worm_host);
                                       break;

                                       case 6: // Who Is Search
                                          memset(worm_host, 0, SZ);
                                          snprintf(worm_host, SZ - 1, 
                                                   "whois %s", p_entry->ip());
                                          run_command(worm_host);
                                       break;

                                       case 7: // Trace Route
                                          memset(worm_host, 0, SZ);
                                          snprintf(worm_host, SZ - 1, 
                                                   "traceroute %s", p_entry->ip());
                                          run_command(worm_host);
                                       break;

                                       case 8: // Ping
                                          memset(worm_host, 0, SZ);
                                          snprintf(worm_host, SZ - 1, 
                                                   "ping %s", p_entry->ip());
                                          run_command(worm_host);
                                       break;

                                       case 9: // Worm Host Menu
                                          running = false;
                                       break;
                                    }
                                 break;
                              }
                           }
                        break;
                     }
                  }
               break;

               case 2:
                  clear();
                  for(i = 0; i < ABOUT_MSG_SZ; i++) {
                     move(((LINES >> 1) - (ABOUT_MSG_SZ >> 1)) + i, 
                          (COLS >> 1) - (strlen(ABOUT_MSG[i]) >> 1));
                     addstr(ABOUT_MSG[i]);
                  }

                  move(LINES - 1, (COLS >> 1) - 10);
                  addstr("[enter for main menu]");

                  refresh();

                  while(getch() != '\n')
                     ;
               break;

               case 3: // Quit
                  if(p_log_file) 
                     delete p_log_file;
                  if(p_props) 
                     delete p_props;
                  return;
            }
         break;
      }

      clear();

      for(int i = 0; i < MENU_SZ; i++) {
         if(curr == i)
            attron(A_REVERSE);
         else
            attroff(A_REVERSE);
         move((LINES >> 1) - (MENU_SZ >> 1) + i, 
              (COLS >> 1) - (strlen(MENU_ARRAY[i]) >> 1));
         addstr(MENU_ARRAY[i]);
      }

      refresh();

      ch = getch();

      if(ch == 'q' || ch == ERR)
         break;
   }

   if(p_props)
      delete p_props;

   if(p_log_file)
      delete p_log_file;
}

void
empty_signal_handler(int)
{
}

int
main(int argc, char ** argv)
{
   if(argc != 2 && argc != 3) {
      fprintf(stderr, "Usage: %s <properties file> [<log file>]\n", argv[0]);
      return -1;
   }
   initscr();
   noecho();
   crmode();
   curs_set(0);
   keypad(stdscr, TRUE);
   signal(SIGINT, empty_signal_handler);
   try {
      main_app(argc, argv);
   }
   catch(...) {
      fprintf(stderr, "[Unhandled Exception!]\n");
      die(0);
   }
   clear();
   refresh();
   die(0);
   return 0;
}
